﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
  AnimmSoundFile のデータを書き換えて Modifier のイベント更新を実行しようとする場合、
  イベントテーブルの増減でメモリ破壊が発生してしまうため、
  独自に確保したメモリ上に構築して、そちら側を使用するようにする。
 */

#pragma once

#include <nw/snd.h>
#include "Constants.h"

namespace nw {
namespace snd {

class AnimSoundFileBuffer
{
public:
    AnimSoundFileBuffer();
    ~AnimSoundFileBuffer();

    bool Initialize(ut::IAllocator* allocator);
    void Finalize();

    void Setup(void* bfasd);

    bool IsAvailable() const { return m_Buffer != NULL; }
    bool IsSetuped() const { return m_IsSetup; }
    u32 GetEventCount() const { return m_Buffer->animEventTableImpl.count; }

    // イベント個数を修正する
    bool AddEvent(int id);
    void RemoveEvent(int id);
    s32 FindIndex(int id);
    void UpdateAllEventID(const u16 idList[], u32 listSize);

    // 直接データを書き換えるため、データテーブルへのアクセス
    internal::AnimSoundFile::AnimEvent& GetEvent(int id);
    internal::AnimSoundFile::EventInfo& GetEventInfo(int id);
    char* GetEventSoundLabel(int id);

    // AnimSoundに渡すためのポインタアクセサ
    void* ptr() { return (void*)m_Buffer; }

public:
    static const s32 INVALID_INDEX = -1;
private:
    static const u16 INVALID_ID = 0xffff;

private:
    struct Reference
    {
        u16 id;
        internal::AnimSoundFile::AnimEvent* animEvent;
    };

    // AnimSoundのバイナリ形式を再現する
    struct Impl
    {
        // SoundFileHeader
        ut::BinaryFileHeader header;
        // このテーブルには1つしかデータはないはず
        internal::Util::BlockReferenceTable blockReferenceTable;

        // AnimSoundFile::DataBlock
        // blockReferenceTable[0] はこのDataBlockを参照している
        internal::AnimSoundFile::DataBlock data;

        // AnimSoundFile::AnimEventTableの構造
        // data.DataBlockBody.GetAnimEventTableが参照している先。
        // Util::Tableは定義としてはサイズ1の配列を持っているだけなので、
        // unionで必要なだけ配列を確保した構造を定義する
        //internal::AnimSoundFile::AnimEventTable animEventTable; // この定義の代わりに以下の構造体を使用
        struct {
            ut::ResU32 count;
            internal::AnimSoundFile::AnimEvent item[Constants::ANIMEVENT_MAX];
        } animEventTableImpl;

        // AnimEvent->GetEventInfoの参照先。
        internal::AnimSoundFile::EventInfo eventInfo[Constants::ANIMEVENT_MAX];

        // EventInfo->GetSoundLabelで参照している先。
        char soundLabel[Constants::ANIMEVENT_MAX][Constants::SOUND_LABEL_MAX_LENGTH];

        //----------------------------------------------------------
        void Resolve(Reference reference[]);
        void EnableEvent(u32 index);
        void DisableEvent(u32 index);
    };

private:
    ut::IAllocator* m_Allocator;
    Impl* m_Buffer;
    bool m_IsSetup;
    Reference m_Reference[Constants::ANIMEVENT_MAX];
};

} // namespace snd
} // namespace nw

//eof
