﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// WaveCodec.cpp : DLL アプリケーション用にエクスポートされる関数を定義します。
//

#include "stdafx.h"
#include <cstdio>   // printf
#include <cstring>  // memcpy
#include "DspAdpcmLib.h"
#include "types.h"

#define _USE_MATH_DEFINES
#include <cmath>

namespace
{

/*!--------------------------------------------------------------------------*
  @brief        エンコード方式を定義する列挙体です。
                C# 側で同じものを定義する必要があります。
 *---------------------------------------------------------------------------*/
enum EncodeMethod
{
    eEncodeMethod_PCM8 = 0,
    eEncodeMethod_PCM16_BE,
    eEncodeMethod_PCM16_LE,
    eEncodeMethod_DSP_ADPCM,
};

/*!--------------------------------------------------------------------------*
  @brief        Normalize 後の 1 サンプルあたりのバイト数です。
 *---------------------------------------------------------------------------*/
const int NORMALIZED_BYTE_PER_SAPMLE = 2;

/*!--------------------------------------------------------------------------*
  @brief        ループ先頭アドレスの境界は下記の倍数バイトである必要があります。
 *---------------------------------------------------------------------------*/
const int ALIGN_MIN_BYTE_CTR = 4;
const int ALIGN_MIN_BYTE_DSPADPCM = 8;

/*!--------------------------------------------------------------------------*
  Name:         GetAlignSamplesForDspAdpcm

  @brief        DSP ADPCM のエンコード時の補正サンプルサイズを返します。

  @param        alignByte   バイト換算のアライメントサイズです。

  @return       バイト換算のアライメントサイズを、サンプル数に換算した値を
                返します。
 *---------------------------------------------------------------------------*/
int GetAlignSamplesForDspAdpcm( int alignByte )
{
    if ( alignByte % ALIGN_MIN_BYTE_DSPADPCM != 0 )
    {
        return -1;
    }

    // DSP ADPCM フレーム (8バイト/フレーム) 換算で何フレームか？
    // (1 フレームあたり、14 サンプル)
    int result = alignByte * 14 / ALIGN_MIN_BYTE_DSPADPCM;

    return result;
}

/*!--------------------------------------------------------------------------*
  Name:         GetAlignSamples

  @brief        バイト換算のアライメントサイズを、エンコード方式に応じた
                サンプル数に換算して返します。

  @param        alignByte   バイト換算のアライメントサイズです。
  @param        method      エンコード方式です。

  @return       バイト換算のアライメントサイズを、エンコード方式に応じた
                サンプル数に換算して返します。
 *---------------------------------------------------------------------------*/
int GetAlignSamples( int alignByte, EncodeMethod method )
{
    int result;
    switch ( method )
    {
        case eEncodeMethod_PCM8:
            if ( alignByte % ALIGN_MIN_BYTE_CTR != 0 ) { result = -1; }
            else { result = alignByte; }
            break;
        case eEncodeMethod_PCM16_BE:
        case eEncodeMethod_PCM16_LE:
            if ( alignByte % ALIGN_MIN_BYTE_CTR != 0 ) { result = -1; }
            else { result = alignByte / 2; }
            break;
        case eEncodeMethod_DSP_ADPCM:
            if ( alignByte % ALIGN_MIN_BYTE_DSPADPCM != 0 ) { result = -1; }
            else { result = GetAlignSamplesForDspAdpcm( alignByte ); }
            break;
        default:
            return -1;
    }
    return result;
}

/*!--------------------------------------------------------------------------*
  Name:         AddSamplesToLoopEnd

  @brief        指定開始位置から指定されたサンプル数だけ、ループエンド位置にコピーします。

  @param[in,out]    samples     入力サンプルです。loopEndSample + addSamples
                                サンプルの長さがある必要があります。
  @param        srcIndex        コピー元の開始インデックスです。
  @param        loopEndSample   本来のループ終了位置です。
  @param        addSamples      コピーするサンプル数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void AddSamplesToLoopEnd(
        s16* samples,
        int srcIndex,
        int loopEndSample,
        int addSamples )
{
    for ( int i = 0; i < addSamples; i++ )
    {
        int srcPos = srcIndex + i;
        int dstPos = loopEndSample + i;
        samples[ dstPos ] = samples[ srcPos ];
    }
}


/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromS8

  @brief        1 サンプルを s8 で表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromS8( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=1, i++ )
    {
        output[ i ] = static_cast<s16>( static_cast<s8>( input[pos] ) << 8 );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromU8

  @brief        1 サンプルを u8 で表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromU8( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=1, i++ )
    {
        output[ i ] = static_cast<s16>(
                ( static_cast<s16>( input[pos] ) - 0x80 ) << 8 );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromS16BE

  @brief        1 サンプルを s16 ビッグエンディアンで表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromS16BE( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=2, i++ )
    {
        output[ i ] = static_cast<s16>(
                ( static_cast<s8>( input[pos] ) << 8 ) | input[pos+1] );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromS16LE

  @brief        1 サンプルを s16 リトルエンディアンで表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromS16LE( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=2, i++ )
    {
        output[ i ] = static_cast<s16>(
                ( static_cast<s8>( input[pos+1] ) << 8 ) | input[pos] );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromS24BE

  @brief        1 サンプルを s24 ビッグエンディアンで表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromS24BE( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=3, i++ )
    {
        output[ i ] = static_cast<s16>(
                ( static_cast<s8>( input[pos] ) << 8 ) | input[pos+1] );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromS24LE

  @brief        1 サンプルを s24 リトルエンディアンで表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromS24LE( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=3, i++ )
    {
        output[ i ] = static_cast<s16>(
                ( static_cast<s8>( input[pos+2] ) << 8 ) | input[pos+1] );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromS32BE

  @brief        1 サンプルを s32 ビッグエンディアンで表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromS32BE( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=4, i++ )
    {
        output[ i ] = static_cast<s16>(
                ( static_cast<s8>( input[pos] ) << 8 ) | input[pos+1] );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         NormalizeFromS32LE

  @brief        1 サンプルを s32 リトルエンディアンで表現しているバイト列を
                s16 (リトルエンディアン) 配列に変換します。

  @param        output      出力である s16 配列を格納します。
  @param        input       入力バイト列です。
  @param        inputByte   入力バイト数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void NormalizeFromS32LE( s16* output, const u8* input, int inputByte )
{
    for ( int pos = 0, i = 0; pos < inputByte; pos+=4, i++ )
    {
        output[ i ] = static_cast<s16>(
                ( static_cast<s8>( input[pos+3] ) << 8 ) | input[pos+2] );
    }
}

const int RESAMPLE_WIDTH = 200;

template< class T >
T abs( T s ) { return ( s >= 0 ) ? s : -s ; }

} // anonymous namespace


/*
 *
 *      公 開 さ れ る 関 数
 *
 */

/*!--------------------------------------------------------------------------*
  Name:         GetBufferSizeForNormalize

  @brief        Normalize の出力を格納するのに必要なバッファサイズを取得します。

  @param        inputByte   入力サンプルバイト数です。
  @param        bytePerSample   入力サンプルの、 1 サンプルあたりのバイト数です。

  @return       Normalize の出力を格納するのに必要なバッファサイズを返します。
 *---------------------------------------------------------------------------*/
int GetBufferSizeForNormalize( int inputByte, int bytePerSample )
{
    int sampleCount = inputByte / bytePerSample;
    int ret = sampleCount * NORMALIZED_BYTE_PER_SAPMLE;
    return ret;
}

/*!--------------------------------------------------------------------------*
  Name:         Normalize

  @brief        入力サンプルを、16bit リトルエンディアンのサンプル列に変換します。

  @param[out]   output  変換後のサンプル列を格納します。
                        あらかじめ、GetNormalizedSampleSize バイト分のメモリを
                        確保しておく必要があります。
  @param        input   入力サンプルを示すバイト列です。
  @param        inputByte   入力サンプルバイト数です。
  @param        bytePerSample   入力サンプルの、 1 サンプルあたりのバイト数です。
  @param        isLittleEndian  入力サンプルがリトルエンディアンなら true です。
                                bytePerSample が 1 のときは無視されます。
  @param        isSigned    入力サンプルが signed なら true です。

  @return       成功したら true, 失敗したら false を返します。
 *---------------------------------------------------------------------------*/
bool Normalize(
        s16*        output,
        const u8*   input,
        int         inputByte,
        int         bytePerSample,
        bool        isLittleEndian,
        bool        isSigned )
{
    switch ( bytePerSample )
    {
        case 1:
            if ( isSigned ) { NormalizeFromS8( output, input, inputByte ); }
            else { NormalizeFromU8( output, input, inputByte ); }
            break;
        case 2:
            if ( isSigned )
            {
                if ( isLittleEndian ) { NormalizeFromS16LE( output, input, inputByte ); }
                else { NormalizeFromS16BE( output, input, inputByte ); }
            }
            else { return false; }
            break;
        case 3:
            if ( isSigned )
            {
                if ( isLittleEndian ) { NormalizeFromS24LE( output, input, inputByte ); }
                else { NormalizeFromS24BE( output, input, inputByte ); }
            }
            else { return false; }
            break;
        case 4:
            if ( isSigned )
            {
                if ( isLittleEndian ) { NormalizeFromS32LE( output, input, inputByte ); }
                else { NormalizeFromS32BE( output, input, inputByte ); }
            }
            else { return false; }
            break;
        default:
            return false;
    }
    return true;
}

/*!--------------------------------------------------------------------------*
  Name:         GetBufferSizeForDevideChannel

  @brief        DevideChannel の出力を格納するのに必要なバッファサイズを取得します。

  @param        sampleCount 入力サンプル数 (チャンネル分割前) です。
  @param        inputByte   入力バイト数です。

  @return       DevideChannel の出力を格納するのに必要なバッファサイズを返します。
 *---------------------------------------------------------------------------*/
int GetBufferSizeForDevideChannel( int sampleCount, int channelMax )
{
    int ret = ( sampleCount / channelMax ) * sizeof( s16 );
    return ret;
}

/*!--------------------------------------------------------------------------*
  Name:         DevideChannel

  @brief        インターリーブされたサンプルを、チャンネル別に分割します。

  @param[out]   output      分割したサンプルの格納先です。
  @param        input       入力されるサンプルで、インターリーブされています。
  @param        inputSampleCount    入力サンプルのサンプル数です。
  @param        channelIndex    何チャンネル目のサンプルを取得するかを指定します。
                                0 から指定します。
  @param        cahnnelCount    入力サンプルのチャンネル数です。

  @return       成功したら true, 失敗したら false を返します。
 *---------------------------------------------------------------------------*/
bool DevideChannel(
        s16*        output, /* GetByteOfDevidedChannel バイトのメモリを事前に確保 */
        const s16*  input,
        int         inputSampleCount,
        int         channelIndex,
        int         channelCount )
{
    if ( channelIndex >= channelCount )
    {
        return false;
    }

    int frameCount = inputSampleCount / channelCount;

    for ( int i = 0; i < frameCount; i++ )
    {
        output[ i ] = input[ i * channelCount + channelIndex ];
    }
    return true;
}

/*!--------------------------------------------------------------------------*
  Name:         GetBufferSizeForAlignLoop

  @brief        AlignLoop の出力を格納するのに必要なバッファサイズを取得します。

  @param        alignByte   何バイト境界でアライメントするかを設定します。
                            CTR の場合、PCM8 / PCM16 は 4 バイトの倍数、
                            DSP ADPCM の場合は 8 バイトの倍数である必要があります。
  @param        loopStartSample ループ開始のサンプルインデックスです。
  @param        loopEndSample   ループ終了のサンプルインデックスです。
  @param        isExpandLoop    ストリームファイルのエンコード時に true にします。
                                ストリームファイルは短いループの場合に
                                うまく再生できないので、true にしておく必要があります。
  @param        method      エンコード方式です。

  @return       AlignLoop の出力を格納するのに必要なバッファサイズを返します。
 *---------------------------------------------------------------------------*/
int GetBufferSizeForAlignLoop(
        int             alignByte,      // バイト単位
        int             loopStartSample,
        int             loopEndSample,
        bool            isExpandLoop,
        EncodeMethod    method )
{
    // アライメント用サンプル追加数を求める
    int alignSamples = GetAlignSamples( alignByte, method );
    if ( alignSamples == -1 ) { return -1; }

    int addSamplesForAlign = alignSamples - ( loopStartSample % alignSamples );
    if ( addSamplesForAlign == alignSamples )
    {
        addSamplesForAlign = 0;
    }

    // ループ展開時のサンプル数を計算
    // (alignSamples x 2 以下のループはうまく再生できないので、展開する)
    int expandLoopSamples = 0;
    if ( isExpandLoop )
    {
        u32 loopLength = loopEndSample - loopStartSample;
        int addLoopTimes = alignSamples * 2 / loopLength;
        expandLoopSamples = loopLength * addLoopTimes;
    }

    // 合算する
    int result = loopEndSample + addSamplesForAlign + expandLoopSamples;
    return result;
}

/*!--------------------------------------------------------------------------*
  Name:         AlignLoop

  @brief        ループのスタートポイント・エンドポイントを補正します。

  @param[out]       output          ループ補正したサンプルの格納先です。
  @param[in,out]    loopStartSample ループ開始のサンプルインデックスです。
  @param[in,out]    loopEndSample   ループ終了のサンプルインデックスです。
  @param            input           入力されるサンプルです。
  @param            alignByte       何バイト境界でアライメントするかを設定します。
                                    CTR の場合、PCM8 / PCM16 は 4 バイトの倍数、
                                    DSP ADPCM の場合は 8 バイトの倍数である必要があります。
  @param            isExpandLoop    ストリームファイルのエンコード時に true にします。
                                    ストリームファイルは短いループの場合に
                                    うまく再生できないので、true にしておく必要があります。
  @param            method          エンコード方式です。

  @return       成功したら true, 失敗したら false を返します。
 *---------------------------------------------------------------------------*/
bool AlignLoop(
        s16* output,
        int* loopStartSample,   // in/out
        int* loopEndSample,     // in/out
        const s16* input,
        int alignByte,
        bool isExpandLoop,
        EncodeMethod method )
{
    // 元データの保存
    int tmpLoopStartSample = *loopStartSample;
    int tmpLoopEndSample = *loopEndSample;

    // ひとまず、input → output にコピー
    std::memcpy( output, input, tmpLoopEndSample * 2 );

    // ループのアライメント調整
    int alignSamples = GetAlignSamples( alignByte, method );
    if ( alignSamples == -1 ) { return false; }

    int addSamplesForAlign = alignSamples - ( tmpLoopStartSample % alignSamples );

    AddSamplesToLoopEnd(
            output,
            tmpLoopStartSample,
            tmpLoopEndSample,
            addSamplesForAlign );
    tmpLoopStartSample += addSamplesForAlign;
    tmpLoopEndSample += addSamplesForAlign;

    // 短いループ展開
    if ( isExpandLoop )
    {
        int loopSamples = ( tmpLoopEndSample - tmpLoopStartSample );
        int addLoopTimes =  alignSamples * 2 / loopSamples;
        for ( int i = 0; i < addLoopTimes; i++ )
        {
            AddSamplesToLoopEnd(
                    output,
                    tmpLoopStartSample,
                    tmpLoopEndSample,
                    loopSamples );
            tmpLoopEndSample += loopSamples;
        }
    }
    *loopStartSample = tmpLoopStartSample;
    *loopEndSample = tmpLoopEndSample;
    return true;
}

/*!--------------------------------------------------------------------------*
  @brief    ストリームサウンドのブロック情報です。
 *---------------------------------------------------------------------------*/
struct StreamBlockInfo
{
    u32 blockSamples;           // 1 ブロックあたりのサンプル数
    u32 blockCount;             // 総ブロック数
    u32 blockByte;              // 1 ブロックあたりのバイト数
    u32 lastBlockSamples;       // 最終ブロックのサンプル数
    u32 lastBlockByte;          // 最終ブロックのバイト数
    u32 lastBlockPaddedByte;    // 最終ブロックのバイト数の 32 バイト切り上げた値
};

/*!--------------------------------------------------------------------------*
  Name:         CalcStreamBlockInfo

  @brief        ストリーム用のブロック情報を取得します。

  @param[out]   info        ブロック情報の格納先です。
  @param        alignByte   バイト換算したブロックのサイズです。
                            AlignLoop で渡す alignByte と同じ値です。
  @param        frameCount  サンプルの長さです。
                            ループ波形の場合は loopEndSample を渡します。
  @param        method      エンコード方式です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void CalcStreamBlockInfo(
        StreamBlockInfo*    info,
        int                 alignByte,      // ストリームのブロックサイズ
        int                 frameCount,     // ループ波形なら、loopEndSample を渡す
        EncodeMethod        method )
{
    // blockSamples 算出
    switch ( method )
    {
        case eEncodeMethod_PCM8:
            info->blockSamples = alignByte;
            break;
        case eEncodeMethod_PCM16_BE:
        case eEncodeMethod_PCM16_LE:
            info->blockSamples = alignByte / 2;
            break;
        case eEncodeMethod_DSP_ADPCM:
            info->blockSamples = alignByte * 14 / 8;
            break;
        default:
            info->blockSamples = 0xffffffff;
            break;
    }

    // blockCount, blockByte, lastBlockSamples 算出
    info->blockCount = ( frameCount + ( info->blockSamples - 1 ) ) / info->blockSamples;
    info->blockByte = alignByte;
    info->lastBlockSamples = frameCount - ( info->blockCount - 1 ) * info->blockSamples;

    // lastBlockByte 算出
    switch ( method )
    {
        case eEncodeMethod_PCM8:
            info->lastBlockByte = info->lastBlockSamples;
            break;
        case eEncodeMethod_PCM16_BE:
        case eEncodeMethod_PCM16_LE:
            info->lastBlockByte = info->lastBlockSamples * 2;
            break;
        case eEncodeMethod_DSP_ADPCM:
            info->lastBlockByte = getBytesForAdpcmSamples( info->lastBlockSamples );
            break;
        default:
            info->lastBlockByte = 0xffffffff;
            break;
    }

    // lastBlockPaddedByte 算出
    info->lastBlockPaddedByte =
        ( info->lastBlockByte % 32 == 0 ) ?
        info->lastBlockByte : ( info->lastBlockByte / 32 + 1 ) * 32;
}

/*!--------------------------------------------------------------------------*
  Name:         GetBufferSizeForHalfwaySampleInfo

  @brief        GetHalfwaySampleInfo の出力を格納するのに必要なバッファサイズを取得します。

  @param        blockSamples    ストリームブロックのサイズです。
                                StreamBlockInfo.blockSamples を渡してください。
  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       GetHalfwaySampleInfo の出力を格納するのに必要なバッファサイズ(バイト)を返します。
 *---------------------------------------------------------------------------*/
int GetBufferSizeForHalfwaySampleInfo(
        int blockSamples,
        int frameCount )    // ループ波形なら loopEndSample を渡す
{
    // 途中経過を残す回数
    int interruptCount = frameCount / blockSamples;
    if ( frameCount % blockSamples != 0 )
    {
        interruptCount += 1;
    }

    /*
    frameCount = 10, blockSamples = 3 の場合
    0, 3, 6, 9 の 4 つ分の途中経過が必要。
    これは、10/3 + 1 で算出される。
    */

    int result = interruptCount * ( sizeof(s16) * 2 );
        // 16 ビット / サンプル のサンプルを、2 サンプル分の履歴を残す

    return result;
}

/*!--------------------------------------------------------------------------*
  Name:         GetHalfwaySampleInfo

  @brief        DspAdpcm ストリームの途中再生に必要な情報を取得します。

  @param[out]   output          途中再生に必要な情報の格納先です。
  @param        outputSamples   output に格納できる最大サンプル数です。
  @param        input           入力するサンプルです。
                                ループ補正後のサンプルを渡してください。
  @param        blockSamples    ストリームブロックのサイズです。
                                StreamBlockInfo.blockSamples を渡してください。
  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       成功したら true, 失敗したら false を返します。
 *---------------------------------------------------------------------------*/
bool GetHalfwaySampleInfo(
        s16*        output,
        int         outputSamples,
        const s16*  input,
        int         blockSamples,
        int         frameCount )
{
    int requiredSize = GetBufferSizeForHalfwaySampleInfo( blockSamples, frameCount );
    if ( outputSamples * static_cast<int>(sizeof(s16)) < requiredSize )
    {
        return false;
    }

    int outIndex = 0;
    for ( int readPos = 0; readPos < frameCount; readPos += blockSamples )
    {
        if ( outIndex >= outputSamples )
        {
            break;
        }

        u16 yn1, yn2;
        if ( readPos == 0 )
        {
            yn1 = yn2 = 0;
        }
        else
        {
            yn1 = input[ readPos - 1 ];
            yn2 = input[ readPos - 2 ];
        }
        output[ outIndex ] = yn1;   outIndex++;
        output[ outIndex ] = yn1;   outIndex++;
    }
    return true;
}

/*!--------------------------------------------------------------------------*
  Name:         GetBufferSizeForEncodePcm8

  @brief        EncodePcm8 の出力を格納するのに必要なバッファサイズを取得します。

  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       EncodePcm8 の出力を格納するのに必要なバッファサイズを返します。
 *---------------------------------------------------------------------------*/
int GetBufferSizeForEncodePcm8( int frameCount )
{
    return frameCount;
}

/*!--------------------------------------------------------------------------*
  Name:         GetBufferSizeForEncodePcm16

  @brief        EncodePcm16 の出力を格納するのに必要なバッファサイズを取得します。
                リトルエンディアンとビッグエンディアンの違いはありません。

  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       EncodePcm16 の出力を格納するのに必要なバッファサイズを返します。
 *---------------------------------------------------------------------------*/
int GetBufferSizeForEncodePcm16( int frameCount )
{
    return frameCount * 2;
}

/*!--------------------------------------------------------------------------*
  Name:         GetBufferSizeForEncodeDspAdpcm

  @brief        EncodeDspAdpcm の出力を格納するのに必要なバッファサイズを取得します。

  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       EncodeDspAdpcm の出力を格納するのに必要なバッファサイズを返します。
 *---------------------------------------------------------------------------*/
int GetBufferSizeForEncodeDspAdpcm( int frameCount )
{
    return getBytesForAdpcmBuffer( frameCount );
}

/*!--------------------------------------------------------------------------*
  Name:         EncodePcm8

  @brief        サンプルを PCM8 にエンコードします。

  @param[out]   output          エンコードしたサンプルの格納先です。
  @param        input           エンコードするサンプルです。
  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void EncodePcm8( u8* output, const s16* input, int frameCount )
{
    for ( int i = 0; i < frameCount; i++ )
    {
        s16 sample = input[ i ];
        if ( sample >= 0 )
        {
            sample = ( sample + 0x80 ) >> 8;
            if ( sample > 0x7f ) { sample = 0x7f; }
        }
        else
        {
            sample = - ( ( -sample + 0x80 ) >> 8 );
            if ( sample < -0x80 ) { sample = -0x80; }
        }
        output[ i ] = static_cast<u8>( sample );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         EncodePcm16BE

  @brief        サンプルを PCM16 (ビッグエンディアン) にエンコードします。

  @param[out]   output          エンコードしたサンプルの格納先です。
  @param        input           エンコードするサンプルです。
  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void EncodePcm16BE( u8* output, const s16* input, int frameCount )
{
    for ( int i = 0; i < frameCount; i++ )
    {
        s16 sample = input[ i ];
        output[ i * 2 ]     = static_cast<u8>( ( sample & 0xff00 ) >> 8 );
        output[ i * 2 + 1 ] = static_cast<u8>( sample & 0x00ff );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         EncodePcm16LE

  @brief        サンプルを PCM16 (リトルエンディアン) にエンコードします。

  @param[out]   output          エンコードしたサンプルの格納先です。
  @param        input           エンコードするサンプルです。
  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void EncodePcm16LE( u8* output, const s16* input, int frameCount )
{
    for ( int i = 0; i < frameCount; i++ )
    {
        s16 sample = input[ i ];
        output[ i * 2 ]     = static_cast<u8>( sample & 0x00ff );
        output[ i * 2 + 1 ] = static_cast<u8>( ( sample & 0xff00 ) >> 8 );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         EncodeDspAdpcm

  @brief        サンプルを DSP ADPCM にエンコードします。

  @param[out]   output          エンコードしたサンプルの格納先です。
  @param[out]   info            エンコード時に出力される DSP ADPCM 情報の格納先です。
  @param        input           エンコードするサンプルです。
  @param        loopStartSample ループ開始のサンプルインデックスです。
  @param        frameCount      サンプルの長さです。
                                ループ波形の場合は loopEndSample を渡します。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void EncodeDspAdpcm(
        u8* output,
        ADPCMINFO* info,
        const s16* input,
        int loopStartSample,
        int frameCount )
{
    // TODO: input の const_cast は dsptool.dll が入れ替わったときに削除する
    encode( const_cast<s16*>(input), output, info, frameCount );
    getLoopContext( output, info, static_cast<u32>( loopStartSample ) );
}

s16 GetSample(
        const s16* input,
        int inputFrameCount,
        int inputLoopStartFrame,
        s32 index )
{
    if ( index < 0 ) return 0;
    while ( index >= inputFrameCount )
    {
        index -= ( inputFrameCount - inputLoopStartFrame );
    }
    return input[ index ];
}

s16 ResampleImpl(
        const s16* input,
        int inputFrameCount,
        int inputLoopStartFrame,
        double p )
{
    const s32 p0 = static_cast<s32>( p );
    const s32 left = p0 - RESAMPLE_WIDTH / 2 + 1;
    const double sine = sin(M_PI * (left - p)); // i == 0 時の sin 値

    double x = 0.0;
    register s32 i;
    for ( i = 0; i < RESAMPLE_WIDTH; i++ )
    {
        const s32 point = i + left;
        const double& smp = GetSample(
                input,
                inputLoopStartFrame,
                inputFrameCount,
                point );
        const double ang = M_PI * (point - p);

        x += (ang == 0) ? smp : smp * sine / ang * (i & 1 ? -1 : 1);
    }

    if ( x >= 0 )
    {
        x += 0.5;
        if ( x > 32767 ) x = 32767;
        return static_cast<s16>( x );
    }
    else
    {
        x = -x;
        x += 0.5;
        if ( x > 32768 ) x = 32768;
        return static_cast<s16>( -x );
    }
}

/*!--------------------------------------------------------------------------*
  Name:         Resample

  @brief        サンプルを所定の長さにリサンプルします。

  @param[out]   output                  リサンプルしたサンプルの格納先です。
  @param[out]   outputLoopStartFrame    リサンプル後のループスタートフレーム数です。
                                        自前で計算する場合は NULL を指定してもかまいません。
  @paramm       outputFrameCount        リサンプル後のサンプルの長さです。
                                        ここで指定した長さにリサンプルします。
                                        output はこの長さ分確保しておく必要があります。
  @param[out]   outputSampleRate        リサンプル後のサンプリングレートです。
                                        自前で計算する場合は NULL を指定してもかまいません。
  @param[in]    input                   リサンプルするサンプルです。
  @param[in]    inputLoopStartFrame     リサンプル前のループスタートフレーム数です。
  @param[in]    inputFrameCount         リサンプル前のサンプル数です。
  @param[in]    inputSampleRate         リサンプル前のサンプリングレートです。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void Resample(
        s16* output,
        int outputFrameCount,
        int* outputLoopStartFrame,
        int* outputSampleRate,
        const s16* input,
        int inputFrameCount,        // ループエンド＝波形長 (サンプル単位)
        int inputLoopStartFrame,    // ループスタートフレーム (サンプル単位)
        int inputSampleRate )
{
    double ratio = static_cast<double>( outputFrameCount ) / inputFrameCount;

    for ( int i = 0; i < outputFrameCount; i++ )
    {
        output[i] = ResampleImpl(
                input,
                inputLoopStartFrame,
                inputFrameCount,
                i / ratio );
    }


    // それぞれ NULL を指定する場合は、Resample 前に自前で計算しておいてください
    if ( outputLoopStartFrame != NULL )
    {
        *outputLoopStartFrame = static_cast<int>( inputLoopStartFrame * ratio );
    }
    if ( outputSampleRate != NULL )
    {
        *outputSampleRate = static_cast<int>( inputSampleRate * ratio );
    }
}

#if 0
/*!--------------------------------------------------------------------------*
  Name:         WaveCodecInitialize

  @brief        WaveCodec.dll を初期化します。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void WaveCodecInitialize()
{
    int result = getDll();
    std::printf("[%s] result(%d)\n", __FUNCTION__, result );
}

/*!--------------------------------------------------------------------------*
  Name:         WaveCodecFinalize

  @brief        WaveCodec.dll の終了処理を行います。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void WaveCodecFinalize()
{
    cleanDll();
    std::printf("[%s]\n", __FUNCTION__ );
}

/*!--------------------------------------------------------------------------*
  Name:         Hoge

  @brief        本 DLL および、本 DLL を経由した別 DLL (dsptool.dll) を呼べるか
                確認するための関数です。

  @return       返り値はありません。
 *---------------------------------------------------------------------------*/
void Hoge( int piyo )
{
    int result = getBytesForAdpcmInfo();
    std::printf("[%s] piyo(%d) getBytesForAdpcmInfo(%d)\n",
            __FUNCTION__, piyo, result );
}
#endif
