﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vcclr.h>
#include "MidiIn.h"
#include "MidiPlayer.h"
#include "Channel.h"

class UnmanagedMidiManager :
    public nn::atk::detail::driver::SoundThread::PlayerCallback,
    public nn::atk::detail::driver::NoteOnCallback
{
  public:
    typedef nn::atk::detail::driver::Channel* (*NoteOnCallback)(
        nn::atk::detail::driver::SequenceSoundPlayer* seqPlayer,
        int bankNo,
        const nn::atk::detail::driver::NoteOnInfo& noteOnInfo,
        void* userData
    );

    UnmanagedMidiManager();
    ~UnmanagedMidiManager();

    void Setup( NoteOnCallback noteOnCallback, void* userData );
    void Shutdown();

    void Reset();

    bool OpenMidiIn( UINT deviceID );

    bool IsActive( DWORD msec ) const;

    void SendMidiMessage( int status, int data1, int data2 );
    void SendMidiMessage( UINT devideID, int status, int data1, int data2 );
    void ProgramChange( int channelIndex, int prgNo );
    void ProgramChange( UINT devideID, int channelIndex, int prgNo );

    const nn::atk::detail::driver::MidiSequencePlayer* GetSequenceSoundPlayer( UINT devideID ) const;

  private:
    struct Player {
        MidiIn midiIn;
        MidiPlayer midiPlayer;
        UINT deviceID;
    };

    virtual void OnUpdateFrameSoundThreadWithAudioFrameFrequency()
    {
        UpdateMidiPlayers();
    }

    virtual void OnUpdateFrameSoundThread()
    {
        UpdateMidiPlayers();
    }

    void UpdateMidiPlayers();

    virtual nn::atk::detail::driver::Channel* NoteOn(
        nn::atk::detail::driver::SequenceSoundPlayer* seqPlayer,
        uint8_t bankIndex,
        const nn::atk::detail::driver::NoteOnInfo& noteOnInfo
    );


    typedef std::list<Player*> PlayerList;
    NoteOnCallback mNoteOnCallback;
    void* mUserData;

    MidiPlayer mGuiPlayer;
    PlayerList mPlayerList;
    bool mRegisterCallbackFlag;
    mutable CRITICAL_SECTION mCritSec;
};

namespace NintendoWare { namespace SoundRuntime {

using namespace System;
using namespace System::Runtime::InteropServices;

[StructLayout(LayoutKind::Sequential)]
public value struct NoteOnInfo
{
    Int32 prgNo;
    Int32 key;
    Int32 velocity;
    Int32 length;
    Int32 initPan;
    Int32 priority;
    IntPtr channelCallback;
    IntPtr channelCallbackData;
};

public delegate ChannelCafe^ NoteOnCallback(
    IntPtr seqPlayer,
    int bankNo,
    NoteOnInfo% noteOnInfo,
    Object^ userData
);

public ref class MidiManagerCafe
{
  public:
    MidiManagerCafe() {
        mUnmanaged = new UnmanagedMidiManager();
        mUnmanagedNoteOnCallback = gcnew UnmanagedNoteOnCallback( NoteOn );
        mNoteOnCallbackHandle = GCHandle::Alloc( mUnmanagedNoteOnCallback );
        mGCHandle = GCHandle::Alloc( this );
    }
    ~MidiManagerCafe() {
        Shutdown();
        mNoteOnCallbackHandle.Free();
        mGCHandle.Free();
        delete mUnmanaged;
    }

    void Setup( NoteOnCallback^ callback, Object^ userData )
    {
        IntPtr ip = Marshal::GetFunctionPointerForDelegate(mUnmanagedNoteOnCallback);
        UnmanagedMidiManager::NoteOnCallback cb =
            static_cast<UnmanagedMidiManager::NoteOnCallback>(ip.ToPointer());

        mUserCallback = callback;
        mUserData = userData;

        mUnmanaged->Setup( cb, GCHandle::ToIntPtr(mGCHandle).ToPointer() );
    }
    void Shutdown()
    {
        mUnmanaged->Shutdown();
    }
    bool OpenMidiIn( UInt32 deviceID )
    {
        return mUnmanaged->OpenMidiIn( deviceID );
    }

    void SendMidiMessage( int status, int data1, int data2 )
    {
        mUnmanaged->SendMidiMessage( status, data1, data2 );
    }
    void SendMidiMessage( UINT devideID, int status, int data1, int data2 )
    {
        mUnmanaged->SendMidiMessage( devideID, status, data1, data2 );
    }
    void ProgramChange( int channelIndex, int prgNo )
    {
        mUnmanaged->ProgramChange( channelIndex, prgNo );
    }
    void ProgramChange( UINT devideID, int channelIndex, int prgNo )
    {
        mUnmanaged->ProgramChange( devideID, channelIndex, prgNo );
    }

    bool IsActive( UInt32 msec )
    {
        return mUnmanaged->IsActive( msec );
    }

    void Reset()
    {
        mUnmanaged->Reset();
    }

    IntPtr GetSequenceSoundPlayerHandle( UINT devideID )
    {
        return IntPtr( const_cast<nn::atk::detail::driver::SequenceSoundPlayer*>(
                    static_cast<const nn::atk::detail::driver::SequenceSoundPlayer*>(
                        mUnmanaged->GetSequenceSoundPlayer( devideID ))) );
    }

  private:
    [UnmanagedFunctionPointer(CallingConvention::Cdecl)]
    delegate IntPtr UnmanagedNoteOnCallback(
        IntPtr seqPlayer,
        int bankNo,
        IntPtr noteOnInfo,
        IntPtr userData
    );

    static IntPtr NoteOn(
        IntPtr seqPlayer,
        int bankNo,
        IntPtr noteOnInfoPtr,
        IntPtr userData
    )
    {
        NoteOnInfo% noteOnInfo = safe_cast<NoteOnInfo%>(Marshal::PtrToStructure(noteOnInfoPtr,NoteOnInfo::typeid));
        GCHandle handle = GCHandle::FromIntPtr(userData);
        MidiManagerCafe^ midiManager = safe_cast<MidiManagerCafe^>(handle.Target);

        if ( midiManager->mUserCallback == nullptr ) return IntPtr::Zero;

        ChannelCafe^ channel = midiManager->mUserCallback(
            seqPlayer,
            bankNo,
            noteOnInfo,
            midiManager->mUserData
        );
        if ( channel == nullptr ) return IntPtr::Zero;
        return channel->GetUnmanagedPtr();
    }

    UnmanagedMidiManager* mUnmanaged;
    UnmanagedNoteOnCallback^ mUnmanagedNoteOnCallback;
    GCHandle mNoteOnCallbackHandle;
    GCHandle mGCHandle;

    NoteOnCallback^ mUserCallback;
    Object^ mUserData;
};

}}
