﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(AFX_STREAM_H__F589A8AE_124F_4FD5_8A1C_79ADDE56E6D9__INCLUDED_)
#define AFX_STREAM_H__F589A8AE_124F_4FD5_8A1C_79ADDE56E6D9__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <cstdio>
#include <cassert>
#include <list>

#include "types.h"
#include "endian.h"

namespace sndlib {
namespace strm {

struct failure {};

class fstream
{
  public:
    fstream() : fp(NULL) {}
    ~fstream() { if (fp) fclose(fp); }
    bool open(const char *s, const char* mode) {
        fp = fopen(s, mode);
        return fp!=NULL;
    }
    int close() { assert(is_open()); int ret = fclose(fp); fp = NULL; return ret; }

    size_t read(void* buffer, size_t size) { return fread(buffer, 1, size, fp); }
    size_t write(const void* buffer, size_t size) { return fwrite(buffer, 1, size, fp);  }
    void fill(size_t size, unsigned char c) { for(unsigned int i=0;i<size;i++) fwrite( &c, 1, 1, fp);  }

    long tell() { return ftell(fp); }
    int seek(long offset, int whence = SEEK_SET) { return fseek(fp, offset, whence); }
    int peek() { int c = fgetc(fp); if (feof(fp)) return EOF; ungetc(c,fp); return c; }

    int flush() { return fp ? fflush(fp) : EOF; }

    void fail() { throw failure(); }

    bool is_open() const { return fp!=NULL; }
    bool operator!() { return fp==NULL; }

  private:
    FILE* fp;
};

struct align {
    int n;

    explicit align(int i) : n(i) {}

    fstream& align_write(fstream& s) const {
        long pos = s.tell();
        int a = (pos % n);
        if (a > 0) a = n - a;
        const char c = '\0';
        while(a > 0) {
            s.write(&c, 1);
            --a;
        }
        return s;
    }
    fstream& align_read(fstream& s) const {
        long pos = s.tell();
        int a = (pos % n);
        if (a > 0) a = n - a;
        s.seek(a, SEEK_CUR);
        return s;
    }
};

template<typename C, typename Endian>
class binary {
  public:
    typedef C value_type;
    typedef Endian endian_type;

    binary() { }
    binary(const value_type& rhs) : v( rhs ) { }
    binary& operator= (const value_type& rhs) { v = rhs; return *this; }
    binary& operator+= (const value_type& rhs) { v += rhs; return *this; }
    binary& operator-= (const value_type& rhs) { v -= rhs; return *this; }
    binary& operator&= (const value_type& rhs) { v &= rhs; return *this; }
    binary& operator|= (const value_type& rhs) { v |= rhs; return *this; }
    operator value_type() const {return v;}
  private:
    value_type v;
};

typedef binary<uint32_t, BigEndian> char4;

#if 0
template<typename Endian>
class ChunkWriter
{
    typedef binary<uint32_t, Endian> size_type;

  public:
    ChunkWriter(strm::fstream& to_, unsigned long id_)
    : to(to_) {
        char4 chunkID;
        size_type chunkSize = 0;
        chunkID = id_;
        to << chunkID << chunkSize;
        pos = to.tell();
    }
    ~ChunkWriter() {
        long cur = to.tell();
        size_type chunkSize;
        chunkSize = cur - pos;
        to.seek(static_cast<long>(pos) - 4);
        to << chunkSize;
        to.seek(cur);
        to << align(2);
    }
  private:
    strm::fstream& to;
    long pos;
};
#endif

template<int SIZE, typename Endian>
class OffsetWriteback
{
    union Buf_ {
        char b[SIZE];
        long l;
    };

  public:
    OffsetWriteback()
    : writeback_(false) {}
    ~OffsetWriteback() {
#ifdef _DEBUG
        assert(list_.empty());
#endif
    }

    void push_cur(strm::fstream& to_) const {
        if (writeback_) {
            Buf_ u;
            memset(&u, 0, SIZE);
            u.l = ofs_;
            Endian::convert( u.b, SIZE );
            to_.write(u.b, SIZE);
        }
        else {
            Pos pos( to_, to_.tell() );
            list_.push_back( pos );
            char buf[SIZE];
            memset(buf, 0, SIZE);
            to_.write(buf, SIZE);
        }
    }
    void writeback(strm::fstream& to_, int ofs_base = 0 ) const {
        assert(! writeback_);

        ofs_ = to_.tell() - ofs_base;

        Buf_ u;
        memset(&u, 0, SIZE);
        u.l = ofs_;
        Endian::convert( u.b, SIZE );

        std::list<Pos>::const_iterator p;
        for(p = list_.begin(); p != list_.end(); ++p) {
            long oldPos = p->to.tell();
            p->to.seek(p->pos);
            p->to.write(u.b, SIZE);
            p->to.seek(oldPos);
        }

        list_.clear();

        writeback_ = true;
    }
    long get_offset() const { assert( writeback_); return ofs_; }

  private:
    struct Pos
    {
        Pos( strm::fstream& to_, long pos_ )
        : to( to_ ), pos( pos_ )
        {}

        Pos& operator =(const Pos& source)
        {
            to = source.to;
            pos = source.pos;
            return *this;
        }

        strm::fstream& to;
        long pos;
    };
    mutable std::list<Pos> list_;
    mutable long ofs_;
    mutable bool writeback_;
};

template<int SIZE, typename Endian>
class SizeWriteback
{
    union Buf_ {
        char b[SIZE];
        long l;
    };

  public:
    SizeWriteback()
    : begin_flag_(false), writeback_(false) {}
    ~SizeWriteback() {
#ifdef _DEBUG
        assert(list_.empty());
#endif
    }

    void push_cur(strm::fstream& to_) const {
        if (writeback_) {
            Buf_ u;
            memset(&u, 0, SIZE);
            u.l = size_;
            Endian::convert( u.b, SIZE );
            to_.write(u.b, SIZE);
        }
        else {
            Pos pos( to_, to_.tell() );
            list_.push_back(pos);
            char buf[SIZE];
            memset(buf, 0, SIZE);
            to_.write(buf, SIZE);
        }
    }
    void begin(strm::fstream& to_ ) const {
        assert( ! begin_flag_ && ! writeback_);

        begin_ = to_.tell();

        begin_flag_ = true;
    }
    void end(strm::fstream& to_) const {
        assert( begin_flag_ && ! writeback_);

        size_ = to_.tell() - begin_;

        Buf_ u;
        memset(&u, 0, SIZE);
        u.l = size_;
        Endian::convert( u.b, SIZE );

        std::list<Pos>::const_iterator p;
        for(p = list_.begin(); p != list_.end(); ++p) {
            long oldPos = p->to.tell();
            p->to.seek(p->pos);
            p->to.write(u.b, SIZE);
            p->to.seek( oldPos );
        }

        list_.clear();

        writeback_ = true;
    }
    long size() const { assert( writeback_ ); return size_; }
  private:
    struct Pos
    {
        Pos( strm::fstream& to_, long pos_ )
        : to( to_ ), pos( pos_ )
        {}

        strm::fstream& to;
        long pos;
    };
    mutable std::list<Pos> list_;
    mutable long begin_;
    mutable long size_;
    mutable bool begin_flag_;
    mutable bool writeback_;
};

template<typename Endian>
struct ImportBinaryType {
    typedef strm::binary<uint8_t,  Endian> u8;
    typedef strm::binary<int8_t,   Endian> s8;
    typedef strm::binary<uint16_t, Endian> u16;
    typedef strm::binary<int16_t,  Endian> s16;
    typedef strm::binary<uint32_t, Endian> u32;
    typedef strm::binary<int32_t,  Endian> s32;
    typedef strm::binary<float32_t, Endian> f32;
};

template<typename C, typename E>
fstream& operator>>(fstream& s, binary<C, E>& a)
{
    if ( s.read((char*)&a, sizeof(a)) != sizeof(a) ) s.fail();
    a = E::convert(a);
    return s;
}

template<typename C, typename E>
fstream& operator<<(fstream& s, const binary<C, E>& a)
{
    const binary<C, E> b = E::convert(a);
    s.write((const char*)&b, sizeof(b));
    return s;
}

inline
fstream& operator>>(fstream& s, align& a)
{
    return a.align_read(s);
}

inline
fstream& operator<<(fstream& s, const align& a)
{
    return a.align_write(s);
}

inline
fstream& operator<<(fstream& s, const std::string& str)
{
    for( std::string::const_iterator p = str.begin() ;
         p != str.end() ; ++p )
    {
        s.write( &*p, 1 );
    }
    s.fill( 1, '\0' );
    return s;
}
} // namespace strm


#if 0
struct DataRef
{
    DataRef( int refType_, int dataType_ )
    : refType( refType_ ), dataType( dataType_ ),
      padding0(0),padding1(0)
    {
    }

    int refType;
    int dataType;
    int padding0;
    int padding1;
};

inline strm::fstream& operator>>(strm::fstream& s, DataRef& a)
{
    char buf[4];
    if ( s.read(buf, sizeof(buf)) != sizeof(buf) ) s.fail();
    a.refType  = buf[0];
    a.dataType = buf[1];
    a.padding0 = buf[2];
    a.padding1 = buf[3];
    return s;
}

inline strm::fstream& operator<<(strm::fstream& s, const DataRef& a)
{
    char buf[4];
    buf[0] = a.refType;
    buf[1] = a.dataType;
    buf[2] = a.padding0;
    buf[3] = a.padding1;
    s.write(buf,sizeof(buf));
    return s;
}
#endif

} // namespace sndlib

#endif // !defined(AFX_STREAM_H__F589A8AE_124F_4FD5_8A1C_79ADDE56E6D9__INCLUDED_)

