﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.Alto.Foundation.Communications
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Threading;

    /// <summary>
    /// Common 向けの HostIO 機能を提供します。
    /// </summary>
    public sealed class HostIO : IComEndPoint
    {
        //private static readonly Lazy<HostIO> instance = new Lazy<HostIO>(() => new HostIO());
        private static readonly HostIO instance = new HostIO();

        private static readonly object connectLock = new object();

        private readonly Dictionary<string, HostIOChannelInfo> channelInfos =
            new Dictionary<string, HostIOChannelInfo>();

        private int connectCount = 0;
        private HostIOPortMappingChannel portMappingChannel;

        private SynchronizationContext syncContext;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private HostIO()
        {
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// インスタンスを取得します。
        /// </summary>
        public static IComEndPoint Instance
        {
            get { return HostIO.instance; }
        }

        /// <summary>
        /// 接続の有無を取得します。
        /// </summary>
        public bool IsConnected
        {
            get
            {
                return this.connectCount > 0 &&
                    this.portMappingChannel != null &&
                    this.portMappingChannel.IsOpened;
            }
        }

        /// <summary>
        /// エンディアンを取得します。
        /// </summary>
        public bool IsLittleEndian
        {
            get { return false; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 接続を開始します。
        /// </summary>
        /// <param name="syncContext">同期コンテキストを指定します。</param>
        public void Connect(SynchronizationContext syncContext)
        {
            //Ensure.Argument.NotNull(syncContext);

            lock (connectLock)
            {
                if (this.portMappingChannel != null)
                {
                    return;
                }

                this.syncContext = syncContext;

                try
                {
                    this.portMappingChannel = new HostIOPortMappingChannel();

                    this.portMappingChannel.ChannelOpend += (sender, e) =>
                    {
                        var context = this.syncContext;
                        if (context != null)
                        {
                            context.Post(state => this.OnChannelOpened(((HostIOChannelEventArgs)state).ChannelInfo), e);
                        }
                        else
                        {
                            this.OnChannelOpened(e.ChannelInfo);
                        }
                    };

                    this.portMappingChannel.ChannelClosed += (sender, e) =>
                    {
                        var context = this.syncContext;
                        if (context != null)
                        {
                            context.Post(state => this.OnChannelClosed(((HostIOChannelEventArgs)state).ChannelInfo), e);
                        }
                        else
                        {
                            this.OnChannelClosed(e.ChannelInfo);
                        }
                    };

                    this.portMappingChannel.Open();
                }
                catch
                {
                    try
                    {
                        this.Disconnect();
                    }
                    catch { }

                    throw;
                }

                this.connectCount++;
            }
        }

        /// <summary>
        /// 切断します。
        /// </summary>
        public void Disconnect()
        {
            lock (connectLock)
            {
                if (this.portMappingChannel == null)
                {
                    return;
                }

                if (this.connectCount > 0)
                {
                    this.connectCount--;

                    if (this.connectCount > 0)
                    {
                        return;
                    }
                }

                //Assertion.Operation.NotNull(this.portMappingChannel);

                this.portMappingChannel.Close();
                this.portMappingChannel = null;

                this.channelInfos.Clear();
            }
        }

        public IComChannel OpenChannel(string portName)
        {
            lock (connectLock)
            {
                HostIOChannelInfo channelInfo;

                if (!this.channelInfos.TryGetValue(portName, out channelInfo))
                {
                    return null;
                }

                return HostIOChannel.Open(channelInfo.Port, portName);
            }
        }

        IComChannel IComEndPoint.OpenChannel(object port)
        {
            //Ensure.Argument.True(port is string);
            return this.OpenChannel(port as string);
        }

        private void OnChannelOpened(HostIOChannelInfo channelInfo)
        {
            Debug.WriteLine(string.Format("[HostIO] ChannelOpened : Name={0}, Port={1}.", channelInfo.Name, channelInfo.Port));

            if (!this.IsConnected)
            {
                return;
            }

            if (this.channelInfos.ContainsKey(channelInfo.Name))
            {
                return;
            }

            this.channelInfos.Add(channelInfo.Name, channelInfo);
        }

        private void OnChannelClosed(HostIOChannelInfo channelInfo)
        {
            Debug.WriteLine(string.Format("[HostIO] ChannelClosed : Name={0}, Port={1}.", channelInfo.Name, channelInfo.Port));

            this.channelInfos.Remove(channelInfo.Name);
        }
    }
}
