﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using NintendoWare.SoundFoundation.Legacies.FileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rXmlSoundArchiveItemList : IEnumerable
    {
        ArrayList itemList = new ArrayList();

        /******************************************************************************
           interface
         ******************************************************************************/
        IEnumerator IEnumerable.GetEnumerator()
        {
            return itemList.GetEnumerator();
        }

        /******************************************************************************
           public
         ******************************************************************************/
        public int Count
        {
            get { return itemList.Count; }
        }

        public void CheckDuplicateLabel()
        {
            Hashtable hash = new Hashtable();
            foreach (Nw4rXmlSoundArchiveItem item in itemList)
            {
                if (hash.ContainsKey(item.Label))
                {
                    throw new Nw4rFileFormatException(
                        "Multiple defined label \"" + item.Label + "\"");
                }
                hash.Add(item.Label, null);
            }
        }

        public bool Contains(string label)
        {
            foreach (Nw4rXmlSoundArchiveItem item in itemList)
            {
                if (label == item.Label) return true;
            }
            return false;
        }
        public void Clear()
        {
            itemList.Clear();
        }

        /******************************************************************************
           protected internal
         ******************************************************************************/
        protected internal void Add(Nw4rXmlSoundArchiveItem item)
        {
            itemList.Add(item);
        }
        protected internal void Remove(Nw4rXmlSoundArchiveItem item)
        {
            itemList.Remove(item);
        }
    }

    public class Nw4rXmlSoundArchiveFileList : Nw4rXmlSoundArchiveItemList
    {
        public void Add(Nw4rXmlSoundArchiveFile file)
        {
            base.Add(file);
        }
        public void Remove(Nw4rXmlSoundArchiveFile file)
        {
            base.Remove(file);
        }
    }

    public class Nw4rXmlSoundArchiveSoundList : Nw4rXmlSoundArchiveItemList
    {
        public void Add(Nw4rXmlSoundArchiveSound sound)
        {
            base.Add(sound);
        }
        public void Remove(Nw4rXmlSoundArchiveSound sound)
        {
            base.Remove(sound);
        }
    }

    public class Nw4rXmlSoundArchiveBankList : Nw4rXmlSoundArchiveItemList
    {
        public void Add(Nw4rXmlSoundArchiveBank bank)
        {
            base.Add(bank);
        }
        public void Remove(Nw4rXmlSoundArchiveBank bank)
        {
            base.Remove(bank);
        }
    }

    public class Nw4rXmlSoundArchivePlayerList : Nw4rXmlSoundArchiveItemList
    {
        public void Add(Nw4rXmlSoundArchivePlayer player)
        {
            base.Add(player);
        }
        public void Remove(Nw4rXmlSoundArchivePlayer player)
        {
            base.Remove(player);
        }
    }

    public class Nw4rXmlSoundArchiveGroupList : Nw4rXmlSoundArchiveItemList
    {
        public void Add(Nw4rXmlSoundArchiveGroup group)
        {
            base.Add(group);
        }
        public void Remove(Nw4rXmlSoundArchiveGroup group)
        {
            base.Remove(group);
        }
    }

    public abstract class Nw4rXmlSoundArchiveItem
    {
        string label;

        public Nw4rXmlSoundArchiveItem(string label)
        {
            this.label = label;
        }

        public string Label
        {
            set { label = value; }
            get { return label; }
        }

        internal abstract XmlElement ToXmlElement(XmlDocument doc);
    }

    public abstract class Nw4rXmlSoundArchiveSound : Nw4rXmlSoundArchiveItem
    {
        Nw4rSound3DParam sound3DParam = new Nw4rSound3DParam();
        int volume = 127;
        int remoteFilter = 0;
        ulong userParam = 0;
        int playerPriority = 64;
        string playerLabel;
        int actorPlayer = 0;

        public Nw4rSound3DParam Sound3DParam
        {
            get { return sound3DParam; }
            set { sound3DParam = value; }
        }

        public int RemoteFilter
        {
            set { remoteFilter = value; }
            get { return remoteFilter; }
        }
        public int Volume
        {
            set { volume = value; }
            get { return volume; }
        }
        public ulong UserParam
        {
            set { userParam = value; }
            get { return userParam; }
        }
        public int PlayerPrio
        {
            set { playerPriority = value; }
            get { return playerPriority; }
        }
        public string Player
        {
            set { playerLabel = value; }
            get { return playerLabel; }
        }
        public int ActorPlayer
        {
            set { actorPlayer = value; }
            get { return actorPlayer; }
        }

        public Nw4rXmlSoundArchiveSound(string label)
            : base(label)
        {
        }

        public void ParseSoundElement(XmlDocument doc, XmlElement seqSoundElem)
        {
            XmlElement elem;

            elem = (XmlElement)seqSoundElem.SelectSingleNode("volume");
            if (elem != null)
            {
                volume = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("remote_filter");
            if (elem != null)
            {
                remoteFilter = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("user_param");
            if (elem != null)
            {
                userParam = UInt32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("player_prio");
            if (elem != null)
            {
                playerPriority = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("player");
            if (elem != null)
            {
                playerLabel = elem.InnerText;
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("actor_player");
            if (elem != null)
            {
                actorPlayer = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("sound3d_param");
            if (elem != null)
            {
                sound3DParam = Nw4rSound3DParam.Parse(doc, elem);
            }
        }

        public void AppendSoundElement(XmlDocument doc, XmlElement seqElem)
        {
            XmlElement elem;
            {
                elem = doc.CreateElement("volume");
                elem.InnerText = this.volume.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("user_param");
                elem.InnerText = this.userParam.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("remote_filter");
                elem.InnerText = this.remoteFilter.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("player_prio");
                elem.InnerText = this.playerPriority.ToString();
                seqElem.AppendChild(elem);
            }
            if (this.Player != null)
            {
                elem = doc.CreateElement("player");
                elem.InnerText = this.playerLabel;
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("actor_player");
                elem.InnerText = this.actorPlayer.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = Sound3DParam.ToXmlElement(doc);
                seqElem.AppendChild(elem);
            }
        }
    }

    public struct Nw4rXmlSoundArchiveFileType
    {
        public const int Invalid = 0;
        public const int Seq = 1;
        public const int Bank = 2;
        public const int Stream = 3;
        public const int WaveSoundSet = 4;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Seq, "Seq" ),
            new KeyValuePair<int, string>( Bank, "Bank" ),
            new KeyValuePair<int, string>( Stream, "Stream" ),
            new KeyValuePair<int, string>( WaveSoundSet, "WaveSoundSet" ),
        };

        public static implicit operator int(Nw4rXmlSoundArchiveFileType s) { return s.val_; }
        public static implicit operator Nw4rXmlSoundArchiveFileType(int s) { return new Nw4rXmlSoundArchiveFileType(s); }
        public static Nw4rXmlSoundArchiveFileType Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlSoundArchiveFileType(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rXmlSoundArchiveInputFileType
    {
        public const int Invalid = 0;
        public const int Xml = 1;
        public const int Text = 2;
        public const int Binary = 3;
        public const int Smf = 4;
        public const int Wave = 5;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Xml, "Xml" ),
            new KeyValuePair<int, string>( Text, "Text" ),
            new KeyValuePair<int, string>( Binary, "Binary" ),
            new KeyValuePair<int, string>( Smf, "Smf" ),
            new KeyValuePair<int, string>( Wave, "Wave" ),
        };

        public static implicit operator int(Nw4rXmlSoundArchiveInputFileType s) { return s.val_; }
        public static implicit operator Nw4rXmlSoundArchiveInputFileType(int s) { return new Nw4rXmlSoundArchiveInputFileType(s); }
        public static Nw4rXmlSoundArchiveInputFileType Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlSoundArchiveInputFileType(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rXmlSoundArchiveEncodeType
    {
        public const int Invalid = 0;
        public const int Adpcm = 1;
        public const int Pcm16 = 2;
        public const int Pcm8 = 3;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Adpcm, "Adpcm" ),
            new KeyValuePair<int, string>( Pcm16, "Pcm16" ),
            new KeyValuePair<int, string>( Pcm8, "Pcm8" ),
        };

        public static implicit operator int(Nw4rXmlSoundArchiveEncodeType s) { return s.val_; }
        public static implicit operator Nw4rXmlSoundArchiveEncodeType(int s) { return new Nw4rXmlSoundArchiveEncodeType(s); }
        public static Nw4rXmlSoundArchiveEncodeType Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlSoundArchiveEncodeType(int e) { val_ = e; }
        private int val_;
    }

    public class Nw4rXmlSoundArchiveFile : Nw4rXmlSoundArchiveItem
    {
        Nw4rXmlSoundArchiveFileType fileType;
        Nw4rXmlSoundArchiveInputFileType inputType;
        Nw4rXmlSoundArchiveEncodeType outputType;
        string filePath;
        string extFilePath;
        int smfconvTimebase = 48;

        public Nw4rXmlSoundArchiveFile(string label)
        : base(label)
        {
        }
        public Nw4rXmlSoundArchiveFileType FileType
        {
            set { fileType = value; }
            get { return fileType; }
        }
        public Nw4rXmlSoundArchiveInputFileType InputType
        {
            set { inputType = value; }
            get { return inputType; }
        }
        public Nw4rXmlSoundArchiveEncodeType EncodeType
        {
            set { outputType = value; }
            get { return outputType; }
        }
        public string FilePath
        {
            set { filePath = value; }
            get { return filePath; }
        }
        public string ExtFilePath
        {
            set { extFilePath = value; }
            get { return extFilePath; }
        }
        public int SmfConvertTimebase
        {
            set { smfconvTimebase = value; }
            get { return smfconvTimebase; }
        }
        public override int GetHashCode()
        {
            int code = fileType.GetHashCode();
            code ^= inputType.GetHashCode();
            code ^= outputType.GetHashCode();
            if (filePath != null)
            {
                code ^= filePath.GetHashCode();
            }
            return code;
        }

        internal static Nw4rXmlSoundArchiveFile Parse(XmlDocument doc, XmlElement fileElem)
        {
            XmlElement elem; // テンポラリ用

            if (!fileElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = fileElem.GetAttribute("name");
            Nw4rXmlSoundArchiveFile file = new Nw4rXmlSoundArchiveFile(label);

            if (fileElem.HasAttribute("path"))
            {
                elem = (XmlElement)fileElem.SelectSingleNode("path");
                if (elem != null)
                {
                    file.FilePath = elem.InnerText;
                }

                elem = (XmlElement)fileElem.SelectSingleNode("ext_path");
                if (elem != null)
                {
                    file.ExtFilePath = elem.InnerText;
                }

                elem = (XmlElement)fileElem.SelectSingleNode("type");
                if (elem != null)
                {
                    file.FileType = Nw4rXmlSoundArchiveFileType.Parse(elem.InnerText);
                }

                elem = (XmlElement)fileElem.SelectSingleNode("input");
                if (elem != null)
                {
                    file.InputType = Nw4rXmlSoundArchiveInputFileType.Parse(elem.InnerText);
                }

                elem = (XmlElement)fileElem.SelectSingleNode("encode");
                if (elem != null)
                {
                    file.EncodeType = Nw4rXmlSoundArchiveEncodeType.Parse(elem.InnerText);
                }

                elem = (XmlElement)fileElem.SelectSingleNode("smfconv_timebase");
                if (elem != null)
                {
                    file.SmfConvertTimebase = Int32.Parse(elem.InnerText);
                }
            }

            return file;
        }

        internal override XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement fileElem = doc.CreateElement("file");
            fileElem.SetAttribute("name", this.Label);

            if (this.FilePath != null)
            {
                if (this.FilePath != null)
                {
                    elem = doc.CreateElement("path");
                    elem.InnerText = this.FilePath;
                    fileElem.AppendChild(elem);
                }
                if (this.ExtFilePath != null)
                {
                    elem = doc.CreateElement("ext_path");
                    elem.InnerText = this.ExtFilePath;
                    fileElem.AppendChild(elem);
                }
                if (this.FileType != Nw4rXmlSoundArchiveFileType.Invalid)
                {
                    elem = doc.CreateElement("type");
                    elem.InnerText = this.FileType.ToString();
                    fileElem.AppendChild(elem);
                }
                if (this.InputType != Nw4rXmlSoundArchiveInputFileType.Invalid)
                {
                    elem = doc.CreateElement("input");
                    elem.InnerText = this.InputType.ToString();
                    fileElem.AppendChild(elem);
                }
                if (this.EncodeType != Nw4rXmlSoundArchiveEncodeType.Invalid)
                {
                    elem = doc.CreateElement("encode");
                    elem.InnerText = this.EncodeType.ToString();
                    fileElem.AppendChild(elem);
                }
                if (this.InputType == Nw4rXmlSoundArchiveInputFileType.Smf)
                {
                    elem = doc.CreateElement("smfconv_timebase");
                    elem.InnerText = this.SmfConvertTimebase.ToString();
                    fileElem.AppendChild(elem);
                }
            }

            return fileElem;
        }
    }

    public class Nw4rXmlSoundArchiveSeq : Nw4rXmlSoundArchiveSound
    {
        string fileLabel;
        string bankLabel;
        int channelPriority = 64;
        bool releasePriorityFixFlag = false;
        string startPosition;
#if ENABLE_COMMENT
    string comment;
#endif

        public Nw4rXmlSoundArchiveSeq(string label)
        : base(label)
        {
        }
        public string FileLabel
        {
            set { fileLabel = value; }
            get { return fileLabel; }
        }
        public string StartPosition
        {
            set { startPosition = value; }
            get { return startPosition; }
        }
        public string Bank
        {
            set { bankLabel = value; }
            get { return bankLabel; }
        }
        public int ChannelPrio
        {
            set { channelPriority = value; }
            get { return channelPriority; }
        }
        public bool ReleasePrioFixFlag
        {
            set { releasePriorityFixFlag = value; }
            get { return releasePriorityFixFlag; }
        }
#if ENABLE_COMMENT
    public string Comment
    {
        set { comment = value; }
        get { return comment; }
    }
#endif

        internal static Nw4rXmlSoundArchiveSeq Parse(XmlDocument doc, XmlElement seqSoundElem)
        {
            XmlElement elem; // テンポラリ用

            if (!seqSoundElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = seqSoundElem.GetAttribute("name");
            Nw4rXmlSoundArchiveSeq seq = new Nw4rXmlSoundArchiveSeq(label);

            XmlElement bankArrayElem = (XmlElement)seqSoundElem.SelectSingleNode("bank_array");
            if (bankArrayElem != null)
            {
                elem = (XmlElement)bankArrayElem.SelectSingleNode("bank");
                if (elem != null)
                {
                    seq.Bank = elem.InnerText;
                }
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("file");
            if (elem != null)
            {
                seq.FileLabel = elem.InnerText;
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("start_position");
            if (elem != null)
            {
                seq.StartPosition = elem.InnerText;
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("channel_prio");
            if (elem != null)
            {
                seq.ChannelPrio = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)seqSoundElem.SelectSingleNode("channel_release_prio");
            if (elem != null)
            {
                if (elem.HasAttribute("use_channel_prio"))
                {
                    seq.ReleasePrioFixFlag = Boolean.Parse(elem.GetAttribute("use_channel_prio"));
                }
            }

#if ENABLE_COMMENT
        elem = (XmlElement)seqSoundElem.SelectSingleNode( "comment" );
        if ( elem != null )
        {
            seq.Comment = elem.InnerText;
        }
#endif

            seq.ParseSoundElement(doc, seqSoundElem);

            return seq;
        }

        internal override XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement seqElem = doc.CreateElement("seq_sound");
            seqElem.SetAttribute("name", this.Label);
            if (this.bankLabel != null)
            {
                XmlElement bankArrayElem = doc.CreateElement("bank_array");
                elem = doc.CreateElement("bank");
                elem.InnerText = this.bankLabel;
                elem.SetAttribute("index", "0");
                bankArrayElem.AppendChild(elem);
                seqElem.AppendChild(bankArrayElem);
            }
            if (this.FileLabel != null)
            {
                elem = doc.CreateElement("file");
                elem.InnerText = this.FileLabel;
                seqElem.AppendChild(elem);
            }
            if (this.StartPosition != null && this.StartPosition != String.Empty)
            {
                elem = doc.CreateElement("start_position");
                elem.InnerText = this.startPosition;
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_prio");
                elem.InnerText = this.channelPriority.ToString();
                seqElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_release_prio");
                elem.SetAttribute("use_channel_prio", this.releasePriorityFixFlag.ToString());
                seqElem.AppendChild(elem);
            }
#if ENABLE_COMMENT
        if ( comment != null )
        {
            elem = doc.CreateElement( "comment" );
            elem.InnerText = this.comment;
            seqElem.AppendChild( elem );
        }
#endif

            AppendSoundElement(doc, seqElem);

            return seqElem;
        }
    }

    public class Nw4rXmlSoundArchiveBank : Nw4rXmlSoundArchiveItem
    {
        string fileLabel;
#if ENABLE_COMMENT
    string comment;
#endif
        public Nw4rXmlSoundArchiveBank(string label)
        : base(label)
        {
        }
        public string FileLabel
        {
            set { fileLabel = value; }
            get { return fileLabel; }
        }
#if ENABLE_COMMENT
    public string Comment
    {
        set { comment = value; }
        get { return comment; }
    }
#endif
        internal static Nw4rXmlSoundArchiveBank Parse(XmlDocument doc, XmlElement bankElem)
        {
            XmlElement elem; // テンポラリ用

            if (!bankElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = bankElem.GetAttribute("name");
            Nw4rXmlSoundArchiveBank bank = new Nw4rXmlSoundArchiveBank(label);

            elem = (XmlElement)bankElem.SelectSingleNode("file");
            if (elem != null)
            {
                bank.FileLabel = elem.InnerText;
            }

#if ENABLE_COMMENT
        elem = (XmlElement)bankElem.SelectSingleNode( "comment" );
        if ( elem != null )
        {
            bank.Comment = elem.InnerText;
        }
#endif
            return bank;
        }

        internal override XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement bankElem = doc.CreateElement("bank");
            bankElem.SetAttribute("name", this.Label);
            if (this.FileLabel != null)
            {
                elem = doc.CreateElement("file");
                elem.InnerText = this.FileLabel;
                bankElem.AppendChild(elem);
            }
#if ENABLE_COMMENT
        if ( comment != null )
        {
            elem = doc.CreateElement( "comment" );
            elem.InnerText = this.comment;
            bankElem.AppendChild( elem );
        }
#endif
            return bankElem;
        }
    }

    public class Nw4rXmlSoundArchiveStrm : Nw4rXmlSoundArchiveSound
    {
        string fileLabel;
        Nw4rXmlPanMode panMode = new Nw4rXmlPanMode();
        Nw4rXmlPanCurve panCurve = new Nw4rXmlPanCurve();
#if ENABLE_COMMENT
    string comment;
#endif
        public Nw4rXmlSoundArchiveStrm(string label)
        : base(label)
        {
        }
        public string FileLabel
        {
            set { fileLabel = value; }
            get { return fileLabel; }
        }
        public Nw4rXmlPanMode PanMode
        {
            set { panMode = value; }
            get { return panMode; }
        }
        public Nw4rXmlPanCurve PanCurve
        {
            set { panCurve = value; }
            get { return panCurve; }
        }
#if ENABLE_COMMENT
    public string Comment
    {
        set { comment = value; }
        get { return comment; }
    }
#endif
        internal static Nw4rXmlSoundArchiveStrm Parse(XmlDocument doc, XmlElement strmSoundElem)
        {
            XmlElement elem; // テンポラリ用

            if (!strmSoundElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = strmSoundElem.GetAttribute("name");
            Nw4rXmlSoundArchiveStrm strm = new Nw4rXmlSoundArchiveStrm(label);

            elem = (XmlElement)strmSoundElem.SelectSingleNode("file");
            if (elem != null)
            {
                strm.FileLabel = elem.InnerText;
            }

            elem = (XmlElement)strmSoundElem.SelectSingleNode("pan_mode");
            if (elem != null)
            {
                strm.PanMode = Nw4rXmlPanMode.Parse(elem.InnerText);
            }

            elem = (XmlElement)strmSoundElem.SelectSingleNode("pan_curve");
            if (elem != null)
            {
                strm.PanCurve = Nw4rXmlPanCurve.Parse(elem.InnerText);
            }

#if ENABLE_COMMENT
        elem = (XmlElement)strmSoundElem.SelectSingleNode( "comment" );
        if ( elem != null )
        {
            strm.Comment = elem.InnerText;
        }
#endif
            strm.ParseSoundElement(doc, strmSoundElem);

            return strm;
        }

        internal override XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement strmElem = doc.CreateElement("strm_sound");
            strmElem.SetAttribute("name", this.Label);
            if (this.FileLabel != null)
            {
                elem = doc.CreateElement("file");
                elem.InnerText = this.FileLabel;
                strmElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pan_mode");
                elem.InnerText = this.PanMode.ToString();
                strmElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pan_curve");
                elem.InnerText = this.PanCurve.ToString();
                strmElem.AppendChild(elem);
            }
#if ENABLE_COMMENT
        if ( comment != null )
        {
            elem = doc.CreateElement( "comment" );
            elem.InnerText = this.comment;
            strmElem.AppendChild( elem );
        }
#endif
            AppendSoundElement(doc, strmElem);

            return strmElem;
        }
    }
    public class Nw4rXmlSoundArchiveWaveSound : Nw4rXmlSoundArchiveSound
    {
        string fileLabel;
        string soundName;
        int soundIndex;
        bool releasePriorityFixFlag = false;
        int channelPriority = 64;
        Nw4rXmlPanMode panMode = new Nw4rXmlPanMode();
        Nw4rXmlPanCurve panCurve = new Nw4rXmlPanCurve();
#if ENABLE_COMMENT
    string comment;
#endif
        public Nw4rXmlSoundArchiveWaveSound(string label)
        : base(label)
        {
        }
        public string FileLabel
        {
            set { fileLabel = value; }
            get { return fileLabel; }
        }
        public string SoundName
        {
            set { soundName = value; }
            get { return soundName; }
        }
        public int SoundIndex
        {
            set { soundIndex = value; }
            get { return soundIndex; }
        }
        public int ChannelPrio
        {
            set { channelPriority = value; }
            get { return channelPriority; }
        }
        public bool ReleasePrioFixFlag
        {
            set { releasePriorityFixFlag = value; }
            get { return releasePriorityFixFlag; }
        }
        public Nw4rXmlPanMode PanMode
        {
            set { panMode = value; }
            get { return panMode; }
        }
        public Nw4rXmlPanCurve PanCurve
        {
            set { panCurve = value; }
            get { return panCurve; }
        }
#if ENABLE_COMMENT
    public string Comment
    {
        set { comment = value; }
        get { return comment; }
    }
#endif
        internal static Nw4rXmlSoundArchiveWaveSound Parse(XmlDocument doc, XmlElement waveSoundElem)
        {
            XmlElement elem; // テンポラリ用

            if (!waveSoundElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = waveSoundElem.GetAttribute("name");
            Nw4rXmlSoundArchiveWaveSound waveSound = new Nw4rXmlSoundArchiveWaveSound(label);

            elem = (XmlElement)waveSoundElem.SelectSingleNode("file");
            if (elem != null)
            {
                waveSound.FileLabel = elem.InnerText;
            }
            elem = (XmlElement)waveSoundElem.SelectSingleNode("sound_name");
            if (elem != null)
            {
                waveSound.SoundName = elem.InnerText;
                waveSound.SoundIndex = Int32.Parse(elem.GetAttribute("index"));
            }

            elem = (XmlElement)waveSoundElem.SelectSingleNode("channel_prio");
            if (elem != null)
            {
                waveSound.ChannelPrio = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)waveSoundElem.SelectSingleNode("channel_release_prio");
            if (elem != null)
            {
                if (elem.HasAttribute("use_channel_prio"))
                {
                    waveSound.ReleasePrioFixFlag = Boolean.Parse(elem.GetAttribute("use_channel_prio"));
                }
            }

            elem = (XmlElement)waveSoundElem.SelectSingleNode("pan_mode");
            if (elem != null)
            {
                waveSound.PanMode = Nw4rXmlPanMode.Parse(elem.InnerText);
            }

            elem = (XmlElement)waveSoundElem.SelectSingleNode("pan_curve");
            if (elem != null)
            {
                waveSound.PanCurve = Nw4rXmlPanCurve.Parse(elem.InnerText);
            }

#if ENABLE_COMMENT
        elem = (XmlElement)waveSoundElem.SelectSingleNode( "comment" );
        if ( elem != null )
        {
            waveSound.Comment = elem.InnerText;
        }
#endif

            waveSound.ParseSoundElement(doc, waveSoundElem);

            return waveSound;
        }

        internal override XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement waveSoundElem = doc.CreateElement("wave_sound");
            waveSoundElem.SetAttribute("name", this.Label);
            if (this.FileLabel != null)
            {
                elem = doc.CreateElement("file");
                elem.InnerText = this.FileLabel;
                waveSoundElem.AppendChild(elem);
            }
            if (this.SoundName != null)
            {
                elem = doc.CreateElement("sound_name");
                elem.InnerText = this.SoundName;
                elem.SetAttribute("index", this.SoundIndex.ToString());
                waveSoundElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_prio");
                elem.InnerText = this.channelPriority.ToString();
                waveSoundElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("channel_release_prio");
                elem.SetAttribute("use_channel_prio", this.releasePriorityFixFlag.ToString());
                waveSoundElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pan_mode");
                elem.InnerText = this.PanMode.ToString();
                waveSoundElem.AppendChild(elem);
            }
            {
                elem = doc.CreateElement("pan_curve");
                elem.InnerText = this.PanCurve.ToString();
                waveSoundElem.AppendChild(elem);
            }
#if ENABLE_COMMENT
        if ( comment != null )
        {
            elem = doc.CreateElement( "comment" );
            elem.InnerText = this.comment;
            waveSoundElem.AppendChild( elem );
        }
#endif
            AppendSoundElement(doc, waveSoundElem);

            return waveSoundElem;
        }
    }
    public class Nw4rXmlSoundArchivePlayer : Nw4rXmlSoundArchiveItem
    {
        int soundMax = 1;
        int heapSize = 0;
#if ENABLE_COMMENT
    string comment;
#endif
        public Nw4rXmlSoundArchivePlayer(string label)
        : base(label)
        {
        }
        public int SoundLimit
        {
            set { soundMax = value; }
            get { return soundMax; }
        }
        public int HeapSize
        {
            set { heapSize = value; }
            get { return heapSize; }
        }
#if ENABLE_COMMENT
    public string Comment
    {
        set { comment = value; }
        get { return comment; }
    }
#endif
        internal static Nw4rXmlSoundArchivePlayer Parse(XmlDocument doc, XmlElement playerElem)
        {
            XmlElement elem; // テンポラリ用

            if (!playerElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = playerElem.GetAttribute("name");
            Nw4rXmlSoundArchivePlayer player = new Nw4rXmlSoundArchivePlayer(label);

            elem = (XmlElement)playerElem.SelectSingleNode("sound_limit");
            if (elem != null)
            {
                player.SoundLimit = Int32.Parse(elem.InnerText);
            }

            elem = (XmlElement)playerElem.SelectSingleNode("heap_size");
            if (elem != null)
            {
                player.HeapSize = Int32.Parse(elem.InnerText);
            }

#if ENABLE_COMMENT
        elem = (XmlElement)playerElem.SelectSingleNode( "comment" );
        if ( elem != null )
        {
            player.Comment = elem.InnerText;
        }
#endif
            return player;
        }

        internal override XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement playerElem = doc.CreateElement("player");
            playerElem.SetAttribute("name", this.Label);

            {
                elem = doc.CreateElement("sound_limit");
                elem.InnerText = this.SoundLimit.ToString();
                playerElem.AppendChild(elem);
            }
            if (this.heapSize > 0)
            {
                elem = doc.CreateElement("heap_size");
                elem.InnerText = this.HeapSize.ToString();
                playerElem.AppendChild(elem);
            }
#if ENABLE_COMMENT
        if ( comment != null )
        {
            elem = doc.CreateElement( "comment" );
            elem.InnerText = this.comment;
            playerElem.AppendChild( elem );
        }
#endif
            return playerElem;
        }
    }

    public class Nw4rXmlSoundArchiveGroup : Nw4rXmlSoundArchiveItem, IEnumerable
    {
        List<Nw4rXmlSoundArchiveGroupItem> items = new List<Nw4rXmlSoundArchiveGroupItem>();
#if ENABLE_COMMENT
    string comment;
#endif
        public Nw4rXmlSoundArchiveGroup(string label)
        : base(label)
        {
        }
        public int Count
        {
            get { return items.Count; }
        }
        public void Add(Nw4rXmlSoundArchiveGroupItem item)
        {
            items.Add(item);
        }
        public void Remove(Nw4rXmlSoundArchiveGroupItem item)
        {
            items.Remove(item);
        }
        public void RemoveDuplicateItem()
        {
            List<Nw4rXmlSoundArchiveGroupItem> newItems = new List<Nw4rXmlSoundArchiveGroupItem>();

            foreach (Nw4rXmlSoundArchiveGroupItem item in items)
            {
                int index = newItems.FindIndex(delegate (Nw4rXmlSoundArchiveGroupItem i)
                {
                    return i.FileLabel == item.FileLabel;
                });
                if (index < 0)
                {
                    newItems.Add(item);
                }
            }
            items = newItems;
        }
        public void Clear()
        {
            items.Clear();
        }
        IEnumerator IEnumerable.GetEnumerator()
        {
            return items.GetEnumerator();
        }

#if ENABLE_COMMENT
    public string Comment
    {
        set { comment = value; }
        get { return comment; }
    }
#endif

        internal static Nw4rXmlSoundArchiveGroup Parse(XmlDocument doc, XmlElement groupElem)
        {
            if (!groupElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = groupElem.GetAttribute("name");
            Nw4rXmlSoundArchiveGroup group = new Nw4rXmlSoundArchiveGroup(label);

            XmlElement groupItemArrayElem = (XmlElement)groupElem.SelectSingleNode("group_item_array");
            if (groupItemArrayElem != null)
            {
                foreach (XmlElement groupItemElem in groupItemArrayElem.SelectNodes("group_item"))
                {
                    if (groupItemElem.HasAttribute("name"))
                    {
                        label = groupItemElem.GetAttribute("name");
                        Nw4rXmlSoundArchiveGroupItem groupItem = new Nw4rXmlSoundArchiveGroupItem(String.Empty);
                        groupItem.FileLabel = label;
                        group.Add(groupItem);
                    }
                }
            }

#if ENABLE_COMMENT
        XmlElement elem = (XmlElement)groupElem.SelectSingleNode( "comment" );
        if ( elem != null )
        {
            group.Comment = elem.InnerText;
        }
#endif
            return group;
        }

        internal override XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            XmlElement groupElem = doc.CreateElement("group");
            groupElem.SetAttribute("name", this.Label);

            if (items.Count > 0)
            {
                XmlElement groupItemArrayElem = doc.CreateElement("group_item_array");
                foreach (Nw4rXmlSoundArchiveGroupItem groupItem in items)
                {
                    elem = doc.CreateElement("group_item");
                    elem.SetAttribute("name", groupItem.FileLabel);
                    groupItemArrayElem.AppendChild(elem);
                }
                groupElem.AppendChild(groupItemArrayElem);
            }

#if ENABLE_COMMENT
        if ( comment != null )
        {
            elem = doc.CreateElement( "comment" );
            elem.InnerText = this.comment;
            groupElem.AppendChild( elem );
        }
#endif
            return groupElem;
        }
    }

    public class Nw4rXmlSoundArchiveGroupItem
    {
        string label;
        string fileLabel;

        public Nw4rXmlSoundArchiveGroupItem(string label)
        {
            this.label = label;
        }
        public string Label
        {
            set { label = value; }
            get { return label; }
        }
        public string FileLabel
        {
            set { fileLabel = value; }
            get { return fileLabel; }
        }
    }

    public class Nw4rXmlSoundArchive : Nw4rXmlFile
    {
        Nw4rXmlSoundArchiveFileList fileList = new Nw4rXmlSoundArchiveFileList();
        Nw4rXmlSoundArchiveBankList bankList = new Nw4rXmlSoundArchiveBankList();
        Nw4rXmlSoundArchiveSoundList soundList = new Nw4rXmlSoundArchiveSoundList();
        Nw4rXmlSoundArchivePlayerList playerList = new Nw4rXmlSoundArchivePlayerList();
        Nw4rXmlSoundArchiveGroupList groupList = new Nw4rXmlSoundArchiveGroupList();
        Nw4rSoundArchivePlayerSettings soundArchivePlayerSettings = new Nw4rSoundArchivePlayerSettings();

        public Nw4rXmlSoundArchiveFileList FileList
        {
            get { return fileList; }
        }
        public Nw4rXmlSoundArchiveSoundList SoundList
        {
            get { return soundList; }
        }
        public Nw4rXmlSoundArchiveBankList BankList
        {
            get { return bankList; }
        }
        public Nw4rXmlSoundArchivePlayerList PlayerList
        {
            get { return playerList; }
        }
        public Nw4rXmlSoundArchiveGroupList GroupList
        {
            get { return groupList; }
        }
        public Nw4rSoundArchivePlayerSettings SoundArchivePlayerSettings
        {
            get { return soundArchivePlayerSettings; }
            set { soundArchivePlayerSettings = value; }
        }

        public void Clear()
        {
            fileList.Clear();
            bankList.Clear();
            soundList.Clear();
            playerList.Clear();
            groupList.Clear();
        }

        protected override string FileTitle { get { return "NintendoWare Sound Archive"; } }

        protected override void LoadBody(XmlDocument doc, XmlElement bodyElem)
        {
            Clear();

            XmlElement soundArchiveElem = (XmlElement)bodyElem.SelectSingleNode("sound_archive");
            if (soundArchiveElem != null)
            {
                XmlElement soundArchivePlayerSettingsElem = (XmlElement)soundArchiveElem.SelectSingleNode("sound_archive_player_settings");
                if (soundArchivePlayerSettingsElem != null)
                {
                    soundArchivePlayerSettings = Nw4rSoundArchivePlayerSettings.Parse(doc, soundArchivePlayerSettingsElem);
                }

                XmlElement fileArrayElem = (XmlElement)soundArchiveElem.SelectSingleNode("file_array");
                if (fileArrayElem != null)
                {
                    foreach (XmlElement fileElem in fileArrayElem.SelectNodes("file"))
                    {
                        Nw4rXmlSoundArchiveFile file = Nw4rXmlSoundArchiveFile.Parse(doc, fileElem);
                        if (file != null)
                        {
                            fileList.Add(file);
                        }
                    }
                }

                XmlElement soundArrayElem = (XmlElement)soundArchiveElem.SelectSingleNode("sound_array");
                if (soundArrayElem != null)
                {
                    foreach (XmlElement seqSoundElem in soundArrayElem.SelectNodes("seq_sound"))
                    {
                        Nw4rXmlSoundArchiveSeq seq = Nw4rXmlSoundArchiveSeq.Parse(doc, seqSoundElem);
                        if (seq != null)
                        {
                            soundList.Add(seq);
                        }
                    }
                    foreach (XmlElement strmSoundElem in soundArrayElem.SelectNodes("strm_sound"))
                    {
                        Nw4rXmlSoundArchiveStrm strm = Nw4rXmlSoundArchiveStrm.Parse(doc, strmSoundElem);
                        if (strm != null)
                        {
                            soundList.Add(strm);
                        }
                    }
                    foreach (XmlElement waveSoundElem in soundArrayElem.SelectNodes("wave_sound"))
                    {
                        Nw4rXmlSoundArchiveWaveSound waveSound = Nw4rXmlSoundArchiveWaveSound.Parse(doc, waveSoundElem);
                        if (waveSound != null)
                        {
                            soundList.Add(waveSound);
                        }
                    }
                }

                XmlElement bankArrayElem = (XmlElement)soundArchiveElem.SelectSingleNode("bank_array");
                if (bankArrayElem != null)
                {
                    foreach (XmlElement bankElem in bankArrayElem.SelectNodes("bank"))
                    {
                        Nw4rXmlSoundArchiveBank bank = Nw4rXmlSoundArchiveBank.Parse(doc, bankElem);
                        if (bank != null)
                        {
                            bankList.Add(bank);
                        }
                    }
                }

                XmlElement playerArrayElem = (XmlElement)soundArchiveElem.SelectSingleNode("player_array");
                if (playerArrayElem != null)
                {
                    foreach (XmlElement playerElem in playerArrayElem.SelectNodes("player"))
                    {
                        Nw4rXmlSoundArchivePlayer player = Nw4rXmlSoundArchivePlayer.Parse(doc, playerElem);
                        if (player != null)
                        {
                            playerList.Add(player);
                        }
                    }
                }

                XmlElement groupArrayElem = (XmlElement)soundArchiveElem.SelectSingleNode("group_array");
                if (groupArrayElem != null)
                {
                    foreach (XmlElement groupElem in groupArrayElem.SelectNodes("group"))
                    {
                        Nw4rXmlSoundArchiveGroup group = Nw4rXmlSoundArchiveGroup.Parse(doc, groupElem);
                        if (group != null)
                        {
                            groupList.Add(group);
                        }
                    }
                }
            }
        }

        protected override void SaveBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement soundArchiveElem = doc.CreateElement("sound_archive");
            bodyElem.AppendChild(soundArchiveElem);

            XmlElement soundArchivePlayerSettingsElem = soundArchivePlayerSettings.ToXmlElement(doc);
            soundArchiveElem.AppendChild(soundArchivePlayerSettingsElem);

            if (fileList.Count > 0)
            {
                XmlElement fileArrayElem = doc.CreateElement("file_array");
                foreach (Nw4rXmlSoundArchiveFile file in fileList)
                {
                    XmlElement fileElem = file.ToXmlElement(doc);
                    fileArrayElem.AppendChild(fileElem);
                }
                soundArchiveElem.AppendChild(fileArrayElem);
            }
            if (soundList.Count > 0)
            {
                XmlElement soundArrayElem = doc.CreateElement("sound_array");
                foreach (Nw4rXmlSoundArchiveSound sound in soundList)
                {
                    XmlElement soundElem = sound.ToXmlElement(doc);
                    soundArrayElem.AppendChild(soundElem);
                }
                soundArchiveElem.AppendChild(soundArrayElem);
            }
            if (bankList.Count > 0)
            {
                XmlElement bankArrayElem = doc.CreateElement("bank_array");
                foreach (Nw4rXmlSoundArchiveBank bank in bankList)
                {
                    XmlElement bankElem = bank.ToXmlElement(doc);
                    bankArrayElem.AppendChild(bankElem);
                }
                soundArchiveElem.AppendChild(bankArrayElem);
            }
            if (playerList.Count > 0)
            {
                XmlElement playerArrayElem = doc.CreateElement("player_array");
                foreach (Nw4rXmlSoundArchivePlayer player in playerList)
                {
                    XmlElement playerElem = player.ToXmlElement(doc);
                    playerArrayElem.AppendChild(playerElem);
                }
                soundArchiveElem.AppendChild(playerArrayElem);
            }
            if (groupList.Count > 0)
            {
                XmlElement groupArrayElem = doc.CreateElement("group_array");
                foreach (Nw4rXmlSoundArchiveGroup group in groupList)
                {
                    XmlElement groupElem = group.ToXmlElement(doc);
                    groupArrayElem.AppendChild(groupElem);
                }
                soundArchiveElem.AppendChild(groupArrayElem);
            }
        }

    }

}
