﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Xml;
using System.Text;
using System.Collections;
using System.Collections.Generic;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rXmlFileVersion
    {
        uint[] version = new uint[3];

        public uint MajorVersion
        {
            set { version[0] = value; }
            get { return version[0]; }
        }
        public uint MinorVersion
        {
            set { version[1] = value; }
            get { return version[1]; }
        }
        public uint BugFixVersion
        {
            set { version[2] = value; }
            get { return version[2]; }
        }
        public int CompareTo(object value)
        {
            Nw4rXmlFileVersion rhs = (Nw4rXmlFileVersion)value;
            for (int i = 0; i < version.Length; i++)
            {
                if (version[i] < rhs.version[i]) return -1;
                if (version[i] > rhs.version[i]) return +1;
            }
            return 0;
        }
        public static Nw4rXmlFileVersion Parse(string versionString)
        {
            Nw4rXmlFileVersion version = new Nw4rXmlFileVersion();

            string[] versionArray = versionString.Split('.');
            for (int i = 0; i < version.version.Length && i < versionArray.Length; i++)
            {
                version.version[i] = UInt32.Parse(versionArray[i]);
            }

            return version;
        }
        public override string ToString()
        {
            return
                version[0].ToString() + "." +
                version[1].ToString() + "." +
                version[2].ToString()
                ;
        }
    }

    public class Nw4rXmlFile
    {
        string generatorName = null;
        string generatorVersion = null;
        string filePath;
        string createUser;
        string createHost;
        string createDate;
        Nw4rXmlFileVersion fileVersion = new Nw4rXmlFileVersion();
        string platform = string.Empty;

        public string GeneratorName
        {
            set { generatorName = value; }
            get { return generatorName; }
        }
        public string GeneratorVersion
        {
            set { generatorVersion = value; }
            get { return generatorVersion; }
        }
        public string FilePath
        {
            get { return filePath; }
        }
        public Nw4rXmlFileVersion FileVersion
        {
            get { return fileVersion; }
        }
        public string Platform
        {
            get { return platform; }
            set { platform = value; }
        }
        public string CreateUser
        {
            get { return createUser; }
        }
        public string CreateHost
        {
            get { return createHost; }
        }
        public string CreateDate
        {
            get { return createDate; }
        }

        protected virtual string FileTitle { get { return String.Empty; } }
        protected virtual void LoadBody(XmlDocument doc, XmlElement bodyElem) { }
        protected virtual void SaveBody(XmlDocument doc, XmlElement bodyElem) { }

        public void Load(string filePath)
        {
            this.filePath = filePath;

            XmlDocument doc = new XmlDocument();
            doc.Load(filePath);

            XmlElement root = doc.DocumentElement;
            if (root.Name != "nintendoware_snd")
            {
                throw new Nw4rFileFormatException("Unexpcted file format \"" + filePath + "\"");
            }
            if (root.HasAttribute("version"))
            {
                string versionString = root.GetAttribute("version");
                fileVersion = Nw4rXmlFileVersion.Parse(versionString);
            }
            if (root.HasAttribute("platform"))
            {
                platform = root.GetAttribute("platform");
            }

            XmlElement headElem = (XmlElement)root.SelectSingleNode("head");
            if (headElem != null)
            {
                XmlElement generatorElem = (XmlElement)headElem.SelectSingleNode("generator");
                if (generatorElem != null)
                {
                    if (generatorElem.HasAttribute("name"))
                    {
                        generatorName = generatorElem.GetAttribute("name");
                    }
                    if (generatorElem.HasAttribute("version"))
                    {
                        generatorVersion = generatorElem.GetAttribute("version");
                    }
                }

                XmlElement createElem = (XmlElement)headElem.SelectSingleNode("create");
                if (createElem != null)
                {
                    if (createElem.HasAttribute("user"))
                    {
                        createUser = createElem.GetAttribute("user");
                    }
                    if (createElem.HasAttribute("host"))
                    {
                        createHost = createElem.GetAttribute("host");
                    }
                    if (createElem.HasAttribute("date"))
                    {
                        createDate = createElem.GetAttribute("date");
                    }
                }
            }

            XmlElement bodyElem = (XmlElement)root.SelectSingleNode("body");
            if (bodyElem != null)
            {
                LoadBody(doc, bodyElem);
            }
        }

        public void Save(string filePath)
        {
            this.filePath = filePath;

            this.fileVersion.MajorVersion = 1;
            this.fileVersion.MinorVersion = 1;
            this.fileVersion.BugFixVersion = 0;

            XmlDocument doc = new XmlDocument();

            XmlDeclaration xmlDecl = doc.CreateXmlDeclaration("1.0", null, null);
            doc.AppendChild(xmlDecl);

            XmlElement rootElem = doc.CreateElement("nintendoware_snd");
            rootElem.SetAttribute("version", fileVersion.ToString());
            rootElem.SetAttribute("platform", platform);
            doc.AppendChild(rootElem);

            XmlElement elem; // テンポラリ用

            {
                XmlElement headElem = doc.CreateElement("head");
                rootElem.AppendChild(headElem);

                elem = doc.CreateElement("create");
                elem.SetAttribute("user", Environment.UserName);
                elem.SetAttribute("host", Environment.MachineName);
                elem.SetAttribute("date", DateTime.Now.ToString("yyyy-MM-ddTHH:mm:ss"));
                headElem.AppendChild(elem);

                elem = doc.CreateElement("title");
                elem.InnerText = FileTitle;
                headElem.AppendChild(elem);

                if (generatorName != null)
                {
                    elem = doc.CreateElement("generator");
                    elem.SetAttribute("name", generatorName);
                    if (generatorVersion != null)
                    {
                        elem.SetAttribute("version", generatorVersion);
                    }
                    headElem.AppendChild(elem);
                }
            }

            {
                XmlElement bodyElem = doc.CreateElement("body");

                SaveBody(doc, bodyElem);

                rootElem.AppendChild(bodyElem);
            }

            // 保存実行
            using (XmlTextWriter xmlWriter = new XmlTextWriter(filePath, Encoding.UTF8))
            {
                xmlWriter.Formatting = Formatting.Indented;
                doc.Save(xmlWriter);
            }
        }
    }

    public class Nw4rAdsrEnvelope
    {
        int attack = 127;
        int hold = 0;
        int decay = 127;
        int sustain = 127;
        int release = 127;

        public int Attack
        {
            get { return attack; }
            set { attack = value; }
        }
        public int Hold
        {
            get { return hold; }
            set { hold = value; }
        }
        public int Decay
        {
            get { return decay; }
            set { decay = value; }
        }
        public int Sustain
        {
            get { return sustain; }
            set { sustain = value; }
        }
        public int Release
        {
            get { return release; }
            set { release = value; }
        }
        internal static Nw4rAdsrEnvelope Parse(XmlDocument doc, XmlElement elem)
        {
            Nw4rAdsrEnvelope envelope = new Nw4rAdsrEnvelope();

            XmlElement attackElem = (XmlElement)elem.SelectSingleNode("attack");
            if (attackElem != null)
            {
                envelope.Attack = Int32.Parse(attackElem.InnerText);
            }

            XmlElement holdElem = (XmlElement)elem.SelectSingleNode("hold");
            if (holdElem != null)
            {
                envelope.Hold = Int32.Parse(holdElem.InnerText);
            }

            XmlElement decayElem = (XmlElement)elem.SelectSingleNode("decay");
            if (decayElem != null)
            {
                envelope.Decay = Int32.Parse(decayElem.InnerText);
            }

            XmlElement sustainElem = (XmlElement)elem.SelectSingleNode("sustain");
            if (sustainElem != null)
            {
                envelope.Sustain = Int32.Parse(sustainElem.InnerText);
            }

            XmlElement releaseElem = (XmlElement)elem.SelectSingleNode("release");
            if (releaseElem != null)
            {
                envelope.Release = Int32.Parse(releaseElem.InnerText);
            }

            return envelope;
        }
        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement adsrEnvElem = doc.CreateElement("adsr_envelope");

            XmlElement attackElem = doc.CreateElement("attack");
            attackElem.InnerText = attack.ToString();
            adsrEnvElem.AppendChild(attackElem);

            XmlElement holdElem = doc.CreateElement("hold");
            holdElem.InnerText = hold.ToString();
            adsrEnvElem.AppendChild(holdElem);

            XmlElement decayElem = doc.CreateElement("decay");
            decayElem.InnerText = decay.ToString();
            adsrEnvElem.AppendChild(decayElem);

            XmlElement sustainElem = doc.CreateElement("sustain");
            sustainElem.InnerText = sustain.ToString();
            adsrEnvElem.AppendChild(sustainElem);

            XmlElement releaseElem = doc.CreateElement("release");
            releaseElem.InnerText = release.ToString();
            adsrEnvElem.AppendChild(releaseElem);

            return adsrEnvElem;
        }
        internal Nw4rAdsrEnvelope Clone()
        {
            Nw4rAdsrEnvelope env = new Nw4rAdsrEnvelope();
            env.Attack = this.Attack;
            env.Hold = this.Hold;
            env.Decay = this.Decay;
            env.Sustain = this.Sustain;
            env.Sustain = this.Sustain;
            return env;
        }
    }

    public struct Nw4rXmlPanMode
    {
        public const int DUAL = 0;    // ステレオを２本のモノラルとしてそれぞれに定位処理を行います。
        public const int BALANCE = 1; // 左右チャンネルの音量バランスを処理します。

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( DUAL, "dual" ),
            new KeyValuePair<int, string>( BALANCE, "balance" ),
        };

        public static implicit operator int(Nw4rXmlPanMode s) { return s.val_; }
        public static implicit operator Nw4rXmlPanMode(int s) { return new Nw4rXmlPanMode(s); }
        public static Nw4rXmlPanMode Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlPanMode(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rXmlPanCurve
    {
        public const int SQRT = 0;               // 平方根のカーブ。中央で-3dB、両端に振り切ると0dBとなります。
        public const int SQRT_0DB = 1;           // 平方根のカーブ。中央で0dB、両端に振り切ると+3dBとなります。
        public const int SQRT_0DB_CLAMP = 2;     // 平方根のカーブ。中央で0dB、両端に振り切ると0dBとなります。
        public const int SINCOS = 3;             // 三角関数のカーブ。中央で-3dB、両端に振り切ると0dBとなります。
        public const int SINCOS_0DB = 4;         // 三角関数のカーブ。中央で0dB、両端に振り切ると+3dBとなります。
        public const int SINCOS_0DB_CLAMP = 5;   // 三角関数のカーブ。中央で0dB、両端に振り切ると0dBとなります。
        public const int LINEAR = 6;             // 線形のカーブ。中央で-6dB、両端に振り切ると0dBとなります。
        public const int LINEAR_0DB = 7;         // 線形のカーブ。中央で0dB、両端に振り切ると+6dBとなります。
        public const int LINEAR_0DB_CLAMP = 8;   // 線形のカーブ。中央で0dB、両端に振り切ると0dBとなります。

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( SQRT, "sqrt" ),
            new KeyValuePair<int, string>( SQRT_0DB, "sqrt_0db" ),
            new KeyValuePair<int, string>( SQRT_0DB_CLAMP, "sqrt_0db_clamp" ),
            new KeyValuePair<int, string>( SINCOS, "sincos" ),
            new KeyValuePair<int, string>( SINCOS_0DB, "sincos_0db" ),
            new KeyValuePair<int, string>( SINCOS_0DB_CLAMP, "sincos_0db_clamp" ),
            new KeyValuePair<int, string>( LINEAR, "linear" ),
            new KeyValuePair<int, string>( LINEAR_0DB, "linear_0db" ),
            new KeyValuePair<int, string>( LINEAR_0DB_CLAMP, "linear_0db_clamp" ),
        };

        public static implicit operator int(Nw4rXmlPanCurve s) { return s.val_; }
        public static implicit operator Nw4rXmlPanCurve(int s) { return new Nw4rXmlPanCurve(s); }
        public static Nw4rXmlPanCurve Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlPanCurve(int e) { val_ = e; }
        private int val_;
    }

}
