﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections;
using System.Collections.Generic;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rSoundMakerProjectConverter
    {
        Nw4rSoundMakerProject project;
        string filePath;
        SortedList<int, Nw4rXmlSoundArchiveFile> fileList;
        int fileLabelNumber;
        Dictionary<string, bool> labelDict;

        class GroupItemInfo
        {
            public string Label;
        }
        class GroupInfo
        {
            public string Label;
#if ENABLE_COMMENT
        public string Comment;
#endif
            public List<GroupItemInfo> GroupItemList = new List<GroupItemInfo>();
        }

        public Nw4rSoundMakerProjectConverter(string filePath)
        {
            this.project = new Nw4rSoundMakerProject();
            this.project.Load(filePath);

            this.filePath = Path.Combine(Directory.GetCurrentDirectory(), filePath);
        }

        private Nw4rXmlSoundArchiveFile FindFileByExtFilePath(string extFilePath)
        {
            foreach (Nw4rXmlSoundArchiveFile file in fileList.Values)
            {
                if (extFilePath == file.ExtFilePath) return file;
            }
            return null;
        }

        private Nw4rXmlSoundArchiveFile AddFile(Nw4rXmlSoundArchiveFile file)
        {
            int hashCode = file.GetHashCode();
            if (fileList.ContainsKey(hashCode))
            {
                file = fileList[hashCode];
            }
            else
            {
                if (!String.IsNullOrEmpty(file.ExtFilePath))
                {
                    string originalExtFilePath = file.ExtFilePath;
                    int index = 0;
                    while (FindFileByExtFilePath(file.ExtFilePath) != null)
                    {
                        file.ExtFilePath = Path.ChangeExtension(
                            originalExtFilePath, "." + index + Path.GetExtension(originalExtFilePath)
                        );
                        index++;
                    }
                }

                file.Label = CreateFileLabel();
                fileList.Add(hashCode, file);
                fileLabelNumber++;
            }

            return file;
        }

        private void AddLabel(string label)
        {
            if (labelDict.ContainsKey(label))
            {
                throw new Nw4rFileFormatException("Multiple defined label \"" + label + "\"");
            }
            labelDict.Add(label, true);
        }

        private Nw4rXmlSoundArchiveSeq Convert(Nw4rSoundSetSeq src, string soundArchiveFilePath)
        {
            AddLabel(src.Label);

            Nw4rXmlSoundArchiveSeq dest = new Nw4rXmlSoundArchiveSeq(src.Label);
            dest.Bank = src.Bank;
            dest.Volume = src.Volume;
            dest.Player = src.Player;
            dest.ActorPlayer = src.ActorPlayer;
            dest.PlayerPrio = src.PlayerPrio;
            dest.ChannelPrio = src.ChannelPrio;
            dest.ReleasePrioFixFlag = src.ReleasePrioFixFlag;
            dest.RemoteFilter = src.RemoteFilter;
            dest.StartPosition = src.StartPosition;
            dest.Sound3DParam = src.Sound3DParam;
            dest.UserParam = src.UserParam;
#if ENABLE_COMMENT
        dest.Comment = src.Comment;
#endif
            Nw4rXmlSoundArchiveFile file = new Nw4rXmlSoundArchiveFile(String.Empty);
            file.FileType = Nw4rXmlSoundArchiveFileType.Seq;
            file.FilePath = NPath.MakeRelative(src.FullPath, Path.GetDirectoryName(soundArchiveFilePath));
            switch (src.FileType)
            {
                case Nw4rSoundSetSeqFileType.Text: file.InputType = Nw4rXmlSoundArchiveInputFileType.Text; break;
                case Nw4rSoundSetSeqFileType.Binary: file.InputType = Nw4rXmlSoundArchiveInputFileType.Binary; break;
                case Nw4rSoundSetSeqFileType.Smf:
                    file.InputType = Nw4rXmlSoundArchiveInputFileType.Smf;
                    file.SmfConvertTimebase = project.SmfConvertTimebase;
                    break;
            }
            file = AddFile(file);
            dest.FileLabel = file.Label;

            return dest;
        }

        private Nw4rXmlSoundArchiveBank Convert(Nw4rSoundSetBank src, string soundArchiveFilePath)
        {
            AddLabel(src.Label);

            Nw4rXmlSoundArchiveBank dest = new Nw4rXmlSoundArchiveBank(src.Label);

            Nw4rXmlSoundArchiveFile file = new Nw4rXmlSoundArchiveFile(String.Empty);
            file.FileType = Nw4rXmlSoundArchiveFileType.Bank;
            file.FilePath = NPath.MakeRelative(src.FullPath, Path.GetDirectoryName(soundArchiveFilePath));
            switch (src.FileType)
            {
                //       case Nw4rSoundSetBankFileType.Text: file.InputType = Nw4rXmlSoundArchiveInputFileType.Text; break;
                case Nw4rSoundSetBankFileType.Binary: file.InputType = Nw4rXmlSoundArchiveInputFileType.Binary; break;
                case Nw4rSoundSetBankFileType.Xml: file.InputType = Nw4rXmlSoundArchiveInputFileType.Xml; break;
            }
            file = AddFile(file);
            dest.FileLabel = file.Label;
#if ENABLE_COMMENT
        dest.Comment = src.Comment;
#endif
            return dest;
        }

        private Nw4rXmlSoundArchiveStrm Convert(Nw4rSoundSetStrm src, string soundArchiveFilePath, Nw4rSoundSet soundSet)
        {
            AddLabel(src.Label);

            Nw4rXmlSoundArchiveStrm dest = new Nw4rXmlSoundArchiveStrm(src.Label);
            dest.Volume = src.Volume;
            dest.Player = src.Player;
            dest.ActorPlayer = src.ActorPlayer;
            dest.PlayerPrio = src.PlayerPrio;
            dest.RemoteFilter = src.RemoteFilter;
            dest.PanMode = src.PanMode;
            dest.PanCurve = src.PanCurve;
            dest.Sound3DParam = src.Sound3DParam;
            dest.UserParam = src.UserParam;
#if ENABLE_COMMENT
        dest.Comment = src.Comment;
#endif
            if (src.TrackList.Count == 0) return dest;


            Nw4rXmlSoundArchiveFile file = new Nw4rXmlSoundArchiveFile(String.Empty);

            if (src.TrackList.Count == 1)
            {
                // シングルトラック
                Nw4rSoundSetStrm.Track srcTrack0 = src.TrackList[0];

                file.FileType = Nw4rXmlSoundArchiveFileType.Stream;
                file.ExtFilePath = "";
                if (project.ExtFileDir != String.Empty)
                {
                    file.ExtFilePath += project.ExtFileDir + "/";
                }
                file.ExtFilePath += Path.ChangeExtension(Path.GetFileName(srcTrack0.FilePath), "brstm");

                file.FilePath = NPath.MakeRelative(srcTrack0.FullPath, Path.GetDirectoryName(soundArchiveFilePath));
                switch (srcTrack0.FileFormat)
                {
                    case Nw4rSoundSetStrmFileFormat.Adpcm:
                        file.EncodeType = Nw4rXmlSoundArchiveEncodeType.Adpcm;
                        file.InputType = Nw4rXmlSoundArchiveInputFileType.Wave;
                        break;
                    case Nw4rSoundSetStrmFileFormat.Pcm16:
                        file.EncodeType = Nw4rXmlSoundArchiveEncodeType.Pcm16;
                        file.InputType = Nw4rXmlSoundArchiveInputFileType.Wave;
                        break;
                    case Nw4rSoundSetStrmFileFormat.Pcm8:
                        file.EncodeType = Nw4rXmlSoundArchiveEncodeType.Pcm8;
                        file.InputType = Nw4rXmlSoundArchiveInputFileType.Wave;
                        break;
                        //case Nw4rSoundSetStrmFileFormat.Strm:
                        //    file.InputType = Nw4rXmlSoundArchiveInputFileType.Binary;
                        //    break;
                }
            }
            else
            {
                // マルチトラック
                Nw4rSoundSetStrm.Track srcTrack0 = src.TrackList[0];

                // rstmファイルパス生成
                String baseFileName = src.Label + ".multi";
                String rstmFullPath =
                    Path.GetDirectoryName(srcTrack0.FullPath) +
                    Path.DirectorySeparatorChar +
                    baseFileName + ".rstm";
                String rstmFilePath = NPath.MakeRelative(rstmFullPath, Path.GetDirectoryName(soundArchiveFilePath));

                // 出力データ構築
                Nw4rXmlStrmSoundStrm strmSoundData = new Nw4rXmlStrmSoundStrm(src.Label);
                foreach (Nw4rSoundSetStrm.Track srcTrack in src.TrackList)
                {
                    Nw4rXmlStrmSoundStrm.Track track = new Nw4rXmlStrmSoundStrm.Track();
                    track.Index = srcTrack.Index;
                    track.FileFormat = srcTrack.FileFormat;
                    track.FullPath = srcTrack.FullPath;
                    track.Pan = srcTrack.Pan;
                    track.Volume = srcTrack.Volume;
                    strmSoundData.TrackList.Add(track);
                }

                strmSoundData.TrackList.Sort(Nw4rXmlStrmSoundStrm.CompareTrackByIndex);

                // すでにファイルがあったら内容を比較する
                bool saveRstmFileFlag = true;
                if (File.Exists(rstmFullPath))
                {
                    Nw4rXmlStrmSound inStrmSound = new Nw4rXmlStrmSound();
                    inStrmSound.Load(rstmFullPath);
                    if (inStrmSound.StrmSoundData != null)
                    {
                        inStrmSound.StrmSoundData.TrackList.Sort(Nw4rXmlStrmSoundStrm.CompareTrackByIndex);
                        if (inStrmSound.StrmSoundData.Equals(strmSoundData))
                        {
                            saveRstmFileFlag = false;
                        }
                    }
                }

                // rstmファイル出力
                if (saveRstmFileFlag)
                {
                    Nw4rXmlStrmSound strmSound = new Nw4rXmlStrmSound();
                    strmSound.GeneratorName = soundSet.GeneratorName;
                    strmSound.GeneratorVersion = soundSet.GeneratorVersion;
                    strmSound.StrmSoundData = strmSoundData;
                    strmSound.Save(rstmFullPath);
                }

                // XmlSoundArchiveのファイルリストに書き出す情報
                file.FileType = Nw4rXmlSoundArchiveFileType.Stream;
                file.FilePath = rstmFilePath;
                file.ExtFilePath = "";
                if (project.ExtFileDir != String.Empty)
                {
                    file.ExtFilePath += project.ExtFileDir + "/";
                }
                file.ExtFilePath += Path.ChangeExtension(baseFileName, "brstm");
                file.InputType = Nw4rXmlSoundArchiveInputFileType.Xml;
            }


            file = AddFile(file);
            dest.FileLabel = file.Label;

            return dest;
        }

        private Nw4rXmlSoundArchiveWaveSound Convert(Nw4rSoundSetWaveSound src, Nw4rSoundSetWaveSoundSet waveSoundSet, Nw4rXmlSoundArchiveFile file)
        {
            AddLabel(src.Label);

            Nw4rXmlSoundArchiveWaveSound dest = new Nw4rXmlSoundArchiveWaveSound(src.Label);
            dest.SoundName = src.Label;
            dest.Volume = src.Volume;
            dest.Player = src.Player;
            dest.ActorPlayer = src.ActorPlayer;
            dest.PlayerPrio = src.PlayerPrio;
            dest.ChannelPrio = src.ChannelPrio;
            dest.ReleasePrioFixFlag = src.ReleasePrioFixFlag;
            dest.RemoteFilter = src.RemoteFilter;
            dest.PanMode = src.PanMode;
            dest.PanCurve = src.PanCurve;
            dest.Sound3DParam = src.Sound3DParam;
            dest.UserParam = src.UserParam;
#if ENABLE_COMMENT
        dest.Comment = src.Comment;
#endif
            dest.FileLabel = file.Label;

            return dest;
        }

        private Nw4rXmlSoundArchivePlayer Convert(Nw4rSoundSetPlayer src)
        {
            AddLabel(src.Label);

            Nw4rXmlSoundArchivePlayer dest = new Nw4rXmlSoundArchivePlayer(src.Label);
            dest.SoundLimit = src.SoundLimit;
            dest.HeapSize = src.HeapSize;

            return dest;
        }

        private GroupInfo Convert(Nw4rSoundSetGroup src)
        {
            AddLabel(src.Label);

            GroupInfo dest = new GroupInfo();
            dest.Label = src.Label;
#if ENABLE_COMMENT
        dest.Comment = src.Comment;
#endif

            foreach (Nw4rSoundSetGroupItem s in src)
            {
                GroupItemInfo d = new GroupItemInfo();
                d.Label = s.Label;
                dest.GroupItemList.Add(d);
            }

            return dest;
        }

        public Nw4rXmlSoundArchive ConvertToXmlSoundArchive(string soundArchiveFilePath)
        {
            Nw4rXmlSoundArchive soundArchive = new Nw4rXmlSoundArchive();
            labelDict = new Dictionary<string, bool>();

            soundArchive.SoundArchivePlayerSettings = project.SoundArchivePlayerSettings;

            Dictionary<string, List<string>> label2FileTable = new Dictionary<string, List<string>>();

            fileList = new SortedList<int, Nw4rXmlSoundArchiveFile>();
            fileLabelNumber = 0;

            List<GroupInfo> groupList = new List<GroupInfo>();

            foreach (Nw4rSoundMakerProject.SoundSet soundSetInfo in project.SoundSets)
            {
                Nw4rSoundSet soundSet = new Nw4rSoundSet();

                string fullPath = Path.Combine(Path.GetDirectoryName(filePath), soundSetInfo.FilePath);
                soundSet.Load(fullPath);

                foreach (Nw4rSoundSetStrm src in soundSet.StrmList)
                {
                    // ストリームサウンド
                    Nw4rXmlSoundArchiveStrm dest = Convert(src, soundArchiveFilePath, soundSet);
                    soundArchive.SoundList.Add(dest);
                }

                foreach (Nw4rSoundSetWaveSoundSet wsdList in soundSet.WaveSoundSetList)
                {
                    AddLabel(wsdList.Label);

                    // rwsdファイルパス生成
                    String rwsdFullPath =
                        Path.GetDirectoryName(fullPath) +
                        Path.DirectorySeparatorChar +
                        "wsd" +
                        Path.DirectorySeparatorChar +
                        wsdList.Label + ".rwsd";
                    String rwsdFilePath = NPath.MakeRelative(rwsdFullPath, Path.GetDirectoryName(soundArchiveFilePath));

                    // rwsdファイル出力
                    Nw4rXmlWaveSoundSet waveSoundSet = new Nw4rXmlWaveSoundSet();
                    waveSoundSet.GeneratorName = soundSet.GeneratorName;
                    waveSoundSet.GeneratorVersion = soundSet.GeneratorVersion;

                    string dir = Path.GetDirectoryName(rwsdFullPath);
                    if (dir != String.Empty)
                    {
                        Directory.CreateDirectory(dir);
                    }
                    waveSoundSet.WaveSoundSet = wsdList;
                    waveSoundSet.Save(rwsdFullPath);

                    // ウェーブサウンド
                    Nw4rXmlSoundArchiveFile file = new Nw4rXmlSoundArchiveFile(String.Empty);
                    file.FileType = Nw4rXmlSoundArchiveFileType.WaveSoundSet;
                    file.FilePath = rwsdFilePath;
                    file.InputType = Nw4rXmlSoundArchiveInputFileType.Xml;

                    file = AddFile(file);
                    label2FileTable[wsdList.Label] = new List<string>();
                    label2FileTable[wsdList.Label].Add(file.Label);

                    int index = 0;
                    foreach (Nw4rSoundSetWaveSound src in wsdList)
                    {
                        Nw4rXmlSoundArchiveWaveSound dest = Convert(src, wsdList, file);
                        dest.SoundIndex = index++;
                        soundArchive.SoundList.Add(dest);
                    }
                }

                foreach (Nw4rSoundSetSeq src in soundSet.SeqList)
                {
                    Nw4rXmlSoundArchiveSeq dest = Convert(src, soundArchiveFilePath);
                    soundArchive.SoundList.Add(dest);
                    label2FileTable[src.Label] = new List<string>();
                    label2FileTable[src.Label].Add(dest.FileLabel);
                }

                foreach (Nw4rSoundSetSeqSoundSet seqSetList in soundSet.SeqSoundSetList)
                {
                    label2FileTable[seqSetList.Label] = new List<string>();

                    foreach (Nw4rSoundSetSeq src in seqSetList)
                    {
                        Nw4rXmlSoundArchiveSeq dest = Convert(src, soundArchiveFilePath);
                        soundArchive.SoundList.Add(dest);
                        label2FileTable[seqSetList.Label].Add(dest.FileLabel);
                    }
                }

                foreach (Nw4rSoundSetBank src in soundSet.BankList)
                {
                    Nw4rXmlSoundArchiveBank dest = Convert(src, soundArchiveFilePath);
                    soundArchive.BankList.Add(dest);
                    label2FileTable[src.Label] = new List<string>();
                    label2FileTable[src.Label].Add(dest.FileLabel);
                }

                foreach (Nw4rSoundSetPlayer src in soundSet.PlayerList)
                {
                    Nw4rXmlSoundArchivePlayer dest = Convert(src);
                    soundArchive.PlayerList.Add(dest);
                }

                foreach (Nw4rSoundSetGroup src in soundSet.GroupList)
                {
                    GroupInfo dest = Convert(src);
                    groupList.Add(dest);
                }
            }

            // ファイルラベル順に並び替えてサウンドアーカイブに追加する
            SortedList<string, Nw4rXmlSoundArchiveFile> fileListWork = new SortedList<string, Nw4rXmlSoundArchiveFile>();

            foreach (Nw4rXmlSoundArchiveFile file in fileList.Values)
            {
                fileListWork.Add(file.Label, file);
            }

            foreach (Nw4rXmlSoundArchiveFile file in fileListWork.Values)
            {
                soundArchive.FileList.Add(file);
            }

            // グループの登録
            foreach (GroupInfo groupInfo in groupList)
            {
                Nw4rXmlSoundArchiveGroup group = new Nw4rXmlSoundArchiveGroup(groupInfo.Label);
#if ENABLE_COMMENT
            group.Comment = groupInfo.Comment;
#endif
                foreach (GroupItemInfo groupItemInfo in groupInfo.GroupItemList)
                {
                    if (!label2FileTable.ContainsKey(groupItemInfo.Label))
                    {
                        throw new Nw4rFileFormatException("Undefined group item label \"" + groupItemInfo.Label + "\"");
                    }
                    foreach (string fileLabel in label2FileTable[groupItemInfo.Label])
                    {
                        Nw4rXmlSoundArchiveGroupItem groupItem = new Nw4rXmlSoundArchiveGroupItem(groupItemInfo.Label);

                        groupItem.FileLabel = fileLabel;

                        group.Add(groupItem);
                    }
                }

                // 重複グループアイテムの除去
                group.RemoveDuplicateItem();
                soundArchive.GroupList.Add(group);
            }

            return soundArchive;
        }

        private string CreateFileLabel()
        {
            return String.Format("FILE_{0:00000}", fileLabelNumber);
        }
    }

}
