﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections;
using System.Collections.Generic;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    /// <summary>
    /// Nw4rSoundMakerProject の概要の説明です。
    /// </summary>
    public class Nw4rSoundMakerProject : Nw4rXmlFile
    {
        private const string IntermediateOutputDirDefault = "cache";

        SoundSetCollection soundSets = new SoundSetCollection();
        Nw4rSoundArchivePlayerSettings soundArchivePlayerSettings = new Nw4rSoundArchivePlayerSettings();
        FileNameToItemNameConvertSettings fileNameConvertSettings = new FileNameToItemNameConvertSettings();
        Nw4rSoundArchiveConvertSettings soundArchiveConvertSettings = new Nw4rSoundArchiveConvertSettings();
        string intermediateOutputDir = IntermediateOutputDirDefault;
        bool aggregateIntermediateOutput = true;
        bool keepIntermediateRseq = true;
        String outputDir = "output";
        String extFileDir = "sound";
        bool postConvertCommandEnabled = true;
        bool preConvertCommandEnabled = true;
        List<string> postConvertCommandList = new List<string>();
        List<string> preConvertCommandList = new List<string>();
        List<Nw4rSoundList.OutputSetting> soundListOutputSettings = new List<Nw4rSoundList.OutputSetting>();
        Dictionary<string, string> itemNamePrefixList = new Dictionary<string, string>();

        int smfconvTimebase = 48;

        public Nw4rSoundMakerProject()
        {
            itemNamePrefixList.Add("strm_sound", "STRM_");
            itemNamePrefixList.Add("wave_sound_set", "WSDSET_");
            itemNamePrefixList.Add("wave_sound", "WSD_");
            itemNamePrefixList.Add("seq_sound_set", "SEQSET_");
            itemNamePrefixList.Add("seq_sound", "SEQ_");
            itemNamePrefixList.Add("bank", "BANK_");
            itemNamePrefixList.Add("player", "PLAYER_");
            itemNamePrefixList.Add("group", "GROUP_");
            itemNamePrefixList.Add("inst", "INST_");
        }

        public class SoundSet
        {
            string path;
            string name;

            public string FilePath
            {
                get { return path; }
                set { path = value; }
            }
            public string Name
            {
                get { return name; }
                set { name = value; }
            }
        }

        public string IntermediateOutputDir
        {
            get { return intermediateOutputDir; }
            set { intermediateOutputDir = value; }
        }

        public bool AggregateIntermediateOutput
        {
            get { return aggregateIntermediateOutput; }
            set { aggregateIntermediateOutput = value; }
        }

        public bool KeepIntermediateRseq
        {
            get { return keepIntermediateRseq; }
            set { keepIntermediateRseq = value; }
        }

        public int SmfConvertTimebase
        {
            get { return smfconvTimebase; }
            set { smfconvTimebase = value; }
        }

        public string PrefixStrmSound
        {
            get { return itemNamePrefixList["strm_sound"]; }
            set { itemNamePrefixList["strm_sound"] = value; }
        }
        public string PrefixWaveSound
        {
            get { return itemNamePrefixList["wave_sound"]; }
            set { itemNamePrefixList["wave_sound"] = value; }
        }
        public string PrefixWaveSoundSet
        {
            get { return itemNamePrefixList["wave_sound_set"]; }
            set { itemNamePrefixList["wave_sound_set"] = value; }
        }
        public string PrefixSeqSound
        {
            get { return itemNamePrefixList["seq_sound"]; }
            set { itemNamePrefixList["seq_sound"] = value; }
        }
        public string PrefixSeqSoundSet
        {
            get { return itemNamePrefixList["seq_sound_set"]; }
            set { itemNamePrefixList["seq_sound_set"] = value; }
        }
        public string PrefixBank
        {
            get { return itemNamePrefixList["bank"]; }
            set { itemNamePrefixList["bank"] = value; }
        }
        public string PrefixPlayer
        {
            get { return itemNamePrefixList["player"]; }
            set { itemNamePrefixList["player"] = value; }
        }
        public string PrefixGroup
        {
            get { return itemNamePrefixList["group"]; }
            set { itemNamePrefixList["group"] = value; }
        }
        public string PrefixInst
        {
            get { return itemNamePrefixList["inst"]; }
            set { itemNamePrefixList["inst"] = value; }
        }

        public bool PostConvertCommandEnabled
        {
            get { return postConvertCommandEnabled; }
            set { postConvertCommandEnabled = value; }
        }

        public bool PreConvertCommandEnabled
        {
            get { return preConvertCommandEnabled; }
            set { preConvertCommandEnabled = value; }
        }

        public List<string> PostConvertCommandList
        {
            get { return postConvertCommandList; }
            set { postConvertCommandList = value; }
        }
        public List<string> PreConvertCommandList
        {
            get { return preConvertCommandList; }
            set { preConvertCommandList = value; }
        }

        public class SoundSetCollection : IEnumerable
        {
            ArrayList itemList = new ArrayList();

            /******************************************************************************
               interface
             ******************************************************************************/
            IEnumerator IEnumerable.GetEnumerator()
            {
                return itemList.GetEnumerator();
            }

            /******************************************************************************
               public
             ******************************************************************************/
            public int Count
            {
                get { return itemList.Count; }
            }

            public void Clear()
            {
                itemList.Clear();
            }

            public void Add(SoundSet soundSet)
            {
                itemList.Add(soundSet);
            }
            public void Remove(SoundSet soundSet)
            {
                itemList.Remove(soundSet);
            }
        }

        public SoundSetCollection SoundSets
        {
            get { return soundSets; }
        }
        public String OutputDir
        {
            get { return outputDir; }
            set { outputDir = (null == value) ? string.Empty : value; }
        }
        public String ExtFileDir
        {
            get { return extFileDir; }
            set { extFileDir = value; }
        }
        public Nw4rSoundArchiveConvertSettings SoundArchiveConvertSettings
        {
            get { return soundArchiveConvertSettings; }
            set { soundArchiveConvertSettings = value; }
        }
        public Nw4rSoundArchivePlayerSettings SoundArchivePlayerSettings
        {
            get { return soundArchivePlayerSettings; }
            set { soundArchivePlayerSettings = value; }
        }
        public FileNameToItemNameConvertSettings FileNameConvertSettings
        {
            get { return fileNameConvertSettings; }
            set { fileNameConvertSettings = value; }
        }
        public List<Nw4rSoundList.OutputSetting> SoundListOutputSettings
        {
            get { return soundListOutputSettings; }
            set { soundListOutputSettings = value; }
        }

        protected override string FileTitle { get { return "NintendoWare SoundMaker Project"; } }

        private void LoadSoundListOutputSettingListInfo(XmlElement listInfoElem, ref Nw4rSoundList.OutputSetting.TListInfo listInfo)
        {
            XmlElement outputElem = (XmlElement)listInfoElem.SelectSingleNode("output");
            if (outputElem != null)
            {
                listInfo.OutputFlag = (outputElem.GetAttribute("disable") == "true") ? false : true;
            }
            XmlElement orderElem = (XmlElement)listInfoElem.SelectSingleNode("order");
            if (orderElem != null)
            {
                listInfo.Order = Int32.Parse(orderElem.InnerText);
            }
        }
        private void LoadSoundListOutputSettingColumnInfo(XmlElement listInfoElem, ref Nw4rSoundList.OutputSetting.TColumnInfo listInfo)
        {
            XmlElement outputElem = (XmlElement)listInfoElem.SelectSingleNode("output");
            if (outputElem != null)
            {
                listInfo.OutputFlag = (outputElem.GetAttribute("disable") == "true") ? false : true;
            }
            XmlElement orderElem = (XmlElement)listInfoElem.SelectSingleNode("order");
            if (orderElem != null)
            {
                listInfo.Order = Int32.Parse(orderElem.InnerText);
            }
        }

        private void LoadSoundListOutputSettingColumnInfoArray(XmlElement listInfoElem, ref Nw4rSoundList.OutputSetting.TSoundListColumnInfo columnInfo)
        {
            XmlElement columnInfoArrayElem = (XmlElement)listInfoElem.SelectSingleNode("column_info_array");
            if (columnInfoArrayElem == null) return;

            foreach (XmlElement columnInfoElem in columnInfoArrayElem.SelectNodes("column_info"))
            {
                if (!columnInfoElem.HasAttribute("name")) continue;
                string name = columnInfoElem.GetAttribute("name");

                if (name == "index") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Index);
                else if (name == "name") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Name);
                else if (name == "type") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Type);
                else if (name == "volume") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Volume);
                else if (name == "player") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Player);
                else if (name == "actor_player") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.ActorPlayer);
                else if (name == "player_prio") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.PlayerPrio);
                else if (name == "comment") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Comment);
            }
        }

        private void LoadSoundListOutputSettingColumnInfoArray(XmlElement listInfoElem, ref Nw4rSoundList.OutputSetting.TGroupListColumnInfo columnInfo)
        {
            XmlElement columnInfoArrayElem = (XmlElement)listInfoElem.SelectSingleNode("column_info_array");
            if (columnInfoArrayElem == null) return;

            foreach (XmlElement columnInfoElem in columnInfoArrayElem.SelectNodes("column_info"))
            {
                if (!columnInfoElem.HasAttribute("name")) continue;
                string name = columnInfoElem.GetAttribute("name");

                if (name == "index") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Index);
                else if (name == "name") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Name);
                else if (name == "comment") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Comment);
            }
        }

        private void LoadSoundListOutputSettingColumnInfoArray(XmlElement listInfoElem, ref Nw4rSoundList.OutputSetting.TPlayerListColumnInfo columnInfo)
        {
            XmlElement columnInfoArrayElem = (XmlElement)listInfoElem.SelectSingleNode("column_info_array");
            if (columnInfoArrayElem == null) return;

            foreach (XmlElement columnInfoElem in columnInfoArrayElem.SelectNodes("column_info"))
            {
                if (!columnInfoElem.HasAttribute("name")) continue;
                string name = columnInfoElem.GetAttribute("name");

                if (name == "index") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Index);
                else if (name == "name") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Name);
                else if (name == "sound_limit") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.SoundLimit);
                else if (name == "comment") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Comment);
            }
        }

        private void LoadSoundListOutputSettingColumnInfoArray(XmlElement listInfoElem, ref Nw4rSoundList.OutputSetting.TBankListColumnInfo columnInfo)
        {
            XmlElement columnInfoArrayElem = (XmlElement)listInfoElem.SelectSingleNode("column_info_array");
            if (columnInfoArrayElem == null) return;

            foreach (XmlElement columnInfoElem in columnInfoArrayElem.SelectNodes("column_info"))
            {
                if (!columnInfoElem.HasAttribute("name")) continue;
                string name = columnInfoElem.GetAttribute("name");

                if (name == "index") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Index);
                else if (name == "name") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Name);
                else if (name == "comment") LoadSoundListOutputSettingColumnInfo(columnInfoElem, ref columnInfo.Comment);
            }
        }

        private void LoadSoundListOutputSettings(XmlDocument doc, XmlElement soundListOutputSettingArrayElem)
        {
            XmlElement elem; // テンポラリ用

            foreach (XmlElement settingElem in soundListOutputSettingArrayElem.SelectNodes("sound_list_output_setting"))
            {
                XmlElement outputFilePathElem = (XmlElement)settingElem.SelectSingleNode("output_filepath");
                if (outputFilePathElem != null && settingElem.HasAttribute("name"))
                {
                    string settingName = settingElem.GetAttribute("name");
                    string outputFilePath = Path.GetFullPath(Path.Combine(Path.GetDirectoryName(FilePath), outputFilePathElem.InnerText));

                    Nw4rSoundList.OutputSetting setting = new Nw4rSoundList.OutputSetting(settingName, outputFilePath);
                    elem = (XmlElement)settingElem.SelectSingleNode("title");
                    if (elem != null)
                    {
                        setting.Title = elem.InnerText;
                    }

                    elem = (XmlElement)settingElem.SelectSingleNode("output_table_type");
                    if (elem != null)
                    {
                        if (elem.InnerText == "single") setting.CombineSoundSet = true;
                        else if (elem.InnerText == "each_sound_set") setting.CombineSoundSet = false;
                    }
                    else
                    {
                        setting.CombineSoundSet = false;
                    }

                    XmlElement listOutputInfoArrayElem = (XmlElement)settingElem.SelectSingleNode("list_output_info_array");
                    if (listOutputInfoArrayElem != null)
                    {
                        foreach (XmlElement listInfoElem in listOutputInfoArrayElem.SelectNodes("list_output_info"))
                        {
                            if (!listInfoElem.HasAttribute("name")) continue;
                            string name = listInfoElem.GetAttribute("name");

                            if (name == "sound")
                            {
                                LoadSoundListOutputSettingListInfo(listInfoElem, ref setting.ListOutputInfo.SoundList);
                                LoadSoundListOutputSettingColumnInfoArray(listInfoElem, ref setting.SoundListColumnInfo);
                            }
                            else if (name == "group")
                            {
                                LoadSoundListOutputSettingListInfo(listInfoElem, ref setting.ListOutputInfo.GroupList);
                                LoadSoundListOutputSettingColumnInfoArray(listInfoElem, ref setting.GroupListColumnInfo);
                            }
                            else if (name == "player")
                            {
                                LoadSoundListOutputSettingListInfo(listInfoElem, ref setting.ListOutputInfo.PlayerList);
                                LoadSoundListOutputSettingColumnInfoArray(listInfoElem, ref setting.PlayerListColumnInfo);
                            }
                            else if (name == "bank")
                            {
                                LoadSoundListOutputSettingListInfo(listInfoElem, ref setting.ListOutputInfo.BankList);
                                LoadSoundListOutputSettingColumnInfoArray(listInfoElem, ref setting.BankListColumnInfo);
                            }
                        }
                    }

                    soundListOutputSettings.Add(setting);
                }
            }
        }

        private XmlElement SaveSoundListOutputSettingListInfo(XmlDocument doc, string name, ref Nw4rSoundList.OutputSetting.TListInfo listInfo)
        {
            XmlElement listInfoElem = doc.CreateElement("list_output_info");
            listInfoElem.SetAttribute("name", name);

            XmlElement outputElem = doc.CreateElement("output");
            if (!listInfo.OutputFlag) outputElem.SetAttribute("disable", "true");
            listInfoElem.AppendChild(outputElem);

            XmlElement orderElem = doc.CreateElement("order");
            orderElem.InnerText = listInfo.Order.ToString();
            listInfoElem.AppendChild(orderElem);

            return listInfoElem;
        }

        private void SaveSoundListOutputSettingColumnInfo(XmlDocument doc, XmlElement listColumnInfoElem, string name, ref Nw4rSoundList.OutputSetting.TColumnInfo columnInfo)
        {
            XmlElement columnInfoElem = doc.CreateElement("column_info");
            columnInfoElem.SetAttribute("name", name);

            XmlElement outputElem = doc.CreateElement("output");
            if (!columnInfo.OutputFlag) outputElem.SetAttribute("disable", "true");
            columnInfoElem.AppendChild(outputElem);

            XmlElement orderElem = doc.CreateElement("order");
            orderElem.InnerText = columnInfo.Order.ToString();
            columnInfoElem.AppendChild(orderElem);

            listColumnInfoElem.AppendChild(columnInfoElem);
        }

        private XmlElement ToXmlElementSoundListOutputSettings(XmlDocument doc)
        {
            XmlElement elem; // テンポラリ用

            if (soundListOutputSettings == null) return null;
            if (soundListOutputSettings.Count == 0) return null;

            XmlElement soundListOutputSettingArrayElem = doc.CreateElement("sound_list_output_setting_array");

            foreach (Nw4rSoundList.OutputSetting setting in soundListOutputSettings)
            {
                XmlElement settingElem = doc.CreateElement("sound_list_output_setting");
                settingElem.SetAttribute("name", setting.SettingName);

                {
                    XmlElement outputFilePathElem = doc.CreateElement("output_filepath");
                    outputFilePathElem.InnerText = NPath.MakeRelative(
                        setting.OutputFilePath, Path.GetDirectoryName(FilePath)
                    );
                    settingElem.AppendChild(outputFilePathElem);
                }
                {
                    elem = doc.CreateElement("title");
                    elem.InnerText = setting.Title;
                    settingElem.AppendChild(elem);
                }
                {
                    elem = doc.CreateElement("output_table_type");
                    if (setting.CombineSoundSet)
                    {
                        elem.InnerText = "single";
                    }
                    else
                    {
                        elem.InnerText = "each_sound_set";
                    }
                    settingElem.AppendChild(elem);
                }

                {
                    XmlElement listOutputInfoArrayElem = doc.CreateElement("list_output_info_array");

                    XmlElement soundListOutputInfoElem = SaveSoundListOutputSettingListInfo(doc, "sound", ref setting.ListOutputInfo.SoundList);
                    listOutputInfoArrayElem.AppendChild(soundListOutputInfoElem);

                    XmlElement groupListOutputInfoElem = SaveSoundListOutputSettingListInfo(doc, "group", ref setting.ListOutputInfo.GroupList);
                    listOutputInfoArrayElem.AppendChild(groupListOutputInfoElem);

                    XmlElement playerListOutputInfoElem = SaveSoundListOutputSettingListInfo(doc, "player", ref setting.ListOutputInfo.PlayerList);
                    listOutputInfoArrayElem.AppendChild(playerListOutputInfoElem);

                    XmlElement bankListOutputInfoElem = SaveSoundListOutputSettingListInfo(doc, "bank", ref setting.ListOutputInfo.BankList);
                    listOutputInfoArrayElem.AppendChild(bankListOutputInfoElem);

                    settingElem.AppendChild(listOutputInfoArrayElem);

                    {
                        XmlElement columnInfoArrayElem = doc.CreateElement("column_info_array");

                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "index", ref setting.SoundListColumnInfo.Index);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "name", ref setting.SoundListColumnInfo.Name);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "type", ref setting.SoundListColumnInfo.Type);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "volume", ref setting.SoundListColumnInfo.Volume);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "player", ref setting.SoundListColumnInfo.Player);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "actor_player", ref setting.SoundListColumnInfo.ActorPlayer);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "player_prio", ref setting.SoundListColumnInfo.PlayerPrio);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "comment", ref setting.SoundListColumnInfo.Comment);

                        soundListOutputInfoElem.AppendChild(columnInfoArrayElem);
                    }

                    {
                        XmlElement columnInfoArrayElem = doc.CreateElement("column_info_array");

                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "index", ref setting.GroupListColumnInfo.Index);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "name", ref setting.GroupListColumnInfo.Name);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "comment", ref setting.GroupListColumnInfo.Comment);

                        groupListOutputInfoElem.AppendChild(columnInfoArrayElem);
                    }

                    {
                        XmlElement columnInfoArrayElem = doc.CreateElement("column_info_array");

                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "index", ref setting.PlayerListColumnInfo.Index);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "name", ref setting.PlayerListColumnInfo.Name);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "sound_limit", ref setting.PlayerListColumnInfo.SoundLimit);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "comment", ref setting.PlayerListColumnInfo.Comment);

                        playerListOutputInfoElem.AppendChild(columnInfoArrayElem);
                    }

                    {
                        XmlElement columnInfoArrayElem = doc.CreateElement("column_info_array");

                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "index", ref setting.BankListColumnInfo.Index);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "name", ref setting.BankListColumnInfo.Name);
                        SaveSoundListOutputSettingColumnInfo(doc, columnInfoArrayElem, "comment", ref setting.BankListColumnInfo.Comment);

                        bankListOutputInfoElem.AppendChild(columnInfoArrayElem);
                    }

                }


                soundListOutputSettingArrayElem.AppendChild(settingElem);
            }

            return soundListOutputSettingArrayElem;
        }

        protected override void LoadBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement elem; // テンポラリ用

            soundSets.Clear();

            XmlElement soundMakerProjectElem = (XmlElement)bodyElem.SelectSingleNode("sound_maker_project");
            if (soundMakerProjectElem == null)
            {
                throw new Nw4rFileFormatException("Invalid Sound Maker project file format.");
            }

            elem = (XmlElement)soundMakerProjectElem.SelectSingleNode("intermediate_output_dir");
            if (elem != null)
            {
                intermediateOutputDir = elem.InnerText;
            }
            else
            {
                intermediateOutputDir = IntermediateOutputDirDefault;
            }

            elem = (XmlElement)soundMakerProjectElem.SelectSingleNode("aggregate_intermediate_output");
            if (elem != null)
            {
                aggregateIntermediateOutput = bool.Parse(elem.InnerText);
            }
            else
            {
                aggregateIntermediateOutput = false;
            }

            elem = (XmlElement)soundMakerProjectElem.SelectSingleNode("keep_intermediate_rseq");
            if (elem != null)
            {
                keepIntermediateRseq = bool.Parse(elem.InnerText);
            }
            else
            {
                keepIntermediateRseq = true;
            }

            elem = (XmlElement)soundMakerProjectElem.SelectSingleNode("output_dir");
            if (elem != null)
            {
                outputDir = elem.InnerText;
            }
            else
            {
                outputDir = string.Empty;
            }

            elem = (XmlElement)soundMakerProjectElem.SelectSingleNode("ext_file_dir");
            if (elem != null)
            {
                extFileDir = elem.InnerText;
            }

            smfconvTimebase = 48; // 互換性のため
            elem = (XmlElement)soundMakerProjectElem.SelectSingleNode("smfconv_timebase");
            if (elem != null)
            {
                smfconvTimebase = Int32.Parse(elem.InnerText);
            }

            XmlElement soundArchiveConvertSettingsElem = (XmlElement)soundMakerProjectElem.SelectSingleNode(Nw4rSoundArchiveConvertSettings.ElementName);
            if (soundArchiveConvertSettingsElem != null)
            {
                soundArchiveConvertSettings = Nw4rSoundArchiveConvertSettings.Parse(doc, soundArchiveConvertSettingsElem);
            }

            XmlElement soundArchivePlayerSettingsElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("sound_archive_player_settings");
            if (soundArchivePlayerSettingsElem != null)
            {
                soundArchivePlayerSettings = Nw4rSoundArchivePlayerSettings.Parse(doc, soundArchivePlayerSettingsElem);
            }

            XmlElement fileNameConvertSettingsElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("filename_to_itemname_convert_settings");
            if (fileNameConvertSettingsElem != null)
            {
                fileNameConvertSettings = FileNameToItemNameConvertSettings.Parse(fileNameConvertSettingsElem);
            }

            XmlElement soundSetArrayElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("sound_set_array");
            if (soundSetArrayElem != null)
            {
                foreach (XmlElement soundSetElem in soundSetArrayElem.SelectNodes("sound_set"))
                {
                    SoundSet soundSet = new SoundSet();

                    if (soundSetElem.HasAttribute("name"))
                    {
                        soundSet.Name = soundSetElem.GetAttribute("name");
                    }

                    elem = (XmlElement)soundSetElem.SelectSingleNode("file");
                    if (elem != null)
                    {
                        if (elem.HasAttribute("path"))
                        {
                            soundSet.FilePath = elem.GetAttribute("path");
                            soundSets.Add(soundSet);
                        }
                    }
                }
            }

            {
                XmlElement convertCommandEnabledElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("post_convert_command_enabled");
                if (convertCommandEnabledElem != null)
                {
                    postConvertCommandEnabled = bool.Parse(convertCommandEnabledElem.InnerText);
                }
            }

            {
                XmlElement convertCommandEnabledElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("pre_convert_command_enabled");
                if (convertCommandEnabledElem != null)
                {
                    preConvertCommandEnabled = bool.Parse(convertCommandEnabledElem.InnerText);
                }
            }

            {
                XmlElement convertCommandArrayElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("post_convert_command_array");
                if (convertCommandArrayElem != null)
                {
                    foreach (XmlElement commandElem in convertCommandArrayElem.SelectNodes("command"))
                    {
                        postConvertCommandList.Add(commandElem.InnerText);
                    }
                }
            }

            {
                XmlElement convertCommandArrayElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("pre_convert_command_array");
                if (convertCommandArrayElem != null)
                {
                    foreach (XmlElement commandElem in convertCommandArrayElem.SelectNodes("command"))
                    {
                        preConvertCommandList.Add(commandElem.InnerText);
                    }
                }
            }

            {
                XmlElement itemNameSettingsElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("item_name_settings");
                if (itemNameSettingsElem != null)
                {
                    foreach (XmlElement itemPrefixElem in itemNameSettingsElem.SelectNodes("item_prefix"))
                    {
                        if (!itemPrefixElem.HasAttribute("kind"))
                            continue;
                        string kind = itemPrefixElem.GetAttribute("kind");
                        if (itemNamePrefixList.ContainsKey(kind))
                        {
                            itemNamePrefixList[kind] = itemPrefixElem.InnerText;
                        }
                    }
                }
            }

            XmlElement soundListOutputSettingArrayElem = (XmlElement)soundMakerProjectElem.SelectSingleNode("sound_list_output_setting_array");
            if (soundListOutputSettingArrayElem != null)
            {
                LoadSoundListOutputSettings(doc, soundListOutputSettingArrayElem);
            }
        }

        protected override void SaveBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement elem; // テンポラリ用

            XmlElement soundMakerProjectElem = doc.CreateElement("sound_maker_project");
            bodyElem.AppendChild(soundMakerProjectElem);

            elem = doc.CreateElement("intermediate_output_dir");
            elem.InnerText = intermediateOutputDir;
            soundMakerProjectElem.AppendChild(elem);

            elem = doc.CreateElement("aggregate_intermediate_output");
            elem.InnerText = aggregateIntermediateOutput.ToString();
            soundMakerProjectElem.AppendChild(elem);

            elem = doc.CreateElement("keep_intermediate_rseq");
            elem.InnerText = keepIntermediateRseq.ToString();
            soundMakerProjectElem.AppendChild(elem);

            elem = doc.CreateElement("output_dir");
            elem.InnerText = outputDir;
            soundMakerProjectElem.AppendChild(elem);

            elem = doc.CreateElement("ext_file_dir");
            elem.InnerText = extFileDir;
            soundMakerProjectElem.AppendChild(elem);

            elem = doc.CreateElement("smfconv_timebase");
            elem.InnerText = smfconvTimebase.ToString();
            soundMakerProjectElem.AppendChild(elem);

            XmlElement soundArchiveConvertSettingsElem = soundArchiveConvertSettings.ToXmlElement(doc);
            soundMakerProjectElem.AppendChild(soundArchiveConvertSettingsElem);

            XmlElement soundArchivePlayerSettingsElem = soundArchivePlayerSettings.ToXmlElement(doc);
            soundMakerProjectElem.AppendChild(soundArchivePlayerSettingsElem);

            XmlElement fileNameConvertSettingsElem = fileNameConvertSettings.ToXmlElement(doc);
            soundMakerProjectElem.AppendChild(fileNameConvertSettingsElem);

            {
                XmlElement soundSetArrayElem = doc.CreateElement("sound_set_array");
                soundMakerProjectElem.AppendChild(soundSetArrayElem);

                foreach (SoundSet soundSet in soundSets)
                {
                    XmlElement soundSetElem = doc.CreateElement("sound_set");
                    if (soundSet.Name != null)
                    {
                        soundSetElem.SetAttribute("name", soundSet.Name);
                    }
                    if (soundSet.FilePath != null)
                    {
                        elem = doc.CreateElement("file");
                        elem.SetAttribute("path", soundSet.FilePath);
                        soundSetElem.AppendChild(elem);
                    }
                    soundSetArrayElem.AppendChild(soundSetElem);
                }
            }

            {
                XmlElement convertCommandArrayElem = doc.CreateElement("post_convert_command_enabled");
                soundMakerProjectElem.AppendChild(convertCommandArrayElem);
                convertCommandArrayElem.InnerText = postConvertCommandEnabled.ToString();
            }

            {
                XmlElement convertCommandArrayElem = doc.CreateElement("pre_convert_command_enabled");
                soundMakerProjectElem.AppendChild(convertCommandArrayElem);
                convertCommandArrayElem.InnerText = preConvertCommandEnabled.ToString();
            }

            {
                XmlElement convertCommandArrayElem = doc.CreateElement("post_convert_command_array");
                soundMakerProjectElem.AppendChild(convertCommandArrayElem);

                foreach (string command in postConvertCommandList)
                {
                    XmlElement commandElem = doc.CreateElement("command");
                    commandElem.InnerText = command;
                    convertCommandArrayElem.AppendChild(commandElem);
                }
            }
            {
                XmlElement convertCommandArrayElem = doc.CreateElement("pre_convert_command_array");
                soundMakerProjectElem.AppendChild(convertCommandArrayElem);

                foreach (string command in preConvertCommandList)
                {
                    XmlElement commandElem = doc.CreateElement("command");
                    commandElem.InnerText = command;
                    convertCommandArrayElem.AppendChild(commandElem);
                }
            }

            {
                XmlElement itemNameSettingsElem = doc.CreateElement("item_name_settings");
                soundMakerProjectElem.AppendChild(itemNameSettingsElem);

                foreach (KeyValuePair<string, string> kvp in itemNamePrefixList)
                {
                    XmlElement itemPrefixElem = doc.CreateElement("item_prefix");
                    itemPrefixElem.SetAttribute("kind", kvp.Key);
                    itemPrefixElem.InnerText = kvp.Value;
                    itemNameSettingsElem.AppendChild(itemPrefixElem);
                }
            }

            XmlElement soundListOutputSettingArrayElem = ToXmlElementSoundListOutputSettings(doc);
            if (soundListOutputSettingArrayElem != null)
            {
                soundMakerProjectElem.AppendChild(soundListOutputSettingArrayElem);
            }
        }

        public struct ChangeCase
        {
            public const int UpperCase = 0;
            public const int LowerCase = 1;
            public const int NoChange = 2;

            readonly static KeyValuePair<int, string>[] strTable = {
                new KeyValuePair<int, string>( UpperCase, "UpperCase" ),
                new KeyValuePair<int, string>( LowerCase, "LowerCase" ),
                new KeyValuePair<int, string>( NoChange, "NoChange" )
            };
            public static implicit operator int(ChangeCase s) { return s.val_; }
            public static implicit operator ChangeCase(int s) { return new ChangeCase(s); }
            public static ChangeCase Parse(string type) { return IntConverter.Parse(type, strTable); }
            public override string ToString() { return IntConverter.ToString(val_, strTable); }

            private ChangeCase(int e) { val_ = e; }
            private int val_;
        }

        public struct ChangeInvalidChar
        {
            public const int ReplaceUnderscore = 0;
            public const int RemoveChar = 1;

            readonly static KeyValuePair<int, string>[] strTable = {
                new KeyValuePair<int, string>( ReplaceUnderscore, "ReplaceUnderscore" ),
                new KeyValuePair<int, string>( RemoveChar, "RemoveChar" ),
            };
            public static implicit operator int(ChangeInvalidChar s) { return s.val_; }
            public static implicit operator ChangeInvalidChar(int s) { return new ChangeInvalidChar(s); }
            public static ChangeInvalidChar Parse(string type) { return IntConverter.Parse(type, strTable); }
            public override string ToString() { return IntConverter.ToString(val_, strTable); }

            private ChangeInvalidChar(int e) { val_ = e; }
            private int val_;
        }

        public class FileNameToItemNameConvertSettings
        {
            public ChangeCase ChangeCase = ChangeCase.UpperCase;
            public ChangeInvalidChar ChangeInvalidChar = ChangeInvalidChar.ReplaceUnderscore;
            public bool WithoutPrefix = false;

            internal XmlElement ToXmlElement(XmlDocument doc)
            {
                XmlElement xmlElem = doc.CreateElement("filename_to_itemname_convert_settings");
                xmlElem.SetAttribute("without_prefix", WithoutPrefix ? "yes" : "no");

                XmlElement changeCaseElem = doc.CreateElement("change_case");
                changeCaseElem.InnerText = ChangeCase.ToString();
                xmlElem.AppendChild(changeCaseElem);

                XmlElement changeInvalidCharElem = doc.CreateElement("change_invalid_char");
                changeInvalidCharElem.InnerText = ChangeInvalidChar.ToString();
                xmlElem.AppendChild(changeInvalidCharElem);

                return xmlElem;
            }
            internal static FileNameToItemNameConvertSettings Parse(XmlElement xmlElem)
            {
                FileNameToItemNameConvertSettings settings = new FileNameToItemNameConvertSettings();

                XmlElement changeCaseElem = (XmlElement)xmlElem.SelectSingleNode("change_case");
                if (changeCaseElem != null)
                {
                    settings.ChangeCase = ChangeCase.Parse(changeCaseElem.InnerText);
                }

                XmlElement changeInvalidCharElem = (XmlElement)xmlElem.SelectSingleNode("change_invalid_char");
                if (changeInvalidCharElem != null)
                {
                    settings.ChangeInvalidChar = ChangeInvalidChar.Parse(changeInvalidCharElem.InnerText);
                }

                if (xmlElem.HasAttribute("without_prefix"))
                {
                    string attr = xmlElem.GetAttribute("without_prefix");
                    if (attr == "yes") settings.WithoutPrefix = true;
                    else if (attr == "no") settings.WithoutPrefix = false;
                }

                return settings;
            }
        }
    }


}

