﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    public class Nw4rInstrumentList
    {
        #region ** HTMLソース

        private static readonly string StyleSheetSource = @"
body {
    margin: 12px;
}

h1 {
    font-size: 16pt;
    text-indent: 1em;
    border-left: solid #7886B6 20px;
    border-bottom: solid #7886B6 2px;
}

h2 {
    font-size: 14pt;
    text-indent: 1em;
    border-left: solid #7886B6 10px;
    border-bottom: solid #7886B6 1px;
}

h3 {
    font-size: 14pt;
}

div {
    margin-bottom: 20px;
}

div.tableset {
    margin: 0em 1em 1em 1em;
}

table {
    border-collapse: collapse;
    border-top: #a1a5a9 1px solid;
    border-bottom: #a1a5a9 1px solid;
}

th {
    padding-left: 0.5em;
    padding-right: 0.5em;
    text-align: left;
    background: #efefef;
}

th.SortKey {
    background: #e4e4ff;
}

tr.Disabled {
    color: #c0c0c0;
}

tr.ColorType1 {
    background-color: #ffaaaa;
}

tr.ColorType2 {
    background-color: #ffe080;
}

tr.ColorType3 {
    background-color: #ffffaa;
}

tr.ColorType4 {
    background-color: #d0ffc0;
}

tr.ColorType5 {
    background-color: #c0ecff;
}

tr.ColorType6 {
    background-color: #b4ddff;
}

tr.ColorType7 {
    background-color: #d8d8ff;
}

tr.ColorType8 {
    background-color: #ffddff;
}

td {
    padding-left: 0.5em;
    padding-right: 0.5em;
    border-top: #a1a5a9 1px dotted;
}

dl {
    font-size: 9pt;
}
";

        private static readonly string JScriptSource = @"
var gTrElems;
var gColId = new Array();
var gColIndexTmp;
var gReverse = new Array();

//==================================================================
// type - ""number"" or ""string""
function SortTable( type, obj )
{
    var cellArray = new Array();
    var rowArray = new Array();
    var trClassArray = new Array();
    var tdClassArray = new Array();
    var row;
    var col;

    if ( !document.getElementById ) { return; }

    var tableElem = obj.parentNode.parentNode.parentNode;
    var thElems = obj.parentNode.getElementsByTagName(""th"");

    for ( col = 0; col < thElems.length; col++ )
    {
        if ( thElems[col] == obj ) gColIndexTmp = col;
    }

    // init
    if ( typeof(gColId[tableElem]) == ""undefined"" ) {
        gReverse[tableElem] = 0;
        gColId[tableElem] = thElems[0];
    }

    var reverseFlag = 0;
    if ( obj == gColId[tableElem] ) {
        gReverse[tableElem] = - ( gReverse[tableElem] - 1 );
        if ( ! gReverse[tableElem] ) { reverseFlag = 1; }
    }
    else { gReverse[tableElem] = 0; }

    gTrElems = tableElem.getElementsByTagName(""tbody"")[0].getElementsByTagName(""tr"");

    //  read table
    if ( reverseFlag ) {
        for ( row = 0; row < gTrElems.length; row++ ) { rowArray[row] = gTrElems.length-row-1; }
    }
    else {
        for ( row = 0; row < gTrElems.length; row++ ) { rowArray[row] = row; }
    }
    for ( row = 0; row < gTrElems.length; row++ )
    {
        var tdElems = gTrElems[row].getElementsByTagName(""td"");
        trClassArray[row] = gTrElems[row].className;
        cellArray[row]  = new Array();
        tdClassArray[row] = new Array();
        for ( col = 0; col < tdElems.length; col++ )
        {
            cellArray[row][col]  = tdElems[col].innerHTML;
            tdClassArray[row][col] = tdElems[col].className;
        }
    }

    //  sort
    if ( type == ""number"" ) { rowArray.sort(CompareByNumber); }
    else if ( type == ""string"" ) { rowArray.sort(CompareByString); }

    if ( gReverse[tableElem] ) { rowArray.reverse(); }

    //  write table
    for ( row = 0; row < gTrElems.length; row++ )
    {
        gTrElems[row].className = trClassArray[rowArray[row]];

        var tdElems = gTrElems[row].getElementsByTagName(""td"");
        for (col = 0; col < tdElems.length; col++)
        {
            tdElems[col].innerHTML = cellArray[rowArray[row]][col];
            tdElems[col].className = tdClassArray[rowArray[row]][col];
        }
    }

    gColId[tableElem] = obj;

    //  table header
    for ( col = 0; col < thElems.length; col++ ) { thElems[col].className = """"; }
    obj.className = ""SortKey"";
}

//==================================================================
function CompareByNumber(a, b)
{
    var sA = gTrElems[a].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;
    var sB = gTrElems[b].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;

    sA = Number( sA );
    sB = Number( sB );

    if ( ( isNaN( sA ) && isNaN( sB ) ) || sA == sB ) { return 0; }

    if ( isNaN( sA ) ) { return 1; }
    if ( isNaN( sB ) ) { return -1; }

    if ( sA > sB ) { return 1; }
    return -1;
}

//==================================================================
function CompareByString(a, b)
{
    var sA = gTrElems[a].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;
    var sB = gTrElems[b].getElementsByTagName(""td"")[gColIndexTmp].innerHTML;

    if ( sA == sB ) { return 0; }

    if ( sA == ""&nbsp;"" ) { return 1; }
    if ( sB == ""&nbsp;"" ) { return -1; }

    if ( sA > sB ) { return 1; }
    return -1;
}

//==================================================================
function ResetAllTableOpenCloseButton()
{
    var inputElems = document.getElementsByTagName(""input"");
    for ( index = 0; index < inputElems.length; index++ )
    {
        if ( inputElems[index].className == ""tableopenclose"" )
        {
            inputElems[index].setAttribute( ""title"", ""close"" );
            inputElems[index].value = ""-"";
        }
    }

}

//==================================================================
function ShowAllSoundSet()
{
    var soundsetListElem = document.getElementById( ""soundsetlist"" );
    var soundsetElems = soundsetListElem.getElementsByTagName(""div"");

    for ( soundsetIndex = 0; soundsetIndex < soundsetElems.length; soundsetIndex++ )
    {
        soundsetElems[soundsetIndex].style.display = ""block"";
    }

    ResetAllTableOpenCloseButton();
}

//==================================================================
function ShowSoundSet( soundsetId )
{
    var soundsetListElem = document.getElementById( ""soundsetlist"" );
    var soundsetElems = soundsetListElem.getElementsByTagName(""div"");

    for ( soundsetIndex = 0; soundsetIndex < soundsetElems.length; soundsetIndex++ )
    {
        soundsetElems[soundsetIndex].style.display = ""none"";
    }

    var targetElem = document.getElementById( soundsetId );
    targetElem.style.display = ""block"";

    var datalistElems = targetElem.getElementsByTagName(""div"");
    for ( index = 0; index < datalistElems.length; index++ )
    {
        datalistElems[index].style.display = ""block"";
    }

    ResetAllTableOpenCloseButton();
}

//==================================================================
function SwitchDisplay( taragetId, obj )
{
    var target = document.getElementById( taragetId );
    if ( target.style.display == ""none"" )
    {
        target.style.display = ""block"";
        obj.setAttribute( ""title"", ""close"" );
        obj.value = ""-"";
    }
    else
    {
        target.style.display = ""none"";
        obj.setAttribute( ""title"", ""open"" );
        obj.value = ""+"";
    }
}
";

        #endregion

        private Nw4rXmlBank _xmlBank = null;
        private ColumnCollection _columns = new ColumnCollection();

        #region ** プロパティ

        public string Title
        {
            get { return Path.GetFileName(_xmlBank.FilePath) + " Instrument List"; }
        }

        public ColumnCollection Columns
        {
            get { return _columns; }
        }

        #endregion

        #region ** メソッド

        public void LoadBank(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

            _xmlBank = new Nw4rXmlBank();
            _xmlBank.Load(filePath);
        }

        public void WriteHtml(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

            using (FileStream stream = File.Open(filePath, FileMode.Create, FileAccess.Write, FileShare.None))
            {

                XmlTextWriter writer = new XmlTextWriter(stream, Encoding.UTF8);
                WriteHtml(writer);
                writer.Flush();

            }
        }

        private void WriteHtml(XmlTextWriter writer)
        {
            writer.Formatting = Formatting.Indented;

            writer.WriteStartDocument();

            writer.WriteComment(" saved from url=(0013)about:internet ");

            writer.WriteStartElement("html");

            WriterHtmlHeader(writer);
            WriteHtmlBody(writer);
            WriterHtmlFooter(writer);

            writer.WriteEndElement(); // html
            writer.WriteEndDocument();
        }

        private void WriterHtmlHeader(XmlTextWriter writer)
        {
            writer.WriteStartElement("head");

            writer.WriteStartElement("meta");
            writer.WriteAttributeString("name", "Author");
            writer.WriteAttributeString("content", _xmlBank.GeneratorName + " " + _xmlBank.GeneratorVersion);
            writer.WriteEndElement();

            writer.WriteStartElement("title");
            writer.WriteString(Title);
            writer.WriteEndElement();

            writer.WriteStartElement("style");
            writer.WriteAttributeString("type", "text/css");
            writer.WriteComment(StyleSheetSource);
            writer.WriteEndElement();

            writer.WriteStartElement("script");
            writer.WriteAttributeString("type", "text/javascript");
            writer.WriteComment(JScriptSource);

            writer.WriteEndElement(); // head
        }

        private void WriteHtmlBody(XmlTextWriter writer)
        {
            writer.WriteStartElement("body");

            // タイトル
            writer.WriteStartElement("h1");
            writer.WriteString(Title);
            writer.WriteEndElement();

            // インストルメントテーブル
            writer.WriteStartElement("h3");
            writer.WriteStartElement("input");
            writer.WriteAttributeString("type", "button");
            writer.WriteAttributeString("class", "tableopenclose");
            writer.WriteAttributeString("onclick", "SwitchDisplay('instrumentList', this)");
            writer.WriteAttributeString("title", "close");
            writer.WriteAttributeString("style", "font-size:10px; width:16px; height:16px; cursor:pointer");
            writer.WriteAttributeString("value", "-");
            writer.WriteEndElement(); // input
            writer.WriteString(" ");
            writer.WriteString("Instrument List");
            writer.WriteEndElement(); // h3


            writer.WriteStartElement("div");
            writer.WriteAttributeString("id", "instrumentList");

            writer.WriteStartElement("table");
            writer.WriteAttributeString("width", "100%");
            writer.WriteStartElement("thead");

            writer.WriteStartElement("tr");

            {
                int index = 0;

                foreach (IColumn column in _columns)
                {

                    writer.WriteStartElement("th");

                    if (index == 0)
                    {
                        writer.WriteAttributeString("class", "SortKey");
                    }

                    writer.WriteAttributeString("onclick", "SortTable('" + SortTypeToString(column.SortType) + "', this)");
                    writer.WriteAttributeString("title", "sort");
                    writer.WriteAttributeString("style", "cursor:pointer;");
                    writer.WriteString(column.Name);

                    writer.WriteEndElement(); // th

                    index++;

                }
            }

            writer.WriteEndElement(); // tr
            writer.WriteEndElement(); // thead

            writer.WriteStartElement("tbody");

            {
                int index = 0;

                foreach (Nw4rXmlBankInst instrument in _xmlBank.InstList)
                {

                    writer.WriteStartElement("tr");

                    string className = string.Empty;

                    if (!instrument.Enabled)
                    {
                        className += "Disabled";
                    }

                    if (0 < instrument.ColorIndex)
                    {
                        if (0 < className.Length)
                        {
                            className += " ";
                        }
                        className += "ColorType" + instrument.ColorIndex.ToString();
                    }

                    if (0 < className.Length)
                    {
                        writer.WriteAttributeString("class", className);
                    }

                    foreach (IColumn column in _columns)
                    {

                        writer.WriteStartElement("td");

                        string text = column.GetText(new ListItem(index, instrument));

                        if (0 < text.Length)
                        {
                            writer.WriteString(text);
                        }
                        else
                        {
                            writer.WriteEntityRef("nbsp");
                        }

                        writer.WriteEndElement();

                    }

                    writer.WriteEndElement(); // tr
                    index++;

                }
            }

            writer.WriteEndElement(); // tbody
            writer.WriteEndElement(); // table

            writer.WriteEndElement(); // div

            writer.WriteEndElement(); // body
        }

        private void WriterHtmlFooter(XmlTextWriter writer)
        {
            // フッタ
            writer.WriteStartElement("hr");
            writer.WriteEndElement(); // hr

            writer.WriteStartElement("dl");

            writer.WriteStartElement("dt");
            writer.WriteString("Bank:");
            writer.WriteEndElement(); // dt
            writer.WriteStartElement("dd");
            writer.WriteString(_xmlBank.FilePath);
            writer.WriteEndElement(); // dd

            writer.WriteStartElement("dt");
            writer.WriteString("DateTime:");
            writer.WriteEndElement(); // dt
            writer.WriteStartElement("dd");
            writer.WriteString(DateTime.Now.ToString("yyyy-MM-ddTHH:mm:ss"));
            writer.WriteEndElement(); // dd

            writer.WriteEndElement(); // dl
        }

        private string SortTypeToString(ColumnSortType type)
        {
            switch (type)
            {
                case ColumnSortType.Number:
                    return "number";

                case ColumnSortType.String:
                    return "string";
            }

            Debug.Assert(false);
            return "string";
        }

        #endregion

        #region ** アイテムクラス

        public class ListItem
        {
            private int _index = -1;
            private Nw4rXmlBankInst _data = null;

            public ListItem(int index, Nw4rXmlBankInst xmlData)
            {
                Debug.Assert(0 <= index);
                Debug.Assert(null != xmlData);

                _index = index;
                _data = xmlData;
            }

            #region ** プロパティ

            public int Index
            {
                get { return _index; }
            }

            public Nw4rXmlBankInst Data
            {
                get { return _data; }
            }

            #endregion
        }

        #endregion

        #region ** カラムクラス

        public enum ColumnSortType
        {
            Number,
            String,
        }

        public interface IColumn
        {
            #region ** プロパティ

            string Name { get; }

            ColumnSortType SortType { get; }

            #endregion

            #region ** メソッド

            string GetText(ListItem item);

            #endregion
        }

        public class ColumnIndex : IColumn
        {
            public const string Name = "index";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                return item.Index.ToString();
            }

            #endregion
        }

        public class ColumnProgramNumber : IColumn
        {
            public const string Name = "program";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                return item.Data.PrgNo.ToString();
            }

            #endregion
        }

        public class ColumnLabel : IColumn
        {
            public const string Name = "name";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.String; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                return item.Data.Label;
            }

            #endregion
        }

        public class ColumnFileFormat : IColumn
        {
            public const string Name = "compression format";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.String; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (0 == item.Data.KeyRegionCount) { return string.Empty; }

                Nw4rXmlBankPcmFileFormat fileFormat = Nw4rXmlBankPcmFileFormat.Invalid;

                foreach (Nw4rXmlBankInstKeyRegion keyRegion in item.Data)
                {
                    foreach (Nw4rXmlBankInstVelRegion velocityRegion in keyRegion)
                    {

                        if (Nw4rXmlBankPcmFileFormat.Invalid == fileFormat)
                        {
                            fileFormat = velocityRegion.FileFormat;
                        }
                        else
                        {
                            if (fileFormat != velocityRegion.FileFormat) { return "*"; }
                        }

                    }
                }

                return FileFormatToString(fileFormat);
            }

            private string FileFormatToString(Nw4rXmlBankPcmFileFormat fileFormat)
            {
                switch (fileFormat)
                {
                    case Nw4rXmlBankPcmFileFormat.Adpcm:
                        return "ADPCM";

                    case Nw4rXmlBankPcmFileFormat.Pcm16:
                        return "PCM16";

                    case Nw4rXmlBankPcmFileFormat.Pcm8:
                        return "PCM8";
                }

                return string.Empty;
            }

            #endregion
        }

        public class ColumnFileName : IColumn
        {
            public const string Name = "file name";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.String; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (0 == item.Data.KeyRegionCount) { return string.Empty; }

                string filePath = string.Empty;

                foreach (Nw4rXmlBankInstKeyRegion keyRegion in item.Data)
                {
                    foreach (Nw4rXmlBankInstVelRegion velocityRegion in keyRegion)
                    {

                        if (null == velocityRegion.FilePath) { continue; }

                        if (0 == filePath.Length)
                        {
                            filePath = velocityRegion.FilePath;
                        }
                        else
                        {
                            if (filePath != velocityRegion.FilePath) { return "*"; }
                        }

                    }
                }

                return Path.GetFileName(filePath);
            }

            #endregion
        }

        public class ColumnOriginalKey : IColumn
        {
            public const string Name = "original key";

            #region ** キーラベル

            private static string[] KeyLabels = new String[] {
                "cnm1", "csm1", "dnm1", "dsm1", "enm1", "fnm1", "fsm1", "gnm1", "gsm1", "anm1", "asm1", "bnm1",
                "cn0", "cs0", "dn0", "ds0", "en0", "fn0", "fs0", "gn0", "gs0", "an0", "as0", "bn0",
                "cn1", "cs1", "dn1", "ds1", "en1", "fn1", "fs1", "gn1", "gs1", "an1", "as1", "bn1",
                "cn2", "cs2", "dn2", "ds2", "en2", "fn2", "fs2", "gn2", "gs2", "an2", "as2", "bn2",
                "cn3", "cs3", "dn3", "ds3", "en3", "fn3", "fs3", "gn3", "gs3", "an3", "as3", "bn3",
                "cn4", "cs4", "dn4", "ds4", "en4", "fn4", "fs4", "gn4", "gs4", "an4", "as4", "bn4",
                "cn5", "cs5", "dn5", "ds5", "en5", "fn5", "fs5", "gn5", "gs5", "an5", "as5", "bn5",
                "cn6", "cs6", "dn6", "ds6", "en6", "fn6", "fs6", "gn6", "gs6", "an6", "as6", "bn6",
                "cn7", "cs7", "dn7", "ds7", "en7", "fn7", "fs7", "gn7", "gs7", "an7", "as7", "bn7",
                "cn8", "cs8", "dn8", "ds8", "en8", "fn8", "fs8", "gn8", "gs8", "an8", "as8", "bn8",
                "cn9", "cs9", "dn9", "ds9", "en9", "fn9", "fs9", "gn9",
            };

            #endregion

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.String; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (0 == item.Data.KeyRegionCount) { return string.Empty; }

                int originalKey = 0;

                foreach (Nw4rXmlBankInstKeyRegion keyRegion in item.Data)
                {
                    foreach (Nw4rXmlBankInstVelRegion velocityRegion in keyRegion)
                    {

                        if (0 == originalKey)
                        {
                            originalKey = velocityRegion.OriginalKey;
                        }
                        else
                        {
                            if (originalKey != velocityRegion.OriginalKey) { return "*"; }
                        }

                    }
                }

                return KeyToString(originalKey);
            }

            /// <summary>
            /// キーを文字列に変換します。
            /// </summary>
            /// <param name="key">キー</param>
            /// <returns>変換後の文字列</returns>
            private String KeyToString(int key)
            {
                if (key < 0 || key > 127) { return string.Empty; }
                return KeyLabels[key];
            }

            #endregion
        }

        public class ColumnAttack : IColumn
        {
            public const string Name = "EnvA";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (Nw4rXmlBankAdsrEnvelopeSelect.Inst != item.Data.AdsrEnvelopeSelect) { return "*"; }
                return item.Data.AdsrEnvelope.Attack.ToString();
            }

            #endregion
        }

        public class ColumnHold : IColumn
        {
            public const string Name = "EnvH";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (Nw4rXmlBankAdsrEnvelopeSelect.Inst != item.Data.AdsrEnvelopeSelect) { return "*"; }
                return item.Data.AdsrEnvelope.Hold.ToString();
            }

            #endregion
        }

        public class ColumnDecay : IColumn
        {
            public const string Name = "EnvD";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (Nw4rXmlBankAdsrEnvelopeSelect.Inst != item.Data.AdsrEnvelopeSelect) { return "*"; }
                return item.Data.AdsrEnvelope.Decay.ToString();
            }

            #endregion
        }

        public class ColumnSustain : IColumn
        {
            public const string Name = "EnvS";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (Nw4rXmlBankAdsrEnvelopeSelect.Inst != item.Data.AdsrEnvelopeSelect) { return "*"; }
                return item.Data.AdsrEnvelope.Sustain.ToString();
            }

            #endregion
        }

        public class ColumnRelease : IColumn
        {
            public const string Name = "EnvR";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                if (Nw4rXmlBankAdsrEnvelopeSelect.Inst != item.Data.AdsrEnvelopeSelect) { return "*"; }
                return item.Data.AdsrEnvelope.Release.ToString();
            }

            #endregion
        }

        public class ColumnVolume : IColumn
        {
            public const string Name = "volume";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                return item.Data.Volume.ToString();
            }

            #endregion
        }

        public class ColumnPitchSemitones : IColumn
        {
            public const string Name = "pitch (semitones)";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                return item.Data.CoarseTune.ToString();
            }

            #endregion
        }

        public class ColumnPitchCents : IColumn
        {
            public const string Name = "pitch (cents)";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.Number; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                return item.Data.FineTune.ToString();
            }

            #endregion
        }

        public class ColumnComment : IColumn
        {
            public const string Name = "comment";

            #region ** プロパティ

            string IColumn.Name
            {
                get { return Name; }
            }

            public ColumnSortType SortType
            {
                get { return ColumnSortType.String; }
            }

            #endregion

            #region ** メソッド

            public string GetText(ListItem item)
            {
                if (null == item) { return string.Empty; }
                return item.Data.Comment;
            }

            #endregion
        }

        #region ** コレクション

        public class ColumnCollection : Collection<IColumn> { }

        #endregion

        #endregion
    }
}
