﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

using NintendoWare.SoundFoundation.Legacies.Resources;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    public class Nw4rFileFormatException : SoundException
    {
        private string _ownerLabel = string.Empty;

        private Nw4rFileFormatException[] _exceptions = null;

        public Nw4rFileFormatException() { }
        public Nw4rFileFormatException(string message) : this(message, string.Empty, null) { }
        public Nw4rFileFormatException(string message, string ownerLabel) : this(message, ownerLabel, null) { }
        public Nw4rFileFormatException(string message, string ownerLabel, Exception innerException)
            : base(message, innerException)
        {
            _ownerLabel = ownerLabel;
        }
        public Nw4rFileFormatException(Nw4rFileFormatExceptionList exceptions)
        {
            _exceptions = (null != exceptions) ? exceptions.ToArray() : null;
        }

        #region ** プロパティ

        public string OwnerLabel
        {
            get { return _ownerLabel; }
        }

        public override string Message
        {
            get { return ErrorHeader + ErrorMessage; }
        }

        public Nw4rFileFormatException[] Exceptions
        {
            get { return _exceptions; }
        }

        protected virtual string ErrorHeader
        {
            get
            {
                if (0 == OwnerLabel.Length)
                {
                    return Messages.ErrorMessageHeader_General;
                }
                else
                {
                    return string.Format(Messages.ErrorMessageHeaderFormat_GeneralWithLabel, OwnerLabel);
                }
            }
        }

        protected virtual string ErrorMessage
        {
            get { return base.Message; }
        }

        #endregion
    }

    #region ** システムエラー

    public class Nw4rFileFormatInternalException : Nw4rFileFormatException
    {
        public Nw4rFileFormatInternalException() { }
        public Nw4rFileFormatInternalException(string message) : base(message) { }
        public Nw4rFileFormatInternalException(Exception innerException) : base(string.Empty, string.Empty, innerException) { }

        #region ** プロパティ

        protected override string ErrorHeader
        {
            get { return Messages.ErrorMessageHeader_Internal; }
        }

        protected override string ErrorMessage
        {
            get
            {
                if (0 == base.ErrorMessage.Length) { return Messages.ErrorMessage_Internal; }
                return base.ErrorMessage;
            }
        }

        #endregion
    }

    public class Nw4rProjectNotLoadedException : Nw4rFileFormatInternalException
    {
        protected override string ErrorMessage
        {
            get { return Messages.ErrorMessage_ProjectNotLoaded; }
        }
    }

    #endregion

    #region ** ファイルエラー

    public class Nw4rFileException : Nw4rFileFormatException
    {
        #region ** フィールド

        private string _filePath = string.Empty;

        #endregion

        public Nw4rFileException(string filePath, string ownerLabel) : base(string.Empty, ownerLabel)
        {
            Debug.Assert(null != filePath);
            _filePath = filePath;
        }

        #region ** プロパティ

        public string FilePath
        {
            get { return _filePath; }
        }

        #endregion
    }

    public class Nw4rFileNotFoundException : Nw4rFileException
    {
        public Nw4rFileNotFoundException(string filePath, string ownerLabel) : base(filePath, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return Messages.ErrorMessage_FileNotFound + " \"" + FilePath + "\""; }
        }

        #endregion
    }

    public class Nw4rFileAccessDeniedException : Nw4rFileException
    {
        public Nw4rFileAccessDeniedException(string filePath, string ownerLabel) : base(filePath, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return Messages.ErrorMessage_FileAcceddDenied + " \"" + FilePath + "\""; }
        }

        #endregion
    }

    public class Nw4rFilePathTooLongException : Nw4rFileException
    {
        public Nw4rFilePathTooLongException(string filePath, string ownerLabel) : base(filePath, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return Messages.ErrorMessage_FilePathTooLong + " \"" + FilePath + "\""; }
        }

        #endregion
    }

    public class Nw4rInvalidFileVersionException : Nw4rFileException
    {
        private UInt16 _version = 0;

        public Nw4rInvalidFileVersionException(string filePath, UInt16 version) : base(filePath, string.Empty)
        {
            _version = version;
        }

        #region ** プロパティ

        public int MajorVersion
        {
            get { return Nw4rFileHeader.GetMajorVersion(_version); }
        }

        public int MinorVersion
        {
            get { return Nw4rFileHeader.GetMinorVersion(_version); }
        }

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_UnsupportedFileVersion, VersionString, FilePath); }
        }

        private string VersionString
        {
            get { return MajorVersion.ToString() + "." + MinorVersion.ToString(); }
        }

        #endregion
    }

    public class Nw4rInvalidFileException : Nw4rFileException
    {
        public Nw4rInvalidFileException(string filePath, string ownerLabel) : base(filePath, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return Messages.ErrorMessage_InvalidFile + " \"" + FilePath + "\""; }
        }

        #endregion
    }

    #endregion

    #region ** ラベルエラー

    public class Nw4rLabelException : Nw4rFileFormatException
    {
        #region ** フィールド

        private string _label = string.Empty;

        #endregion

        public Nw4rLabelException(string label, string ownerLabel) : base(string.Empty, ownerLabel)
        {
            Debug.Assert(null != label);
            _label = label;
        }

        #region ** プロパティ

        public string Label
        {
            get { return _label; }
        }

        #endregion
    }

    public class Nw4rLabelNotFoundException : Nw4rLabelException
    {
        public Nw4rLabelNotFoundException(string label, string ownerLabel) : base(label, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_LabelNotFound, Label); }
        }

        #endregion
    }

    public class Nw4rMultipleLabelException : Nw4rLabelException
    {
        public Nw4rMultipleLabelException(string label) : base(label, string.Empty) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_LabelMultipleDefinition, Label); }
        }

        #endregion
    }

    public class Nw4rBankLabelNotFoundException : Nw4rLabelException
    {
        public Nw4rBankLabelNotFoundException(string label, string ownerLabel) : base(label, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_BankLabelNotFound, Label); }
        }

        #endregion
    }

    public class Nw4rPlayerLabelNotFoundException : Nw4rLabelException
    {
        public Nw4rPlayerLabelNotFoundException(string label, string ownerLabel) : base(label, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_PlayerLabelNotFound, Label); }
        }

        #endregion
    }

    public class Nw4rInvalidLabelException : Nw4rLabelException
    {
        public Nw4rInvalidLabelException(string label, string ownerLabel) : base(label, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_InvalidLabel, Label); }
        }

        #endregion
    }

    public class Nw4rSequenceStartLabelNotFoundException : Nw4rLabelException
    {
        public Nw4rSequenceStartLabelNotFoundException(string label, string ownerLabel) : base(label, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_StartLabelNotFound, Label); }
        }

        #endregion
    }

    public class Nw4rMultipleInnerGroupException : Nw4rLabelException
    {
        public Nw4rMultipleInnerGroupException(string label, string ownerLabel) : base(label, ownerLabel) { }

        #region ** プロパティ

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_InnerGroupMultipleDefinition, Label); }
        }

        #endregion
    }

    #endregion

    #region ** パラメータエラー

    public class Nw4rParameterException : Nw4rFileFormatException
    {
        #region ** フィールド

        private string _parameterName = string.Empty;

        #endregion

        public Nw4rParameterException(string parameterName, string ownerLabel) : base(string.Empty, ownerLabel, null) { }

        #region ** プロパティ

        public string ParameterName
        {
            get { return _parameterName; }
        }

        protected override string ErrorMessage
        {
            get { return string.Format(Messages.ErrorMessageFormat_InvalidParameter, ParameterName); }
        }

        #endregion
    }

    #endregion
}
