﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;
using NintendoWare.SoundFoundation.Legacies.FileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rBinarySoundArchive
    {
        FileStream mFileStream;
        int mVersion;
        string[] mSymbolTable;
        string[] mSoundSymbolTable;
        GroupInfo[] mGroupInfoTable;

        public struct GroupInfo
        {
            public uint size;
            public uint waveDataSize;
        }

        public string[] SoundLabelList
        {
            get { return mSoundSymbolTable; }
        }
        public GroupInfo[] GroupInfoList
        {
            get { return mGroupInfoTable; }
        }

        public void Close()
        {
            if (mFileStream != null)
            {
                mFileStream.Close();
                mFileStream = null;
            }
        }

        public void Open(string filePath)
        {
            mFileStream = File.OpenRead(filePath);
            BinaryReader reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

            string fileChunk = new string(reader.ReadChars(4));
            if (fileChunk != "RSAR")
            {
                throw new Nw4rFileFormatException("Unexpcted file format \"" + filePath + "\"");
            }

            int byteOrder = reader.ReadUInt16();
            mVersion = reader.ReadUInt16();

            if (mVersion < 0x0010)
            {
                throw new Nw4rFileFormatException("Unsupported File Version \"" + mVersion + "\"");
            }

            uint fileSize = reader.ReadUInt32();
            int headerSize = reader.ReadUInt16();
            int dataBlocks = reader.ReadUInt16();

            uint symbolDataOffset = reader.ReadUInt32();
            uint symbolDataSize = reader.ReadUInt32();

            uint infoDataOffset = reader.ReadUInt32();
            uint infoDataSize = reader.ReadUInt32();

            {
                mFileStream.Seek(symbolDataOffset, SeekOrigin.Begin);
                reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                uint offsetBase = symbolDataOffset + 8;

                string symbolChunk = new string(reader.ReadChars(4));
                if (symbolChunk != "SYMB")
                {
                    throw new Nw4rFileFormatException("Unexpcted chunk \"" + filePath + "\"");
                }
                uint chunkSize = reader.ReadUInt32();
                uint tableOffset = reader.ReadUInt32();

                mFileStream.Seek(offsetBase + tableOffset, SeekOrigin.Begin);
                reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                uint symbolCount = reader.ReadUInt32();
                uint[] symbolOffsetTable = new uint[symbolCount];
                for (int i = 0; i < symbolCount; i++)
                {
                    symbolOffsetTable[i] = reader.ReadUInt32();
                }

                mSymbolTable = new string[symbolCount];
                for (int i = 0; i < symbolCount; i++)
                {
                    mFileStream.Seek(offsetBase + symbolOffsetTable[i], SeekOrigin.Begin);
                    reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                    mSymbolTable[i] = String.Empty;
                    while (reader.PeekChar() != '\0')
                    {
                        mSymbolTable[i] += reader.ReadChar();
                    }
                    reader.ReadChar();
                }
            }

            // INFOブロック
            {
                mFileStream.Seek(infoDataOffset, SeekOrigin.Begin);
                reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                uint offsetBase = infoDataOffset + 8;

                string infoChunk = new string(reader.ReadChars(4));
                if (infoChunk != "INFO")
                {
                    throw new Nw4rFileFormatException("Unexpcted chunk \"" + filePath + "\"");
                }
                uint chunkSize = reader.ReadUInt32();
                reader.ReadUInt32(); // soundTableOffset dataref header
                uint soundTableOffset = reader.ReadUInt32(); // soundTableOffset dataref offset
                reader.ReadUInt32(); // bankTableRef dataref header
                reader.ReadUInt32(); // bankTableRef dataref offset
                reader.ReadUInt32(); // playerTableRef dataref header
                reader.ReadUInt32(); // playerTableRef dataref offset
                reader.ReadUInt32(); // fileTableRef dataref header
                reader.ReadUInt32(); // fileTableRef dataref offset
                reader.ReadUInt32(); // groupTableRef dataref header
                uint groupTableOffset = reader.ReadUInt32(); // groupTableRef dataref offset

                // サウンド情報テーブル
                mFileStream.Seek(offsetBase + soundTableOffset, SeekOrigin.Begin);
                reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                uint soundCount = reader.ReadUInt32();
                uint[] soundOffsetTable = new uint[soundCount];
                for (int i = 0; i < soundCount; i++)
                {
                    reader.ReadUInt32();
                    soundOffsetTable[i] = reader.ReadUInt32();
                }

                // サウンド情報
                mSoundSymbolTable = new string[soundCount];
                for (int i = 0; i < soundCount; i++)
                {
                    mFileStream.Seek(offsetBase + soundOffsetTable[i], SeekOrigin.Begin);
                    reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                    uint stringId = reader.ReadUInt32();
                    mSoundSymbolTable[i] = mSymbolTable[stringId];
                }

                // グループ情報テーブル
                mFileStream.Seek(offsetBase + groupTableOffset, SeekOrigin.Begin);
                reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                uint groupCount = reader.ReadUInt32();
                --groupCount; // 無名グループを除く
                uint[] groupOffsetTable = new uint[groupCount];
                for (int i = 0; i < groupCount; i++)
                {
                    reader.ReadUInt32();
                    groupOffsetTable[i] = reader.ReadUInt32();
                }

                mGroupInfoTable = new GroupInfo[groupCount];
                for (int i = 0; i < groupCount; i++)
                {
                    mFileStream.Seek(offsetBase + groupOffsetTable[i], SeekOrigin.Begin);
                    reader = BinaryReaderBigEndian.CreateInstance(mFileStream, new System.Text.ASCIIEncoding());

                    reader.ReadUInt32(); // stringId
                    reader.ReadInt32(); // entryNum
                    reader.ReadUInt32(); // extFilePathRef header
                    reader.ReadUInt32(); // extFilePathRef offset
                    reader.ReadUInt32(); // offset
                    mGroupInfoTable[i].size = reader.ReadUInt32(); // size
                    reader.ReadUInt32(); // waveDataOffset
                    mGroupInfoTable[i].waveDataSize = reader.ReadUInt32(); // waveDataSize
                }
            }
        }

    }

}
