﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Converter
{
    #region ** 固定値

    public enum LineLevel
    {
        Level0 = 0, 		// 詳細情報
        Level1 = 1, 		// 情報
        Level2 = 2, 		// 警告
        Level3 = 3, 		// エラー

        Detail = Level0, 	// 詳細情報
        Information = Level1, 	// 情報
        Warning = Level2, 	// 警告
        Error = Level3, 	// エラー
    }

    #endregion

    public class OutputLine
    {
        #region ** フィールド

        private LineLevel _level = LineLevel.Level1;
        private string _text = string.Empty;
        private Nw4rItem[] _targetItems = null;

        #endregion

        public OutputLine(LineLevel level, string text) : this(level, text, new Nw4rItem[0]) { }
        public OutputLine(LineLevel level, string text, Nw4rItem targetItem) : this(level, text, new Nw4rItem[] { targetItem }) { }
        public OutputLine(LineLevel level, string text, Nw4rItem[] targetItems)
        {
            _level = level;
            _text = (null != text) ? text : string.Empty;
            _targetItems = (null == targetItems) ? new Nw4rItem[0] : targetItems;
        }

        #region ** プロパティ

        public LineLevel Level
        {
            get { return _level; }
            set { _level = value; }
        }

        public string Text
        {
            get { return _text; }
            set { _text = value; }
        }

        public Nw4rItem[] TargetItems
        {
            get { return _targetItems; }
            set { _targetItems = value; }
        }

        #endregion
    }

    public class InformationLine : OutputLine
    {
        public InformationLine(string text) : this(LineLevel.Information, text, null) { }
        public InformationLine(string text, Nw4rSoundArchiveItem targetItem) : this(LineLevel.Information, text, targetItem) { }
        public InformationLine(LineLevel level, string text) : this(level, text, null) { }
        public InformationLine(LineLevel level, string text, Nw4rSoundArchiveItem targetItem) : base(level, text, targetItem) { }
    }

    public class WarningLine : OutputLine
    {
        public WarningLine(string text) : this(text, new Nw4rItem[0]) { }
        public WarningLine(string text, Nw4rItem targetItem) : base(LineLevel.Warning, text, targetItem) { }
        public WarningLine(string text, Nw4rItem[] targetItems) : base(LineLevel.Warning, text, targetItems) { }
    }

    public class ErrorLine : OutputLine
    {
        public ErrorLine(string text) : this(text, new Nw4rItem[0]) { }
        public ErrorLine(string text, Nw4rItem targetItem) : base(LineLevel.Error, text, targetItem) { }
        public ErrorLine(string text, Nw4rItem[] targetItems) : base(LineLevel.Error, text, targetItems) { }
    }

    public abstract class OutputMessage
    {
        public string Text
        {
            get { return Header + Message; }
        }

        protected virtual string Header
        {
            get { return string.Empty; }
        }

        protected abstract string Message { get; }
    }

    #region ** メッセージ クラス

    public abstract class InformationMessage : OutputMessage { }

    public class BeginConvertMessage : InformationMessage
    {
        private string _projectName = string.Empty;

        public BeginConvertMessage(string projectName)
        {
            _projectName = (null == projectName) ? string.Empty : projectName;
        }

        protected override string Message
        {
            get { return string.Format(Resources.Messages.MessageFormat_BeginConvert, _projectName); }
        }
    }

    public class EndConvertSucceededMessage : InformationMessage
    {
        private string _projectName = string.Empty;
        private long _convertTime = 0;

        public EndConvertSucceededMessage(string projectName, long convertTime)
        {
            _projectName = (null == projectName) ? string.Empty : projectName;
            _convertTime = (0 > convertTime) ? 0 : convertTime;
        }

        protected override string Message
        {
            get
            {
                if (0 >= _convertTime)
                {
                    return string.Format(Resources.Messages.MessageFormat_EndConvertSucceeded, _projectName);
                }
                else
                {
                    return string.Format(Resources.Messages.MessageFormat_EndConvertSucceeded_WithTime,
                                            _projectName, _convertTime);
                }
            }
        }
    }

    public class EndConvertFailedMessage : InformationMessage
    {
        private string _projectName = string.Empty;

        public EndConvertFailedMessage(string projectName)
        {
            _projectName = (null == projectName) ? string.Empty : projectName;
        }

        protected override string Message
        {
            get { return string.Format(Resources.Messages.MessageFormat_EndConvertFailed, _projectName); }
        }
    }

    public class EndConvertCanceledMessage : InformationMessage
    {
        private string _projectName = string.Empty;

        public EndConvertCanceledMessage(string projectName)
        {
            _projectName = (null == projectName) ? string.Empty : projectName;
        }

        protected override string Message
        {
            get { return string.Format(Resources.Messages.MessageFormat_EndConvertCanceled, _projectName); }
        }
    }

    public class WarningMessage : OutputMessage
    {
        private string _label = string.Empty;
        private string _message = string.Empty;

        public WarningMessage(string label, string message)
        {
            _label = (null == label) ? string.Empty : label;
            _message = (null == message) ? string.Empty : message;
        }

        protected override string Header
        {
            get
            {
                if (0 == _label.Length)
                {
                    return Resources.Messages.WarningMessageHeader_General;
                }
                else
                {
                    return string.Format(Resources.Messages.WarningMessageHeaderFormat_GeneralWithLabel, _label);
                }
            }
        }

        protected override string Message
        {
            get { return _message; }
        }

        protected string Label
        {
            get { return _label; }
        }
    }

    public class WarningUnreferencedItem : WarningMessage
    {
        public WarningUnreferencedItem(string label) : base(label, null) { }

        protected override string Message
        {
            get { return string.Format(Resources.Messages.WarningMessageFormat_UnreferencedItem, Label); }
        }
    }

    public class WarningIndependentItem : WarningMessage
    {
        public WarningIndependentItem(string label) : base(label, null) { }

        protected override string Message
        {
            get { return string.Format(Resources.Messages.WarningMessageFormat_IndependentItem, Label); }
        }
    }

    #endregion
}
