﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Converter
{
    internal class Nw4rConvert2SequenceCommand : Nw4rConvertCommandT<Nw4rSequenceSound, Nw4rSequenceSoundOutput>
    {
        #region ** 固定値

        private static readonly string ConverterFileName = "nw4r_seqconv.exe";
        private static readonly string ConvertOption_UPDATE = "--update ";
        private static readonly string ConvertOption_OUTPUT = "-o ";

        #endregion

        public Nw4rConvert2SequenceCommand(Nw4rSequenceSound input, Nw4rSequenceSoundOutput output)
            : base(input, output)
        {
            if (Nw4rSoundSetSeqFileType.Smf == input.XmlData.FileType && null == input.XmlIntermediateFile)
            {
                throw new Nw4rFileFormatInternalException("sequence intermediate file is not created.");
            }
        }

        #region ** プロパティ

        protected override bool ShouldConvert
        {
            get
            {
                try
                {
                    return Output.Dirty;
                }
                catch (DependencyFileNotFoundException exception)
                {
                    throw new Nw4rFileNotFoundException(exception.FileName, Input.Label);
                }
            }
        }

        private string InputFilePath
        {
            get { return (null != Input.XmlIntermediateFile) ? Input.XmlIntermediateFile.FilePath : Input.XmlData.FullPath; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnInvoke()
        {
            // 出力ディレクトリの生成
            CreateDirectory(Path.GetDirectoryName(Output.File.FilePath));

            string commandArgs = string.Empty;

            if (!ForcedConvert)
            {
                commandArgs += ConvertOption_UPDATE;
            }

            // 出力ファイルパス
            commandArgs += ConvertOption_OUTPUT + "\"" + Output.File.FilePath + "\" ";

            // 入力ファイルパス
            commandArgs += "\"" + InputFilePath + "\"";

            // テキスト出力
            string converterFilePath = Path.Combine(Nw4rSpecialFolder.CommandLineTools, ConverterFileName);
            OutputLineInternal(new InformationLine(converterFilePath + " " + commandArgs));

            // コンバート処理（処理完了までブロッキング）
            if (!Util.StartProcess(converterFilePath, commandArgs, string.Empty, OnCommandOutput, OnCommandError, ProcessHandler))
            {
                throw new Nw4rCommandException(string.Empty);
            }
        }

        protected override void OnClean()
        {
            try
            {
                File.Delete(Output.File.FilePath);
            }
            catch (Exception)
            {
            }
        }

        private void OnCommandOutput(string message)
        {
            OutputLineInternal(new InformationLine(message));
        }

        private void OnCommandError(string message)
        {
            OutputLineInternal(new ErrorLine(message, Input));
            //throw new Nw4rFileFormatException( message, Input.Label );
        }

        #endregion
    }
}
