﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Text;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat
{

    public enum LexerToken
    {
        // Character
        Return = '\n',
        Plus = '+', Minus = '-',
        Mul = '*', Div = '/',
        Colon = ':', Comma = ',', Period = '.',
        LeftParen = '(', RightParen = ')',
        LeftBracket = '[', RightBracket = ']',
        LeftBrace = '{', RightBrace = '}',
        Eq = '=',
        Or = '|', And = '&',
        AtMark = '@',

        // State
        End = 256,

        // Parameters
        Symbol, Number, String,

        // 2 Charactor Operators
        LessThan, LessEqual,
        GreaterThan, GreaterEqual,
        Equal,
        LeftShift, RightShift,
    }

    public class LexerState
    {
        public string LastLine;
        public string FilePath;
        public int LineNumber;
    }

    public class LexerException : SoundException
    {
        LexerState lexerState;

        public LexerException(string s, LexerState lexerState) : base(s)
        {
            this.lexerState = lexerState;
        }
        public LexerException(string s, Lexer lexer) : base(s)
        {
            this.lexerState = lexer.State;
        }
        public override string Message
        {
            get
            {
                return
                    lexerState.FilePath + ":" + lexerState.LineNumber + ": " + base.Message + "\n" +
                    lexerState.LastLine
                    ;
            }
        }
    }

    public class Lexer : IDisposable
    {
        Stack fileStack = new Stack();
        Hashtable macroTable = new Hashtable();
        Stack tokenStack = new Stack();
        ArrayList strSearchPaths = new ArrayList();
        Stack ifStateStack = new Stack();
        Hashtable dependFiles = new Hashtable();
        bool lastTokenAvailable = false;
        bool ignoreIfState = false;

        LexerToken lastToken = LexerToken.Return;
        string strLastToken;
        Int64 numberValue;
        string strValue = null;

        StringBuilder strBuilder = new StringBuilder();

        const int INCLUDE_DEPTH_MAX = 256;

        public Lexer(string strFilePath)
        {
            fileStack.Push(new FileInfo(strFilePath));
            dependFiles[Path.GetFullPath(strFilePath)] = 1;
        }

        public void Dispose()
        {
            foreach (FileInfo finfo in fileStack)
            {
                finfo.Close();
            }
        }

        public LexerState State
        {
            get
            {
                LexerState state = new LexerState();
                state.LastLine = LastLine;
                state.FilePath = CurrentFilePath;
                state.LineNumber = LineNumber;
                return state;
            }
        }

        public string CurrentFilePath
        {
            get
            {
                FileInfo finfo = (FileInfo)fileStack.Peek();
                return finfo.FilePath;
            }
        }
        public Int64 NumberValue
        {
            get { return numberValue; }
        }
        public string StringValue
        {
            get { return strValue; }
        }
        public LexerToken LastToken
        {
            get { return lastToken; }
        }
        public string LastTokenString
        {
            get { return strLastToken; }
        }
        public int LineNumber
        {
            get
            {
                FileInfo finfo = (FileInfo)fileStack.Peek();
                return finfo.LineNumber;
            }
        }
        public string LastLine
        {
            get
            {
                FileInfo finfo = (FileInfo)fileStack.Peek();
                return finfo.LastLine;
            }
        }
        public ICollection DependFiles
        {
            get { return dependFiles.Keys; }
        }

        public void AppendSearchPath(string path)
        {
            string absPath = Path.Combine(Directory.GetCurrentDirectory(), path);
            strSearchPaths.Add(absPath);
        }

        bool IsXDigit(char c)
        {
            if (Char.IsDigit(c)) return true;

            switch (c)
            {
                case 'a':
                case 'A':
                case 'b':
                case 'B':
                case 'c':
                case 'C':
                case 'd':
                case 'D':
                case 'e':
                case 'E':
                case 'f':
                case 'F':
                    return true;
            }

            return false;
        }

        int DigitParse(char c)
        {
            switch (c)
            {
                case '0': return 0;
                case '1': return 1;
                case '2': return 2;
                case '3': return 3;
                case '4': return 4;
                case '5': return 5;
                case '6': return 6;
                case '7': return 7;
                case '8': return 8;
                case '9': return 9;
                case 'a': case 'A': return 10;
                case 'b': case 'B': return 11;
                case 'c': case 'C': return 12;
                case 'd': case 'D': return 13;
                case 'e': case 'E': return 14;
                case 'f': case 'F': return 15;
            }

            return 0;
        }

        char Escape(char c)
        {
            switch (c)
            {
                case '0': return '\0';
                case 'a': return '\a';
                case 'b': return '\b';
                case 't': return '\t';
                case 'n': return '\n';
                case 'v': return '\v';
                case 'f': return '\f';
                case 'r': return '\r';
            }

            return c;
        }

        bool GetChar(out char ch)
        {
            if (fileStack.Count == 0)
            {
                ch = ' ';
                return false;
            }

            FileInfo finfo = (FileInfo)fileStack.Peek();
            if (finfo.IsEnd)
            {
                finfo.Close();
                fileStack.Pop();
                if (fileStack.Count == 0)
                {
                    if (ifStateStack.Count > 0)
                    {
                        throw new LexerException("Mismatched #endif", this);
                    }
                    ch = ' ';
                    return false;
                }
                finfo = (FileInfo)fileStack.Peek();
            }
            if (finfo.GetChar(out ch)) return true;

            // End of file
            ch = '\n';
            return true;
        }

        bool GetChar_SkipSpace(out char ch)
        {
            do
            {
                if (!GetChar(out ch)) return false;

                // コメント
                if (ch == ';')
                {
                    do
                    {
                        if (!GetChar(out ch)) return false;
                    } while (ch != '\n');
                }
            } while (ch != '\n' && Char.IsWhiteSpace(ch));

            return true;
        }

        bool CheckIfState()
        {
            if (ignoreIfState) return true;

            foreach (IfState ifState in ifStateStack)
            {
                if (!ifState.Current) return false;
            }

            return true;
        }

        string ParseSymbol(char startChar)
        {
            strBuilder.Length = 0;
            strBuilder.Append(startChar);

            char ch;
            while (GetChar(out ch))
            {
                if (!Char.IsLetterOrDigit(ch) && ch != '_')
                {
                    UnGet();
                    break;
                }
                strBuilder.Append(ch);
            }

            return strBuilder.ToString();
        }

        string ParseString(char endChar)
        {
            char ch;

            strBuilder.Length = 0;

            while (GetChar(out ch))
            {
                if (ch == '\\')
                { // エスケープ
                    if (!GetChar(out ch))
                    {
                        throw new LexerException("Syntax error '\"'", this);
                    }
                    ch = Escape(ch);
                }
                else
                {
                    if (ch == endChar)
                    {
                        return strBuilder.ToString();
                    }
                    if (ch == '\n')
                    {
                        throw new LexerException("Newline in constant", this);
                    }
                }
                strBuilder.Append(ch);
            }

            throw new LexerException("Syntax error '\"'", this);
        }

        LexerToken ParseToken()
        {
            char ch;

            if (lastTokenAvailable)
            {
                lastTokenAvailable = false;
                return lastToken;
            }

            if (tokenStack.Count != 0)
            {
                TokenInfo info = (TokenInfo)tokenStack.Pop();

                lastToken = info.token;
                strLastToken = info.tokenString;
                strValue = info.strValue;
                numberValue = info.numberValue;
                return lastToken;
            }

            if (!GetChar_SkipSpace(out ch))
            {
                return LexerToken.End;
            }

            strLastToken = ch.ToString();
            numberValue = 0;
            strValue = null;

            if (ch == '#')
            {
                if (lastToken != LexerToken.Return)
                {
                    throw new LexerException("The # sign of a preprocessor directive must be the first charactor on a line", this);
                }
                return ParseDirective();
            }

            if (ch == '\n')
            {
                return LexerToken.Return;
            }

            if (!CheckIfState())
            {
                // skip line
                while (GetChar_SkipSpace(out ch) && ch != '\n')
                {
                }
                if (ch == '\n') return LexerToken.Return;
                else return LexerToken.End;
            }

            switch (ch)
            {

                case '\\':
                    throw new LexerException("Illegal escape sequence", this);

                case '-':
                case '+':
                case '*':
                case '/':
                case ':':
                case ',':
                case '.':
                case '(':
                case ')':
                case '{':
                case '}':
                case '[':
                case ']':
                case '|':
                case '&':
                case '@':
                    return (LexerToken)ch;

                case '=':
                    if (!GetChar(out ch))
                    {
                        return LexerToken.Eq;
                    }
                    switch (ch)
                    {
                        case '=':
                            return LexerToken.Equal;

                        default:
                            UnGet();
                            return LexerToken.Eq;
                    }

                case '<':
                    if (!GetChar(out ch))
                    {
                        return LexerToken.LessThan;
                    }
                    switch (ch)
                    {
                        case '=':
                            return LexerToken.LessEqual;
                        case '<':
                            return LexerToken.LeftShift;

                        default:
                            UnGet();
                            return LexerToken.LessThan;
                    }

                case '>':
                    if (!GetChar(out ch))
                    {
                        return LexerToken.GreaterThan;
                    }
                    switch (ch)
                    {
                        case '=':
                            return LexerToken.GreaterEqual;
                        case '>':
                            return LexerToken.RightShift;

                        default:
                            UnGet();
                            return LexerToken.GreaterThan;
                    }

                case '"': // 文字列
                    strBuilder.Length = 0;
                    strValue = ParseString('\"');
                    return LexerToken.String;

                case '\'': // 文字定数
                    if (!GetChar(out ch))
                    {
                        throw new LexerException("Syntax error '\''", this);
                    }
                    if (ch == '\'')
                    {
                        throw new LexerException("Empty character constant", this);
                    }
                    if (ch == '\\')
                    { // エスケープ
                        if (!GetChar(out ch))
                        {
                            throw new LexerException("Syntax error '\''", this);
                        }
                        ch = Escape(ch);
                    }
                    numberValue = ch;
                    if (!GetChar(out ch))
                    {
                        throw new LexerException("Syntax error '\''", this);
                    }
                    if (ch != '\'')
                    {
                        throw new LexerException("Too many characters in constant", this);
                    }
                    return LexerToken.Number;

                case '0': // 数字
                    if (!GetChar(out ch))
                    {
                        numberValue = 0;
                        return LexerToken.Number;
                    }
                    if (ch == 'x' || ch == 'X')
                    {
                        // 0x 16進数モード
                        if (!GetChar(out ch))
                        {
                            throw new LexerException("Hexadecimal constants are illegal.", this);
                        }
                        if (!IsXDigit(ch))
                        {
                            throw new LexerException("Hexadecimal constants are illegal.", this);
                        }
                        numberValue = DigitParse(ch);
                        while (GetChar(out ch))
                        {
                            if (!IsXDigit(ch))
                            {
                                UnGet();
                                break;
                            }
                            if ((numberValue >> 60) != 0)
                            {
                                throw new LexerException("Constant too big", this);
                            }
                            numberValue <<= 4;
                            numberValue += DigitParse(ch);
                        }
                        return LexerToken.Number;
                    }
                    if (ch == 'b' || ch == 'B')
                    {
                        // 0b 2進数モード
                        if (!GetChar(out ch))
                        {
                            throw new LexerException("Binary number constants are illegal.", this);
                        }
                        if (ch != '0' && ch != '1')
                        {
                            throw new LexerException("Binary number constants are illegal.", this);
                        }
                        numberValue = DigitParse(ch);
                        while (GetChar(out ch))
                        {
                            if (ch != '0' && ch != '1')
                            {
                                UnGet();
                                break;
                            }
                            if ((numberValue >> 60) != 0)
                            {
                                throw new LexerException("Constant too big", this);
                            }
                            numberValue <<= 1;
                            numberValue += DigitParse(ch);
                        }
                        return LexerToken.Number;
                    }
                    UnGet();
                    ch = '0';
                    goto case '1';

                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                case '8':
                case '9':
                    numberValue = DigitParse(ch);
                    while (GetChar(out ch))
                    {
                        if (!Char.IsDigit(ch))
                        {
                            UnGet();
                            break;
                        }
                        if ((numberValue >> 60) != 0)
                        {
                            throw new LexerException("Constant too big", this);
                        }
                        numberValue *= 10;
                        numberValue += DigitParse(ch);
                    }
                    return LexerToken.Number;

                default:
                    if (Char.IsLetter(ch) || ch == '_')
                    { // シンボル
                        strValue = ParseSymbol(ch);

                        if (!macroTable.Contains(strValue))
                        {
                            return LexerToken.Symbol;
                        }

                        // マクロ置換
                        Macro macro = (Macro)macroTable[strValue];
                        macro.AppendToTokenStack(tokenStack);

                        return ParseToken();
                    }

                    // エラー
                    throw new LexerException("Unknown charactor \'" + ch + "\'", this);
            }
        }

        public LexerToken ReadToken()
        {
            lastToken = ParseToken();
            return lastToken;
        }

        public LexerToken PeekToken()
        {
            LexerToken token = ReadToken();

            lastTokenAvailable = true;

            return token;
        }

        // c4 = 60
        public static bool ParseKeyString(string str, out int key)
        {
            key = 0;
            int[] key_offset = new int[7] {
                9, 11, 0, 2, 4, 5, 7
            };
            CharEnumerator cp = str.GetEnumerator();
            if (!cp.MoveNext()) return false;
            char c = cp.Current;
            if ('a' <= c && c <= 'g')
            {
                key = key_offset[c - 'a'];
            }
            else if ('A' <= c && c <= 'G')
            {
                key = key_offset[c - 'A'];
            }
            else
            {
                return false;
            }

            if (!cp.MoveNext()) return false;
            c = cp.Current;
            switch (c)
            {
                case 'n':
                case 'N':
                    if (!cp.MoveNext()) return false;
                    c = cp.Current;
                    break;
                case 's':
                case 'S':
                    key++;
                    if (!cp.MoveNext()) return false;
                    c = cp.Current;
                    break;
                case 'f':
                case 'b':
                case 'F':
                case 'B':
                    key--;
                    if (!cp.MoveNext()) return false;
                    c = cp.Current;
                    break;
            }

            bool minus_flag = false;
            if (c == 'm' || c == 'M')
            {
                minus_flag = true;
                if (!cp.MoveNext()) return false;
                c = cp.Current;
            }

            if (!Char.IsDigit(c)) return false;

            int octave = c - '0';
            while (cp.MoveNext())
            {
                c = cp.Current;
                if (!Char.IsDigit(c)) return false;
                octave *= 10;
                octave += c - '0';
            }

            if (minus_flag) octave = -octave;

            key += (octave + 1) * 12;
            return true;
        }

        public void ThrowUnexpectedTokenException(string expectedToken)
        {
            string errToken = (strLastToken != "\n") ? "\"" + strLastToken + "\"" : "new line";
            throw new LexerException(
                "Expected " + expectedToken + ", but found " + errToken,
                this
            );
        }

        public void VerifyToken(LexerToken token, string expectedToken)
        {
            if (token != lastToken)
            {
                ThrowUnexpectedTokenException(expectedToken);
            }
        }
        public void VerifyMinMaxValue(string name, long value, long min, long max)
        {
            if (value < min || max < value)
            {
                throw new LexerException(
                    name + " is out of range, " + min + " <= " + value + " <= " + max +
                    " not satisfied.",
                    this
                );
            }
        }

        void UnGet()
        {
            FileInfo finfo = (FileInfo)fileStack.Peek();
            finfo.UnGet();
        }

        LexerToken ParseDirective()
        {
            char ch;
            string fullPath = null;

            if (!GetChar_SkipSpace(out ch))
            {
                // # のみ
                return LexerToken.End;
            }

            if (ch == '\n')
            {
                // # のみ
                return LexerToken.Return;
            }

            // ディレクティブ名取得
            if (!Char.IsLetter(ch) && ch != '_')
            {
                ThrowUnexpectedTokenException("preprocessor directive");
            }
            string directive = ParseSymbol(ch);

            // ディレクティブ毎の処理
            if (directive == "if")
            {
                if (!GetChar(out ch) || !Char.IsWhiteSpace(ch) || ch == '\n')
                {
                    throw new LexerException("Expected space after a #if", this);
                }
                ExprParser exprParser = new ExprParser();

                ignoreIfState = true;
                bool flag = Convert.ToBoolean(exprParser.Parse(this));
                ignoreIfState = false;

                ifStateStack.Push(new IfState(flag));
            }
            else if (directive == "ifdef" || directive == "ifndef")
            {
                if (!GetChar(out ch) || !Char.IsWhiteSpace(ch) || ch == '\n')
                {
                    throw new LexerException("Expected space after a #" + directive, this);
                }
                if (!GetChar_SkipSpace(out ch) || ch == '\n')
                {
                    throw new LexerException("No identifier appears after a #" + directive, this);
                }
                if (!Char.IsLetter(ch) && ch != '_')
                {
                    ThrowUnexpectedTokenException("identifier");
                }
                string macroName = ParseSymbol(ch);
                bool flag = macroTable.Contains(macroName);
                if (directive == "ifndef") flag = !flag;
                ifStateStack.Push(new IfState(flag));
            }
            else if (directive == "elif")
            {
                if (ifStateStack.Count == 0)
                {
                    throw new LexerException("Unexpected #elif", this);
                }
                IfState ifState = (IfState)ifStateStack.Peek();
                if (ifState.ElseFlag)
                {
                    throw new LexerException("Unexpected #elif", this);
                }
                ifState.ElseFlag = true;

                if (!GetChar(out ch) || !Char.IsWhiteSpace(ch) || ch == '\n')
                {
                    throw new LexerException("Expected space after a #elif", this);
                }

                ExprParser exprParser = new ExprParser();
                ignoreIfState = true;
                bool flag = Convert.ToBoolean(exprParser.Parse(this));
                ignoreIfState = false;
                ifStateStack.Push(new IfState(flag, true));
            }
            else if (directive == "else")
            {
                if (ifStateStack.Count == 0)
                {
                    throw new LexerException("Unexpected #else", this);
                }
                IfState ifState = (IfState)ifStateStack.Peek();
                if (ifState.ElseFlag)
                {
                    throw new LexerException("Unexpected #else", this);
                }
                ifState.ElseFlag = true;
            }
            else if (directive == "endif")
            {
                bool flag;
                do
                {
                    if (ifStateStack.Count == 0)
                    {
                        throw new LexerException("Unexpected #endif", this);
                    }
                    IfState ifState = (IfState)ifStateStack.Pop();
                    flag = ifState.ElifFlag;
                } while (flag);
            }
            else if (!CheckIfState())
            {
                // skip line
                while (GetChar_SkipSpace(out ch) && ch != '\n')
                {
                }
                if (ch == '\n') return LexerToken.Return;
                else return LexerToken.End;
            }
            else if (directive == "define")
            {
                // #define
                if (!GetChar(out ch) || !Char.IsWhiteSpace(ch) || ch == '\n')
                {
                    throw new LexerException("Expected space after a #define", this);
                }

                if (!GetChar_SkipSpace(out ch) || ch == '\n')
                {
                    throw new LexerException("No identifier appears after a #define", this);
                }
                if (!Char.IsLetter(ch) && ch != '_')
                {
                    ThrowUnexpectedTokenException("identifier");
                }

                string macroName = ParseSymbol(ch);

                Macro macro = new Macro();

                LexerToken token;
                while ((token = PeekToken()) != LexerToken.End)
                {
                    if (token == LexerToken.Return) break;

                    token = ReadToken();
                    if (token == LexerToken.Symbol)
                    {
                        // 再帰置換チェック
                        if (macroTable.Contains(StringValue))
                        {
                            Macro m = (Macro)macroTable[StringValue];
                            macro.AppendMacro(m);
                            continue;
                        }
                    }

                    TokenInfo tinfo = new TokenInfo();
                    tinfo.token = token;
                    tinfo.numberValue = NumberValue;
                    tinfo.strValue = StringValue;
                    tinfo.tokenString = LastTokenString;

                    macro.AppendToken(tinfo);
                }

                if (macroTable.Contains(macroName))
                {
                    // ２重定義

                    Macro m = (Macro)macroTable[macroName];

                    if (macro != m)
                    {
                        throw new LexerException("Multiple defined macro \"" + macroName + "\"", this);
                    }
                }

                macroTable[macroName] = macro;
            }
            else if (directive == "undef")
            {
                // #undef
                if (!GetChar(out ch) || !Char.IsWhiteSpace(ch) || ch == '\n')
                {
                    throw new LexerException("Expected space after a #undef", this);
                }
                if (!GetChar_SkipSpace(out ch) || ch == '\n')
                {
                    throw new LexerException("No identifier appears after a #undef", this);
                }
                if (!Char.IsLetter(ch) && ch != '_')
                {
                    ThrowUnexpectedTokenException("identifier");
                }
                string macroName = ParseSymbol(ch);
                if (macroTable.Contains(macroName))
                {
                    macroTable.Remove(macroName);
                }
            }
            else if (directive == "include")
            {
                // #include
                LexerToken t = ReadToken();

                if (t != LexerToken.String && t != LexerToken.LessThan)
                {
                    ThrowUnexpectedTokenException("filename");
                }
                if (fileStack.Count > INCLUDE_DEPTH_MAX)
                {
                    throw new LexerException("#include nesting overflow", this);
                }

                bool bSystemPath = (t == LexerToken.LessThan);

                // ファイルパスの取得
                string filePath = bSystemPath ? ParseString('>') : StringValue;

                if (Path.IsPathRooted(filePath))
                {
                    // 絶対パス
                    fullPath = filePath;
                }
                else if (bSystemPath)
                {
                    // サーチパスによるファイル探索
                    foreach (string searchPath in strSearchPaths)
                    {
                        string path = Path.Combine(searchPath, filePath);
                        if (File.Exists(path))
                        {
                            fullPath = path;
                            break;
                        }
                    }

                    if (fullPath == null)
                    {
                        throw new LexerException("Cannot find file \"" + filePath + "\"", this);
                    }
                }
                else
                {
                    // 相対パス
                    FileInfo finfo = (FileInfo)fileStack.Peek();
                    string dirPath = Path.GetDirectoryName(finfo.FilePath);
                    fullPath = Path.Combine(dirPath, filePath);
                }

            }
            else
            {
                throw new LexerException("Invalid preprocessor command string \"" + directive + "\"", this);
            }

            LexerToken tkn = ParseToken();
            if (tkn != LexerToken.Return && tkn != LexerToken.End)
            {
                throw new LexerException("Unexpected tokens following preprocessor directive - expected a newline", this);
            }

            if (fullPath != null)
            {
                fileStack.Push(new FileInfo(fullPath));
                dependFiles[Path.GetFullPath(fullPath)] = 1;
            }

            return tkn;
        }

        class TokenInfo
        {
            public LexerToken token = LexerToken.End;
            public Int64 numberValue = 0;
            public string strValue = null;

            public string tokenString = String.Empty;

            public override bool Equals(Object obj)
            {
                return obj is TokenInfo && this == (TokenInfo)obj;
            }
            public override int GetHashCode()
            {
                return token.GetHashCode() ^ numberValue.GetHashCode() ^ strValue.GetHashCode() ^ tokenString.GetHashCode();
            }
            public static bool operator ==(TokenInfo lhs, TokenInfo rhs)
            {
                if (lhs.token != rhs.token) return false;

                switch (lhs.token)
                {
                    case LexerToken.Symbol:
                    case LexerToken.String:
                        if (lhs.strValue != rhs.strValue) return false;
                        break;
                    case LexerToken.Number:
                        if (lhs.numberValue != rhs.numberValue) return false;
                        break;
                    default:
                        break;
                }

                return true;
            }
            public static bool operator !=(TokenInfo lhs, TokenInfo rhs)
            {
                return !(lhs == rhs);
            }
        }

        class Macro
        {
            ArrayList tokenList = new ArrayList();

            public Macro()
            {
            }
            public void AppendToken(TokenInfo token)
            {
                tokenList.Add(token);
            }
            public void AppendMacro(Macro m)
            {
                foreach (TokenInfo tinfo in m.tokenList)
                {
                    tokenList.Add(tinfo);
                }
            }
            public void AppendToTokenStack(Stack stack)
            {
                for (int i = tokenList.Count - 1; i >= 0; i--)
                {
                    stack.Push((TokenInfo)tokenList[i]);
                }
            }
            public override bool Equals(Object obj)
            {
                return obj is Macro && this == (Macro)obj;
            }
            public override int GetHashCode()
            {
                return tokenList.GetHashCode();
            }
            public static bool operator ==(Macro lhs, Macro rhs)
            {
                if (lhs.tokenList.Count != rhs.tokenList.Count) return false;

                IEnumerator e1 = lhs.tokenList.GetEnumerator();
                IEnumerator e2 = rhs.tokenList.GetEnumerator();
                while (e1.MoveNext() && e2.MoveNext())
                {
                    TokenInfo t1 = (TokenInfo)e1.Current;
                    TokenInfo t2 = (TokenInfo)e2.Current;
                    if (t1 != t2) return false;
                }

                return true;
            }
            public static bool operator !=(Macro lhs, Macro rhs)
            {
                return !(lhs == rhs);
            }
        }

        class FileInfo
        {
            StreamReader strm;
            string filePath;
            string strLineBuffer = String.Empty;
            int linePos;
            int lineNumber;
            bool bEnd = false;

            public FileInfo(string filePath)
            {
                this.filePath = filePath;
                strm = new StreamReader(filePath);

                lineNumber = 0;
                linePos = 0;
            }
            public void Close()
            {
                strm.Close();
            }

            public bool IsEnd
            {
                get { return bEnd; }
            }
            public string FilePath
            {
                get { return filePath; }
            }
            public int LineNumber
            {
                get { return lineNumber; }
            }
            public string LastLine
            {
                get { return strLineBuffer; }
            }
            public bool GetChar(out char ch)
            {
                if (linePos >= strLineBuffer.Length - 1)
                {
                    strLineBuffer = strm.ReadLine();
                    if (strLineBuffer == null)
                    {
                        ch = ' ';
                        bEnd = true;
                        return false;
                    }

                    strLineBuffer += '\n';

                    linePos = 0;
                    lineNumber++;
                }
                else
                {
                    linePos++;
                }

                ch = strLineBuffer[linePos];

                return true;
            }

            public void UnGet()
            {
                linePos--;
            }
        }

        class IfState
        {
            bool flag;
            bool elifFlag;
            bool elseFlag;
            public IfState(bool flag)
            {
                this.flag = flag;
                this.elifFlag = false;
                this.elseFlag = false;
            }
            public IfState(bool flag, bool elifFlag)
            {
                this.flag = flag;
                this.elifFlag = elifFlag;
                this.elseFlag = false;
            }
            public bool Current
            {
                get
                {
                    if (!elseFlag) return flag;
                    return !flag;
                }
            }
            public bool ElseFlag
            {
                set { elseFlag = value; }
                get { return elseFlag; }
            }
            public bool ElifFlag
            {
                get { return elifFlag; }
            }
        }

    }

}

