﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;
    using MsgRsrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// Cafe サウンドプロジェクトをインポートします。
    /// </summary>
    public class SoundSetDocumentImporterCafe : IDocumentImporter
    {
        private readonly DocumentService documentService = null;
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits = null;
        private string includePath = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="documentService">ドキュメントサービスを指定します。</param>
        /// <param name="intermediateOutputTraits">中間出力特性を指定します。</param>
        /// <param name="includePath">インクルードパスを指定します。</param>
        public SoundSetDocumentImporterCafe(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits,
            string includePath)
        {
            Ensure.Argument.NotNull(documentService);
            Ensure.Argument.NotNull(intermediateOutputTraits);

            this.documentService = documentService;
            this.intermediateOutputTraits = intermediateOutputTraits;
            this.includePath = includePath;
        }

        public SoundSetDocumentImporterCafe(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
            : this(documentService, intermediateOutputTraits, string.Empty)
        {
        }

        /// <summary>
        /// インポート元のドキュメントタイプ名を取得します。
        /// </summary>
        public string SourceDocumentTypeName
        {
            get { return Platforms.Cafe.SoundSetDocument; }
        }

        /// <summary>
        /// インポート後のドキュメントタイプ名を取得します。
        /// </summary>
        public string DestinationDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundSetDocumentTypeName; }
        }

        /// <summary>
        /// ドキュメントをインポートします。
        /// </summary>
        /// <param name="document">インポートするドキュメントを指定します。</param>
        /// <returns>インポートしたドキュメント参照の配列を返します。</returns>
        public DocumentReference[] Import(Document document)
        {
            Assertion.Argument.NotNull(document);
            Ensure.Argument.True(document is SoundSetDocument);
            Ensure.Argument.True(document.Resource is FileResource);

            System.Reflection.AssemblyProductAttribute asmprd =
                (System.Reflection.AssemblyProductAttribute)
                Attribute.GetCustomAttribute(
                                             System.Reflection.Assembly.GetEntryAssembly(),
                                             typeof(System.Reflection.AssemblyProductAttribute)
                                             );
#if false
            DialogResult result =
                MessageBox.Show(MsgRsrc.MessageResourceCafe.Message_ImportNotesCafe,
                                asmprd.Product,
                                MessageBoxButtons.OKCancel,
                                MessageBoxIcon.Warning
                                );
            if (result == DialogResult.Cancel)
            {
                return null;
            }
#endif
            string typeName = this.intermediateOutputTraits.SoundSetDocumentTypeName;
            List<DocumentReference> results = new List<DocumentReference>();
            try
            {
                DocumentReference[] soundSetDocumentRefs = this.ImportSoundSetDocument(document);
                results.AddRange(soundSetDocumentRefs);
            }
            catch
            {
                foreach (DocumentReference soundSetDocumentRef in results)
                {
                    soundSetDocumentRef.Close();
                }
                throw;
            }

            return results.ToArray();
        }

        /// <summary>
        /// サウンドセットドキュメントをインポートします。
        /// </summary>
        /// <param name="filePath">サウンドセットファイルパス。</param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        public DocumentReference[] Import(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFileRead, filePath);
            try
            {
                using (DocumentReference documentRef =
                       this.documentService.OpenDocument(new FileResource(filePath)))
                {
                    if (documentRef == null)
                    {
                        throw new FileNotSupportedException(filePath);
                    }

                    if (!(documentRef.Document is SoundSetDocument))
                    {
                        throw new FileNotSupportedException(filePath);
                    }

                    return this.ImportSoundSetDocument(documentRef.Document);
                }
            }
            catch (Exception e)
            {
                e.Data["ImportFilePath"] = filePath;
                throw e;
            }
        }

        private DocumentReference[] ImportSoundSetDocument(Document document)
        {
            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            try
            {
                string filePath = document.Resource.Key;
                ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFile, filePath);
                // サウンドセットドキュメント
                DocumentReference importedSoundSetDocumentRef =
                    this.documentService.ConvertDocument(document, this.intermediateOutputTraits.SoundSetDocumentTypeName);

                if (importedSoundSetDocumentRef == null)
                {
                    throw new FileNotSupportedException(filePath);
                }
                else if (!(importedSoundSetDocumentRef.Document is SoundSetDocument))
                {
                    importedSoundSetDocumentRef.Close();
                    throw new FileNotSupportedException(filePath);
                }

                importedDocumentRefs.Add(importedSoundSetDocumentRef);

                // バンクドキュメント
                foreach (DocumentReference importedBankDocumentRef in
                         this.ImportBankDocuments(document as SoundSetDocument))
                {
                    importedDocumentRefs.Add(importedBankDocumentRef);
                }

                // テキストシーケンスサウンドドキュメント
                foreach (DocumentReference importedTextSequenceSoundDocumentRef in
                         this.ImportTextSequenceSoundDocuments(document as SoundSetDocument))
                {
                    importedDocumentRefs.Add(importedTextSequenceSoundDocumentRef);
                }
            }
            catch
            {
                foreach (DocumentReference importedDocumentRef in importedDocumentRefs)
                {
                    importedDocumentRef.Close();
                }
            }

            return importedDocumentRefs.ToArray();

        }

        ///
        private T[] GetSoundSetItems<P, T>( SoundSet soundSet) where P : SoundSetItem where T : SoundSetItem
        {
            foreach( SoundSetItem itemPack in soundSet.Children )
            {
                if (itemPack is P)
                {
                    List<T> items = new List<T>();
                    foreach( T item in itemPack.Children )
                    {
                        items.Add( item);
                    }
                    return items.ToArray();
                }
            }
            return new T[0];
        }

        /// <summary>
        /// バンクドキュメントをインポートします。
        /// </summary>
        /// <param name="legacySoundSetDocument">
        /// レガシーサウンドセットドキュメントを指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        private DocumentReference[] ImportBankDocuments(
            SoundSetDocument legacySoundSetDocument)
        {
            Assertion.Argument.NotNull(legacySoundSetDocument);

            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            string soundSetDirectoryPath =
                Path.GetDirectoryName( legacySoundSetDocument.Resource.Key);

            try
            {
                BankDocumentImporterCafe importer =
                    new BankDocumentImporterCafe(this.documentService,
                                                this.intermediateOutputTraits);

                SoundSetBankBase[] banks = GetSoundSetItems<SoundSetBankPack, SoundSetBankBase>
                    ( legacySoundSetDocument.SoundSet);

                foreach( SoundSetBankBase legacySoundSetBank in banks )
                {
                    string path = Path.Combine(
                        soundSetDirectoryPath,
                        legacySoundSetBank.FilePath
                        ).GetFullPath();

                    DocumentReference[] docRefs = importer.Import(path);
                    importedDocumentRefs.AddRange(docRefs);
                }
            }
            catch
            {
                foreach (DocumentReference documentRef in importedDocumentRefs)
                {
                    documentRef.Close();
                }

                throw;
            }

            return importedDocumentRefs.ToArray();
        }

        /// <summary>
        /// テキストシーケンスサウンドドキュメントをインポートします。
        /// </summary>
        /// <param name="legacySoundSetDocument">
        /// レガシーサウンドセットドキュメントを指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        private DocumentReference[] ImportTextSequenceSoundDocuments( SoundSetDocument legacySoundSetDocument)
        {
            Assertion.Argument.NotNull(legacySoundSetDocument);

            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            string soundSetDirectoryPath =
                Path.GetDirectoryName( legacySoundSetDocument.Resource.Key);
            HashSet<string> importedFilePaths = new HashSet<string>();

            try
            {
                // シーケンスサウンド
                {
                    SequenceSoundBase[] sequenceSounds =
                        GetSoundSetItems<SequenceSoundPack, SequenceSoundBase>
                        ( legacySoundSetDocument.SoundSet);

                    foreach( SequenceSoundBase sequenceSound in sequenceSounds )
                    {
                        if( sequenceSound.FileType != SequenceSoundFileType.Text )
                        {
                            continue;
                        }

                        importedDocumentRefs.AddRange
                            ( this.ImportTextSequenceSoundDocument
                              (Path.Combine(soundSetDirectoryPath, sequenceSound.FilePath).GetFullPath(),
                                importedFilePaths));
                    }
                }

                // シーケンスサウンドセット内のシーケンスサウンド
                SequenceSoundSetBase[] sequenceSoundSets =
                    GetSoundSetItems<SequenceSoundSetPack, SequenceSoundSetBase>
                    ( legacySoundSetDocument.SoundSet);

                foreach( SequenceSoundSetBase sequenceSoundSet in sequenceSoundSets )
                {
                    foreach( SequenceSoundBase sequenceSound in sequenceSoundSet.Children )
                    {
                        if( sequenceSound.FileType != SequenceSoundFileType.Text )
                        {
                            continue;
                        }

                        importedDocumentRefs.AddRange
                            ( this.ImportTextSequenceSoundDocument
                              ( Path.Combine( soundSetDirectoryPath, sequenceSound.FilePath).GetFullPath(),
                                importedFilePaths));
                    }
                }
            }
            catch
            {
                foreach (DocumentReference documentRef in importedDocumentRefs)
                {
                    documentRef.Close();
                }

                throw;
            }

            return importedDocumentRefs.ToArray();
        }

        /// <summary>
        /// テキストシーケンスサウンドドキュメントをインポートします。
        /// </summary>
        /// <param name="filePaths">テキストシーケンスサウンドファイルパスを指定します。</param>
        /// <param name="importedFilePaths">
        /// インポート済みのファイルパスコレクションを指定します。
        /// </param>
        /// <returns>インポートしたテキストシーケンスサウンドドキュメントへの参照を返します。</returns>
        private DocumentReference[] ImportTextSequenceSoundDocument(
                                                                string filePath,
                                                                HashSet<string> importedFilePaths
                                                                )
        {
            Ensure.Argument.StringNotEmpty(filePath);
            Ensure.Argument.NotNull(importedFilePaths);

            if (importedFilePaths.Contains(filePath))
            {
                return new DocumentReference[0];
            }

            if (File.Exists(filePath) == false)
            {
                ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportWarning_FileNotFound, filePath);
                return new DocumentReference[0];
            }

            TextSequenceSoundDocument document = new TextSequenceSoundDocument()
            {
                Resource = new FileResource((filePath)),
            };

            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFileRead, filePath);
            try
            {
                new TextSequenceSoundDocumentReader(Platforms.Any.TextSequenceSoundDocument).
                    Read(document.Resource, document);
            }
            catch (Exception e)
            {
                e.Data["ImportFilePath"] = filePath;
                throw e;
            }

            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFile, filePath);
            document.Text =
                document.Text.Replace(
                                     ".cseq", "." +
                                     this.intermediateOutputTraits.TextSequenceSoundFileExtension
                                     );
            document.Text =
                document.Text.Replace(
                                      ".cinl", "." +
                                      this.intermediateOutputTraits.BankIncludeFileExtension
                                      );
            document.Resource = new FileResource(
                Path.ChangeExtension(
                                     filePath,
                                     this.intermediateOutputTraits.TextSequenceSoundFileExtension
                                     )
                );

            List<DocumentReference> importedDocumentRefs = new List<DocumentReference>();

            importedDocumentRefs.Add(
                new DocumentReference(null)
                {
                    Document = document,
                });
            importedFilePaths.Add(filePath);

            SequenceSoundTextLexer lexer = new SequenceSoundTextLexer(filePath)
            {
                SkipNotFoundFiles = true,
            };

            if (string.IsNullOrEmpty(this.includePath) == false)
            {
                lexer.AppendSearchPath(this.includePath);
            }

            try
            {
                while (lexer.ReadToken() != SequenceSoundTextLexer.Token.End)
                {
                    // 依存ファイル収集のため空回し
                }
            }
            catch (Exception)
            {
                // cseq 依存ファイル収集が不完全な場合ここに来るが、
                // エラーにせず、インポートを続ける
            }

            foreach (string dependentFilePath in lexer.DependFiles)
            {
                if (dependentFilePath == filePath)
                {
                    continue;
                }

                if (dependentFilePath.EndsWith(".cseq"))
                {
                    importedDocumentRefs.AddRange(
                        this.ImportTextSequenceSoundDocument(dependentFilePath, importedFilePaths)
                        );
                }
            }

            return importedDocumentRefs.ToArray();
        }
    }
}
