﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.Foundation.IO;
using NintendoWare.SoundFoundation.Core.Threading;
using NintendoWare.SoundFoundation.Utilities;
using NintendoWare.SoundProjectUtility.Resources;
using System;
using System.IO;
using System.Threading;

namespace NintendoWare.SoundProjectUtility
{
    public class Program
    {
        public class Settings : LocalizableSettings
        {
            [CommandLineSubCommand(
                "list-files",
                Description = nameof(MessageResources.CommandLine_UsageMessage_ListFiles),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public PrintFilesStrategy.Settings ListFiles { get; set; }

            [CommandLineSubCommand(
                "update-project",
                Description = nameof(MessageResources.CommandLine_UsageMessage_UpdateProject),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public UpdateProjectStrategy.Settings UpdateProject { get; set; }
        }

        internal static int Main(string[] args)
        {
            EnvironmentUtility.RemoveMultipleEnvironmentVariables();
            Thread.CurrentThread.InitializeCurrentUICulture();

            bool result = Run(args);
            return result ? 0 : 1;
        }

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        private static bool Run(string[] args)
        {
            Settings settings;
            var resultWriter = new ResultWriter();

            try
            {
                var commandLineParserSettings = new CommandLineParserSettings()
                {
                    ApplicationDescription = MessageResources.CommandLine_ApplicationDescription,
                };

                if (new CommandLineParser(commandLineParserSettings).ParseArgs(args, out settings) == false)
                {
                    return true;
                }
            }
            catch (ArgumentException)
            {
                return false;
            }

            try
            {
                if (settings.ListFiles != null)
                {
                    var strategy = new PrintFilesStrategy(settings.ListFiles);
                    strategy.Execute(resultWriter);
                }
                else if (settings.UpdateProject != null)
                {
                    var strategy = new UpdateProjectStrategy(settings.UpdateProject);
                    strategy.Execute(resultWriter);
                }

                return true;
            }
            catch (FileNotFoundException e)
            {
                resultWriter.Error.WriteLine(
                    string.Format(
                        MessageResources.CommandLine_ErrorMessage_Error,
                        string.Format(MessageResources.CommandLine_ErrorMessage_NotFoundFile, e.FileName)));
            }
            catch (ApplicationException e)
            {
                resultWriter.Error.WriteLine(
                    string.Format(
                        MessageResources.CommandLine_ErrorMessage_Error,
                        e.Message));
            }
#if !DEBUG
            catch (Exception e)
            {
                resultWriter.Error.WriteLine(string.Format(MessageResources.CommandLine_ErrorMessage_Error, e.Message));
            }
#endif

            return false;
        }
    }
}
