﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Core.Resources;
using NintendoWare.SoundFoundation.Documents;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.FileFormats;
using NintendoWare.SoundMaker.Framework.Resources;
using System.Reflection;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    public static class OpenHelper
    {
        private const string OpenPreCommandFileName = "SoundMaker_preopen_command.bat";
        private const string ProjectSpecificOpenPreCommandExtension = ".preopen_command.bat";

        private static string ApplicationDirectory
        {
            get { return Path.GetDirectoryName(ApplicationBase.Instance.Traits.ApplicationAssembly.Location); }
        }

        private static string ProjectFilePath
        {
            get
            {
                // NOTE:
                // .fsst がオープンされる時点ではプロジェクトの読み込みはまだ終わっておらず、
                // ProjectService.ProjectFilePath は設定されていないため、
                // 別の方法でプロジェクトファイルのパスの取得を試みます。

                foreach (var document in ApplicationBase.Instance.DocumentService.Documents)
                {
                    if (document is SoundProjectDocument)
                    {
                        var filePath = document.Resource.Key;
                        if (File.Exists(filePath))
                        {
                            return filePath;
                        }
                    }
                }

                return string.Empty;
            }
        }

        /// <summary>
        /// ファイルを開く前にユーザー指定のコマンドを実行します。
        /// </summary>
        /// <param name="filePath">これから開こうとしているファイルのパスです。</param>
        public static void FileOpenPreCommand(string filePath)
        {
            string path = EnumeratePreOpenCommandPath(filePath)
                .Where(command => File.Exists(command))
                .FirstOrDefault();

            if (path == null)
            {
                return;
            }

            // NOTE: ローカライズ不要にするため、メッセージ文をコード内に埋め込みます。
            var commandName = Path.GetFileName(path);
            string errorMessage = $"'{commandName}' returned error.";
            string exceptionMessage = $"Executing '{commandName}' raised exception.";

            ExecuteCommand(path, filePath, errorMessage, exceptionMessage);
        }

        /// <summary>
        /// 実行する preopen コマンドの候補のパスを優先度の順に列挙します。
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        private static IEnumerable<string> EnumeratePreOpenCommandPath(string filePath)
        {
            // プロジェクトファイルを開くときは、まだプロジェクトディレクトリが決まっていないので、そのファイルのディレクトリにあるスクリプトを実行します。
            if (IsProjectFile(filePath))
            {
                yield return Path.ChangeExtension(filePath, ProjectSpecificOpenPreCommandExtension);
                yield return Path.Combine(Path.GetDirectoryName(filePath), OpenPreCommandFileName);
            }

            if (!string.IsNullOrEmpty(ProjectFilePath))
            {
                yield return Path.ChangeExtension(ProjectFilePath, ProjectSpecificOpenPreCommandExtension);
                yield return Path.Combine(Path.GetDirectoryName(ProjectFilePath), OpenPreCommandFileName);
            }

            if (!string.IsNullOrEmpty(ApplicationDirectory))
            {
                yield return Path.Combine(ApplicationDirectory, OpenPreCommandFileName);
            }
        }

        private static bool IsProjectFile(string filePath)
        {
            return Path.GetExtension(filePath).ToLower() == "." + ApplicationBase.Instance.Traits.IntermediateOutputTraits.SoundProjectFileExtension;
        }

        private static void ExecuteCommand(string commandPath, string filePath, string errorMessage, string exceptionMessage)
        {
            var projFilePath = IsProjectFile(filePath) ? filePath : ProjectFilePath;

            using (Process process = new Process())
            {
                var stringBuilder = new StringBuilder();
                stringBuilder.AppendLine("Opening file:");
                stringBuilder.AppendLine(filePath);
                stringBuilder.AppendLine();
                stringBuilder.AppendLine("Output:");

                process.StartInfo.FileName = commandPath;
                process.StartInfo.Arguments = $"\"{filePath}\" \"{projFilePath}\"";
                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardOutput = true;
                process.StartInfo.RedirectStandardError = true;
                process.OutputDataReceived += (sender, args) =>
                    {
                        if (string.IsNullOrWhiteSpace(args.Data) == false)
                        {
                            stringBuilder.AppendLine(args.Data);
                        }
                    };
                process.ErrorDataReceived += (sender, args) =>
                    {
                        if (string.IsNullOrWhiteSpace(args.Data) == false)
                        {
                            stringBuilder.AppendLine(args.Data);
                        }
                    };

                try
                {
                    process.Start();
                    process.BeginOutputReadLine();
                    process.BeginErrorReadLine();
                    process.WaitForExit();

                    if (process.ExitCode != 0)
                    {
                        stringBuilder.AppendLine();
                        stringBuilder.AppendLine($"ExitCode: {process.ExitCode}");

                        var messageBox = new TextDisplayMessageBox(
                            errorMessage,
                            stringBuilder.ToString(),
                            TextDisplayMessageBoxStyle.OKButton);
                        messageBox.ShowDialog();
                    }
                }
                catch (Exception e)
                {
                    var messageBox = new TextDisplayMessageBox(
                        exceptionMessage,
                        $"Opening file:\n{filePath}\n\nMessage:\n{e.Message}",
                        TextDisplayMessageBoxStyle.OKButton);
                    messageBox.ShowDialog();
                }
            }
        }
    }
}
