﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.ComponentModel;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// パラメータパネル
    /// </summary>
    public partial class ParameterPanel : DockingPageWithCommandTarget, IToolWindowPage, IKeyPreviewReceiver
    {
        public const string PageName = "Parameter";

        private UserControl currentPanel = null;
        private Document currentDocument = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ParameterPanel()
            : base(PageName)
        {
            InitializeComponent();
            InitializeCommandBindings();

            this.instrumentParameterPanel.Instrument = null;
            this.instrumentParameterPanel.OperationHistory = null;
            this.instrumentParameterPanel.VelocityRegion = null;

            // テキストとアイコンをロードする
            Text = MessageResource.ToolWindowName_ParameterPanel;
            Image = ImageResource.BitmapIconParameterWindow.
                MakeNewTransparent(UIServiceBase.TransparentColor);


            // 初期配置を設定する
            DockVisible = false;
            FloatingBounds =
                new Rectangle(-1, -1,
                               Width + (SystemInformation.FrameBorderSize.Width * 2),
                               (Height + SystemInformation.CaptionHeight +
                                 SystemInformation.FrameBorderSize.Height));

            //
            this.instrumentParameterPanel.OperationExecuted += OnOperationExecuted;
            this.soundParameterPanel.OperationExecuted += OnOperationExecuted;
            this.soundParameterPanel.Enabled = false;
            this.soundParameterPanel.Visible = false;
        }

        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        /// <summary>
        ///
        /// </summary>
        public SoundParameterPanel SoundParameterPanel
        {
            get
            {
                return this.soundParameterPanel;
            }
        }

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        private void InitializeCommandBindings()
        {
            BindCommand(EditCommands.Undo.ID, QueryStatusUndo, ExecuteUndo);
            BindCommand(EditCommands.Redo.ID, QueryStatusRedo, ExecuteRedo);
        }

        /// <summary>
        /// "元に戻す"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusUndo(Command command)
        {
            if (this.currentDocument == null)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return this.currentDocument.OperationHistory.CanUndo() ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "元に戻す"を実行します。
        /// </summary>
        private bool ExecuteUndo(Command command)
        {
            if (this.currentDocument.OperationHistory.Undo() == false)
            {
                return false;
            }
            return true;
        }

        /// <summary>
        /// "やり直す"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusRedo(Command command)
        {
            if (this.currentDocument == null)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return this.currentDocument.OperationHistory.CanRedo() ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "やり直す"を実行します。
        /// </summary>
        private bool ExecuteRedo(Command command)
        {
            if (this.currentDocument.OperationHistory.Redo() == false)
            {
                return false;
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOperationExecuted(object sender, OperationExecutedEventArgs e)
        {
            IParameterPanelOperationExecutedListener listener =
                this.currentPanel as IParameterPanelOperationExecutedListener;

            if (listener != null)
            {
                listener.OperationExecuted(e);
            }

            MainWindow.BuildCommandUI();
        }

#if false
        /// <summary>
        ///
        /// </summary>
        public OperationHistory OperationHistory
        {
            get { return this.instrumentParameterPanel.OperationHistory; }
            set { this.instrumentParameterPanel.OperationHistory = value; }
        }
#endif
        /// <summary>
        ///
        /// </summary>
        public Instrument Instrument
        {
            get { return this.instrumentParameterPanel.Instrument; }
            set { this.instrumentParameterPanel.Instrument = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public VelocityRegion VelocityRegion
        {
            get { return this.instrumentParameterPanel.VelocityRegion; }
            set { this.instrumentParameterPanel.VelocityRegion = value; }
        }

        /// <summary>
        /// サウンドを取得/設定します。
        /// </summary>
        public Component Item
        {
            set
            {
                if (value == null)
                {
                    Items = null;
                }
                else
                {
                    Items = new Component[] { value };
                }
            }
        }

        /// <summary>
        /// 複数のサウンドを取得/設定します。
        /// </summary>
        public Component[] Items
        {
            get { return this.soundParameterPanel.Items; }
            set
            {
                // 今は1つのサウンドしか対応しないようにします。
                // 実装的には対応を進めておくこと。
                if (value != null &&
                    value.Length != 1)
                {
                    value = null;
                }

                if (IsSame(this.soundParameterPanel.Items, value) == false)
                {
                    this.soundParameterPanel.Items = value;
                }
            }
        }

        /// <summary>
        /// バンクパネルを設定します。
        /// </summary>
        public void SetBankPanel(BankPanel panel)
        {
            Debug.Assert(panel.Document != null, "Document is null");

            this.currentPanel = panel;
            this.currentDocument = panel.Document;

            this.instrumentParameterPanel.OperationHistory = panel.Document.OperationHistory;
            this.soundParameterPanel.OperationHistory = null;

            this.instrumentParameterPanel.Enabled = true;
            this.instrumentParameterPanel.Visible = true;
            this.instrumentParameterPanel.ReadOnly = false;
            this.soundParameterPanel.Enabled = false;
            this.soundParameterPanel.Visible = false;

            this.UpdateUI();
        }

        /// <summary>
        /// サウンドセットパネルを設定します。
        /// </summary>
        public void SetSoundSetPanel(SoundSetPanel panel)
        {
            Instrument = null;
            VelocityRegion = null;

            this.currentPanel = panel;
            this.currentDocument = panel.Document;

            this.instrumentParameterPanel.OperationHistory = null;
            this.soundParameterPanel.OperationHistory = panel.Document.OperationHistory;

            this.instrumentParameterPanel.Enabled = false;
            this.instrumentParameterPanel.Visible = false;
            this.instrumentParameterPanel.ReadOnly = false;
            this.soundParameterPanel.Enabled = true;
            this.soundParameterPanel.Visible = true;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetFindResultPanel(FindResultPanel2 panel)
        {
            this.currentPanel = panel;
            //this.currentDocument = panel.Document;
            this.currentDocument = null;

            this.instrumentParameterPanel.OperationHistory = null;
            //this.soundParameterPanel.OperationHistory = panel.Document.OperationHistory;
            this.soundParameterPanel.OperationHistory = null;

            bool isSoundSet = true;
            if (panel.TargetDocumentKind == FindArgs.TargetDocumentKinds.SoundSet)
            {
                Instrument = null;
                VelocityRegion = null;
            }
            else
            {
                isSoundSet = false;
                Item = null;
            }
            this.instrumentParameterPanel.Enabled = !isSoundSet;
            this.instrumentParameterPanel.Visible = !isSoundSet;
            this.instrumentParameterPanel.ReadOnly = !isSoundSet;
            this.soundParameterPanel.Enabled = isSoundSet;
            this.soundParameterPanel.Visible = isSoundSet;

            this.UpdateUI();
        }

        /// <summary>
        /// UIの更新を行います。
        /// </summary>
        public void UpdateUI()
        {
            instrumentParameterPanel.UpdateUI();
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsSame(Component[] itemsA, Component[] itemsB)
        {
            if (itemsA == null ||
                itemsB == null ||
                itemsA.Length != itemsB.Length)
            {
                return false;
            }

            for (int index = 0; index < itemsA.Length; index++)
            {
                if (itemsA[index] != itemsB[index])
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnExpandChanged(object sender, EventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        /// ウインドウのサイズが変更された時に呼ばれる
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            SuspendLayout();

            base.OnResize(e);

            if (null != this.instrumentParameterPanel)
            {
                this.instrumentParameterPanel.Size = ClientRectangle.Size;
            }

            if (this.soundParameterPanel != null)
            {
                this.soundParameterPanel.Size = ClientRectangle.Size;
            }

            ResumeLayout();
        }

        ///--------------------------------
        /// <summary>
        /// プログラムナンバーの取得
        /// </summary>
        private int GetProgramNo()
        {
            // ★未実装
            return -1;
            /*
            if ( panel_List.ContainsFocus != false ) {
                return panel_List.GetProgramNo();
            }

            return -1;
            */
        }

        ///--------------------------------
        /// <summary>
        /// デフォルトキーの取得
        /// </summary>
        private int GetDefaultKey()
        {
            // ★未実装
            return -1;
            /*
            //
            if ( panel_List.ContainsFocus != false ) {
                return panel_List.GetDefaultKey();
            }

            return -1;
            */
        }

#if false
        /// <summary>
        ///
        /// </summary>
        public void KeyOn( int key )
        {
            int				programNo = GetProgramNo();

            if ( programNo >= 0 ) {
                // ★未実装
                throw new NotImplementedException();
                /*SoundMaker.NoteOn( programNo, key, _BankPanel.SelectedVelocity );*/
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void KeyOff( int key )
        {
            // ★未実装
            throw new NotImplementedException();
            /*SoundMaker.NoteOff( key );*/
        }
#endif

        /// <summary>
        ///
        /// </summary>
        private void OnUndo(object sender, System.EventArgs e)
        {
            if (this.currentPanel != null)
            {
                (this.currentPanel as ICommandTarget).Execute(EditCommands.Undo);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnRedo(object sender, System.EventArgs e)
        {
            if (this.currentPanel != null)
            {
                (this.currentPanel as ICommandTarget).Execute(EditCommands.Redo);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void IKeyPreviewReceiver.KeyDown(KeyEventArgs e)
        {
            if (this.currentPanel is IParameterPanelKeyPreviewListener)
            {
                (this.currentPanel as IParameterPanelKeyPreviewListener).KeyDown(e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void IKeyPreviewReceiver.KeyUp(KeyEventArgs e)
        {
            if (this.currentPanel is IParameterPanelKeyPreviewListener)
            {
                (this.currentPanel as IParameterPanelKeyPreviewListener).KeyUp(e);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public interface IParameterPanelOperationExecutedListener
    {
        void OperationExecuted(OperationExecutedEventArgs e);
    }

    /// <summary>
    ///
    /// </summary>
    public interface IParameterPanelKeyPreviewListener
    {
        void KeyDown(KeyEventArgs e);
        void KeyUp(KeyEventArgs e);
    }

    /// <summary>
    /// コマンドターゲットを持つDockingPage
    /// </summary>
    public class DockingPageWithCommandTarget : DockingPage, ICommandTarget
    {
        private CommandBindingCollection commandBindings = new CommandBindingCollection();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DockingPageWithCommandTarget(string pageName) : base(pageName)
        {
        }

        /// <summary>
        /// コンストラクタ（デザイナ用）
        /// </summary>
        internal DockingPageWithCommandTarget() : base(string.Empty)
        {
        }

        /// <summary>
        /// 指定コマンドを処理するコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">処理するコマンド。</param>
        /// <returns>コマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return null;
            }

            return this;
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return CommandStatus.Unsupported;
            }

            return this.commandBindings[command.ID].QueryStatus(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        bool ICommandTarget.Execute(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return false;
            }

            return this.commandBindings[command.ID].Execute(command);
        }

        /// <summary>
        /// コマンドを追加します。
        /// </summary>
        protected void BindCommand(string id, CommandBinding.QueryStatusHandler queryStatusHandler, CommandBinding.ExecuteHandler executeHandler)
        {
            this.commandBindings.Add
                (new CommandBinding(this, id, queryStatusHandler, executeHandler));
        }
    }
}
