﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Drawing.Imaging;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Preview.Service;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public partial class InGamePanel : DockingPage, IToolWindowPage, ICommandTarget
    {
        public const string PageName = "InGame";

        private ToolStripAdapter menuAdapter;
        private CommandBindingCollection commandBindings = new CommandBindingCollection();

        private CommonListAdapter listAdapter = null;
        private ComponentList componentList = new ComponentList(null);

        public event EventHandler SelectChanged;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InGamePanel() : base(PageName)
        {
            InitializeComponent();
            InitializeCommandBindings();
            InitializeMenu();

            // テキストとアイコンをロードする
            Text = FrameworkResources.MessageResource.ToolWindowName_InGame;
            Image = FrameworkResources.ImageResource.
                BitmapIconInGameWindow.MakeNewTransparent(UIServiceBase.TransparentColor);

            DockVisible = false;

            // 初期サイズを設定する
            FloatingBounds = new Rectangle
                (-1, -1,
                  Width + (SystemInformation.FrameBorderSize.Width * 2),
                  (Height + SystemInformation.CaptionHeight +
                    SystemInformation.FrameBorderSize.Height));

            //
            this.listAdapter = new CommonListAdapter(this.ListDecorationEvaluator);
            this.listAdapter.Setters.Add
                (ProjectParameterNames.SndEdit,
                  SndEdit.SetSndEdit);
            this.listAdapter.Setters.Add
                (ProjectParameterNames.SndEditEnabled,
                  SndEdit.SetSndEditEnabled);
            this.listAdapter.ListItemCreator = delegate (Component component)
                {
                    CommonListItem item = new CommonListItem(component);
                    item.ConstValueGetters.Add("SndEditState", SndEdit.GetSndEditState);
                    item.ReadOnlyNames.Add(ProjectParameterNames.Name);
                    return item;
                };
            this.listAdapter.PreviewPlay += OnPreviewPlay;

            InitializeComponentList();

            this.listCtrl.ReadOnly = false;
            this.listCtrl.AllowDragItem = false;
            this.listCtrl.HeaderHeight = 28;
            this.listCtrl.SelectChanged += OnSelectChanged;
            this.listCtrl.ItemDoubleClicked += OnItemDoubleClicked;

            //this.listCtrl.QueryFileDropped += OnQueryFileDropped;
            //this.listCtrl.FileDropped += OnFileDropped;
            //this.listCtrl.EditBegan += OnEditBegan;
            //this.listCtrl.EditEnded += OnEditEnded;

            IListSubItem subItem = new ListSubItemSndEditWithCheck();
            this.listCtrl.SubItems.Remove(subItem.Name);
            this.listCtrl.SubItems.Add(new ListSubItemSndEdit());

            //this.listCtrl.HeaderSource = _headerAdapters[ApplicationBase.Instance.Traits.ListTraits.ListNames[typeof(Bank)]];

            this.listCtrl.HeaderSource = FormsApplication.Instance.UIService.DefaultHeaderAdapters[ListTraits.ListName_SndEdit];
            this.listCtrl.ItemsSource = this.listAdapter;

            //
            btn_DeleteAll.Image = FrameworkResources.ImageResource.IconDeleteAll;
            btn_TransferAll.Image = FrameworkResources.ImageResource.IconUpdateAll;

            //
            RealtimeEditService service = Application.RealtimeEditService;
            service.MonitoringTargetStateChanged += OnMonitoringTargetStateChanged;
            service.MonitoringTargetAdded += OnMonitoringTargetAdded;
            service.MonitoringTargetRemoved += OnMonitoringTargetRemoved;
            service.MonitoringTargetRenamed += OnMonitoringTargetRenamed;
            service.MonitoringTargetCleared += OnMonitoringTargetCleared;
            service.TargetMachineInformationChanged += OnTargetMachineInformationChanged;

            //
            ShowTargetMachineInformation(false);
            UpdateMemoryUsage(false, 0, 0);
            UpdateItemCount(false, 0, 0);
            UpdateFileCount(false, 0, 0);

            UpdateToolBarButtons();
        }

        /// <summary>
        ///
        /// </summary>
        private ListHeaderAdapterDictionary _headerAdapters = new ListHeaderAdapterDictionary();

        /// <summary>
        /// リストヘッダアダプタディクショナリを取得します。
        /// </summary>
        public IReadOnlyDictionary<string, ListHeaderAdapterBase> HeaderAdapters
        {
            get { return _headerAdapters; }
        }

        /// <summary>
        /// セルの装飾に関する情報を提供します。
        /// </summary>
        protected CommonListDecorationEvaluator ListDecorationEvaluator { get; } =
            new CommonListDecorationEvaluator();

        /// <summary>
        /// 指定コマンドを処理するコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">処理するコマンド。</param>
        /// <returns>コマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return null;
            }

            return this;
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return CommandStatus.Unsupported;
            }

            return this.commandBindings[command.ID].QueryStatus(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        bool ICommandTarget.Execute(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return false;
            }

            return this.commandBindings[command.ID].Execute(command);
        }

        /// <summary>
        ///
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
#if false
            switch( keyData )
            {
                btn_Delete.PerformClick();
                return true;
            }
#endif

            return base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool CanPlay
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void Play(Sound sound)
        {
            Debug.Assert(false, "Not implimented");
        }

        /// <summary>
        ///
        /// </summary>
        protected void OnSndEditConnectionChanged(object sender, ConnectionChangedEventArgs e)
        {
            switch (e.State)
            {
                case ConnectionState.Connected:
                    ShowTargetMachineInformation(true);
                    break;

                case ConnectionState.Disconnected:
                    ShowTargetMachineInformation(false);
                    break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FormsApplication Application
        {
            get
            {
                return FormsApplication.Instance as FormsApplication;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (null != this.menuAdapter) { return; }

            this.menuAdapter = ToolStripAdapter.FromToolStrip
                (Application.CommandService, this.contextMenuStrip);
            this.menuAdapter.CommandTarget = Application;
            this.menuAdapter.CommandExecuted += OnCommandExecuted;

            BuildCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUI()
        {
            if (this.menuAdapter != null)
            {
                this.menuAdapter.BuildUI();
            }
        }

        /// <summary>
        /// コンテキストメニューのコマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnCommandExecuted(object sender, CommandEventArgs e)
        {
            if (e.IsExecuted != false)
            {
                if (e.Command == EditCommands.Delete)
                {
                }
            }

            MainWindow.BuildCommandUI();
            BuildCommandUI();
        }

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        private void InitializeCommandBindings()
        {
            //
            this.commandBindings.Add
                (new CommandBinding(this, EditCommands.Delete.ID,
                                      QueryStatusDelete, ExecuteDelete));
            this.commandBindings.Add
                (new CommandBinding(this, SndEditCommands.Transfer.ID,
                                      QueryStatusTransfer, ExecuteTransfer));
            this.commandBindings.Add
                (new CommandBinding(this, EditCommands.SelectAll.ID,
                                      QueryStatusSelectAll, ExecuteSelectAll));

            this.commandBindings.Add
                (new CommandBinding(this, EditCommands.DisableItem.ID,
                                      QueryStatusDisableItem, ExecuteDisableItem));
            this.commandBindings.Add
                (new CommandBinding(this, EditCommands.EnableItem.ID,
                                      QueryStatusEnableItem, ExecuteEnableItem));

            // プレビュー関連コマンドハンドラ
            this.commandBindings.Add
                (new CommandBinding(this, PreviewCommands.Play.ID,
                                      QueryStatusPlay, ExecutePlay));
            this.commandBindings.Add
                (new CommandBinding(this, PreviewCommands.Play2.ID,
                                      QueryStatusPlay, ExecutePlay));
        }

        /// <summary>
        ///
        /// </summary>
        private void InitializeComponentList()
        {
            this.componentList.Clear();

            foreach (Component component in Application.RealtimeEditService.MonitoringTargets)
            {
                this.componentList.Add(component);
            }
            this.listAdapter.AddItems(this.componentList);
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateComponentList()
        {
            var monitoringTargets = Application.RealtimeEditService.MonitoringTargets;

            // 対象となっていない Componentをリストから削除します。
            var removeComponents = this.componentList
                .Where<Component>(c => monitoringTargets.Contains(c) == false)
                .ToArray();
            foreach (var removeComponent in removeComponents)
            {
                this.componentList.Remove(removeComponent);
            }

            // 追加、並び替えを行います。
            for (int newIndex = 0; newIndex < monitoringTargets.Length; newIndex++)
            {
                var component = monitoringTargets[newIndex];
                int index = this.componentList.IndexOf(component);
                if (index >= 0)
                {
                    if (index != newIndex)
                    {
                        // 対象のコレクションと同じ並びにします。
                        this.componentList.Move(component, newIndex);
                    }
                }
                else
                {
                    this.componentList.Add(component);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowTargetMachineInformation(bool b)
        {
            lbl_Memory.Visible = b;
            lbl_ItemCount.Visible = b;
            lbl_FileCount.Visible = b;
        }

        /// <summary>
        /// メモリ情報を更新します。
        /// </summary>
        private void UpdateMemoryUsage(bool outOfMemory, uint memoryMax, uint memoryUsage)
        {
            Color color = Control.DefaultForeColor;

            uint usage = memoryUsage < 1024 ? 1 : memoryUsage / 1024;
            uint max = memoryMax < 1024 ? 1 : memoryMax / 1024;
            string messageText = string.Empty;

            if (outOfMemory == true)
            {
                messageText = MessageResource.Label_MemoryUsageOutOfMemory;
                color = Color.Red;
            }
            else
            {
                messageText = MessageResource.Label_MemoryUsage;
            }

            lbl_Memory.ForeColor = color;
            lbl_Memory.Text = String.Format(messageText, usage, max, (uint)(((float)usage / max) * 100));
        }

        /// <summary>
        /// アイテム数情報を更新します。
        /// </summary>
        private void UpdateInformation(bool error, uint max, uint count, Label lblControl, string message, string errorMessage)
        {
            string text = String.Empty;
            Color color = Control.DefaultForeColor;

            if (error != false)
            {
                text = errorMessage;
                color = Color.Red;
            }
            else
            {
                text = message;
            }

            lblControl.ForeColor = color;
            lblControl.Text = String.Format(text, count, max);
        }

        /// <summary>
        /// アイテム数情報を更新します。
        /// </summary>
        private void UpdateItemCount(bool error, uint max, uint count)
        {
            UpdateInformation(error, max, count,
                               lbl_ItemCount,
                               MessageResource.Label_ItemCountOverMaxCount,
                               MessageResource.Label_ItemCount);
        }

        /// <summary>
        /// ファイル数情報を更新します。
        /// </summary>
        private void UpdateFileCount(bool error, uint max, uint count)
        {
            UpdateInformation(error, max, count,
                               lbl_FileCount,
                               MessageResource.Label_FileCountOverMaxCount,
                               MessageResource.Label_FileCount);
        }

        /// <summary>
        /// ツールバーのボタンの更新を行います。
        /// </summary>
        private void UpdateToolBarButtons()
        {
        }

        /// <summary>
        /// リストで選択されているComponentを取得します。
        /// </summary>
        private Component[] SelectedComponents
        {
            get
            {
                Component[] components = this.listCtrl.GetSelectedItems()
                    .Select(i => i.Target)
                    .Cast<Component>()
                    .ToArray();
                return components;
            }
        }

        /// <summary>
        /// "削除"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusDelete(Command command)
        {
            Component[] components = SelectedComponents;
            bool enabledDelete = components.Length > 0 ? true : false;

            return (enabledDelete != false ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "削除"の実行
        /// </summary>
        private bool ExecuteDelete(Command command)
        {
            var caretItemSetter = new CaretItemSetter();
            caretItemSetter.PreserveNextItem(this.listCtrl);

            foreach (Component component in SelectedComponents)
            {
                this.listAdapter.SetValue(component, ProjectParameterNames.SndEdit, false);
            }

            caretItemSetter.SetCaretByPreservedItem(this.listCtrl);

            OnSelectChanged();
            UpdateComponentList();
            this.listCtrl.Invalidate();
            MainWindow.RefreshPanels();
            return true;
        }

        /// <summary>
        /// "再転送"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusTransfer(Command command)
        {
            Component[] components = SelectedComponents;
            bool enabled = components.Length > 0 ? true : false;

            return (enabled != false ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "再転送"の実行
        /// </summary>
        private bool ExecuteTransfer(Command command)
        {
            Component[] components = SelectedComponents;
            Transfer(components);
            return true;
        }

        /// <summary>
        /// "すべて選択"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusSelectAll(Command command)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// "すべて選択"の実行
        /// </summary>
        private bool ExecuteSelectAll(Command command)
        {
            SelectAll();
            return true;
        }

        /// <summary>
        /// "再生"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusPlay(Command command)
        {
#if false
            if( PreviewPlayerOperator.SelectedPreviewPlayer == null )
            {
                return CommandStatus.SupportedAndVisible;
            }
#else
            if (CanPlay == false)
            {
                return CommandStatus.SupportedAndVisible;
            }
#endif

            Sound[] sounds = SelectedComponents
                .Where(s => s is Sound)
                .Cast<Sound>()
                .ToArray();
            if (sounds.Length != 1)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// "再生"の実行
        /// </summary>
        private bool ExecutePlay(Command command)
        {
            Sound[] sounds = SelectedComponents
                .Where(s => s is Sound)
                .Cast<Sound>()
                .ToArray();
#if false
            // TODO: 未実装 PreviewPlayerOperator.Play( sounds[0]);
#else
            Play(sounds[0]);
#endif
            return true;
        }

        /// <summary>
        /// "アイテムを無効にする"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusDisableItem(Command command)
        {
            Component[] components = SelectedComponents;
            bool enabled = components.Length > 0 ? true : false;

            return (enabled != false ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "アイテムを無効にする"の実行
        /// </summary>
        private bool ExecuteDisableItem(Command command)
        {
            EnableItem(SelectedComponents, false);
            return true;
        }

        /// <summary>
        /// "アイテムを有効にする"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusEnableItem(Command command)
        {
            return QueryStatusDisableItem(command);
        }

        /// <summary>
        /// "アイテムを有効にする"の実行
        /// </summary>
        private bool ExecuteEnableItem(Command command)
        {
            EnableItem(SelectedComponents, true);
            return true;
        }

        /// <summary>
        /// 転送します。
        /// </summary>
        private void Transfer(Component[] components)
        {
            Application.RealtimeEditService.UpdateMonitoringTargets(components);
        }

        /// <summary>
        /// 全てのアイテムを選択します。
        /// </summary>
        private void SelectAll()
        {
            this.listCtrl.SelectAll();
            this.listCtrl.Invalidate();
            OnSelectChanged();
        }

        /// <summary>
        /// アイテムを無効/有効にします。
        /// </summary>
        private void EnableItem(Component[] components, bool b)
        {
            foreach (Component component in components)
            {
                this.listAdapter.SetValue
                    (component, ProjectParameterNames.SndEditEnabled, b);
            }
            this.listCtrl.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlay(object sender, EventArgs e)
        {
            Sound sound = sender as Sound;
            if (sound == null)
            {
                return;
            }

#if false
            //Debug.Assert( sound != null, "Sender is not sound");
            // TODO: 未実装 PreviewPlayerOperator.Play( sound);
#else
            Play(sound);
#endif
        }

        /// <summary>
        /// 編集対象の状態が変化した時に呼ばれます。
        /// </summary>
        private void OnMonitoringTargetStateChanged(Component[] components)
        {
            this.listCtrl.Invalidate();
            MainWindow.RedrawPanels();
        }

        /// <summary>
        /// 編集対象が追加された時に呼ばれます。
        /// </summary>
        private void OnMonitoringTargetAdded(Component[] components)
        {
            UpdateComponentList();
            this.listCtrl.Invalidate();
            MainWindow.RedrawPanels();
        }

        /// <summary>
        /// 編集対象が削除された時に呼ばれます。
        /// </summary>
        private void OnMonitoringTargetRemoved(Component[] components)
        {
            UpdateComponentList();
            this.listCtrl.Invalidate();
            MainWindow.RedrawPanels();
        }

        /// <summary>
        /// 編集対象の名前が変更された時に呼ばれます。
        /// </summary>
        private void OnMonitoringTargetRenamed(Component[] components)
        {
            UpdateComponentList();
            this.listCtrl.Invalidate();
            MainWindow.RedrawPanels();
        }

        /// <summary>
        /// 編集対象が全て削除された時に呼ばれます。
        /// </summary>
        private void OnMonitoringTargetCleared(object sender, EventArgs e)
        {
            UpdateComponentList();
            this.listCtrl.Invalidate();
        }

        /// <summary>
        /// 実機の情報が更新された時に呼ばれます。
        /// </summary>
        private void OnTargetMachineInformationChanged(RealtimeEditService.TargetMachineInformation info)
        {
            UpdateMemoryUsage(info.OutOfMemory, info.MemoryMax, info.MemoryUsage);
            UpdateItemCount(info.ItemOverflow, info.ItemCountMax, info.ItemCount);
            UpdateFileCount(info.FileOverflow, info.FileCountMax, info.FileCount);
        }

        /// <summary>
        /// リストの選択が変更された時に呼ばれる
        /// </summary>
        private void OnSelectChanged()
        {
            OnSelectChanged(this, EventArgs.Empty);
        }

        ///
        private void OnSelectChanged(object sender, EventArgs e)
        {
            if (SelectChanged != null)
            {
                SelectChanged(this, e);
            }

            BuildCommandUI();
        }

        /// <summary>
        /// アイテムをダブルクリックした時に呼ばれます。
        /// </summary>
        private void OnItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ComponentListItem item = e.Item as ComponentListItem;
            Component component = item.Target;

            MainWindow.ShowPageByComponent(component);
        }

        /// <summary>
        /// "すべて削除"ボタンが押された時に呼ばれます。
        /// </summary>
        private void OnClick_BtnDeleteAll(object sender, EventArgs e)
        {
            foreach (Component component in Application.RealtimeEditService.MonitoringTargets)
            {
                component.Parameters[ProjectParameterNames.SndEdit].Value = false;
            }

            MainWindow.RedrawPanels();
        }

        /// <summary>
        /// "すべて再転送"ボタンが押された時に呼ばれます。
        /// </summary>
        private void OnClick_BtnTransferAll(object sender, EventArgs e)
        {
            RealtimeEditService service = Application.RealtimeEditService;
            service.UpdateAllMonitoringTargets();
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ListSubItemSndEdit : ListPartParameter, IListSubItem
    {
        /// <summary>
        ///
        /// </summary>
        string IListSubItem.Name
        {
            get { return "SndEdit"; }
        }

        /// <summary>
        ///
        /// </summary>
        bool IListSubItem.EditByKey
        {
            get { return true; }
        }

        /// <summary>
        ///
        /// </summary>
        IListPart[] IListSubItem.GetParts(ListDrawDescriptor desc)
        {
            ListPart part0 = null;
            //ListPart part1 = null;
            Rectangle bounds = new Rectangle();

            int width = desc.Bounds.Width < 20 ? desc.Bounds.Width : 20;
            int height = desc.Bounds.Height < 17 ? desc.Bounds.Height : 17;
            int x = (desc.Bounds.Width - width) / 2;
            int y = (desc.Bounds.Height - height) / 2;

            bounds.X = x;
            bounds.Y = y;
            bounds.Width = width;
            bounds.Height = height;

            part0 = new ListPart("SndEdit", "SndEdit", null, bounds,
                                (ListPartActions.EditByClick |
                                 ListPartActions.EditByClickOnCaret));

#if false
            //
            bounds = Rectangle.Empty;
            part1 = new ListPart("Blank", "PreviewPlay", null, bounds,
                                 ListPartActions.EditByDoubleClick);

            //
            return new[] { part0, part1 };
#else
            return new[] { part0 };
#endif
        }

        /// <summary>
        ///
        /// </summary>
        string IListSubItem.RepresentativeEditorName(bool immediateEdit)
        {
            return "PreviewPlay";
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ListPartSndEditDrawer : IListPartDrawer
    {
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return "SndEdit"; }
        }

        /// <summary>
        ///
        /// </summary>
        public void Draw(ListDrawDescriptor desc)
        {
#if false
            Image image = ImageResource.BitmapIconPlaySmall;

            if (image != null)
            {
                Rectangle bounds = desc.Bounds;
                if (VisualStyleRenderer.IsSupported)
                {
                    VisualStyleRenderer renderer =
                        new VisualStyleRenderer(VisualStyleElement.Button.PushButton.Normal);
                    bounds = new Rectangle(bounds.X, bounds.Y, bounds.Width + 1, bounds.Height + 1);
                    renderer.DrawBackground(desc.Gc, bounds);
                }
                else
                {
                    ControlPaint.DrawButton(desc.Gc, bounds, ButtonState.Normal);
                }
                ImageAttributes attr = new ImageAttributes();
                attr.SetColorKey(Color.Magenta, Color.Magenta);
                bounds = new Rectangle(
                                       bounds.X + (bounds.Width - image.Width) / 2,
                                       bounds.Y + (bounds.Height - image.Height) / 2,
                                       image.Width,
                                       image.Height
                                       );
                desc.Gc.DrawImage(
                                  image,
                                  bounds,
                                  0, 0, image.Width, image.Height,
                                  GraphicsUnit.Pixel,
                                  attr
                                  );
            }
#else
            CommonListItem item = desc.Item as CommonListItem;

            Image image = GetIcon(item);
            if (image != null)
            {
                Rectangle bounds = desc.Bounds;
                ImageAttributes attr = new ImageAttributes();
                attr.SetColorKey(Color.Magenta, Color.Magenta);
                bounds = new Rectangle(bounds.X + (bounds.Width - image.Width) / 2,
                                        bounds.Y + (bounds.Height - image.Height) / 2,
                                        image.Width,
                                        image.Height);

                desc.Gc.DrawImage(image,
                                   bounds,
                                   0, 0, image.Width, image.Height,
                                   GraphicsUnit.Pixel,
                                   attr);
            }

            //Rectangle bounds = desc.Bounds;
            //ControlPaint.DrawButton(desc.Gc, bounds, ButtonState.Normal);
#endif
        }

        /// <summary>
        ///
        /// </summary>
        public int DrawWidth(ListDrawDescriptor desc)
        {
            return desc.Bounds.Width + 2;
        }

        /// <summary>
        ///
        /// </summary>
        //public Image GetIcon( Component component)
        public Image GetIcon(CommonListItem item)
        {
#if false
            FormsApplicationCafe app = FormsApplicationCafe.Instance as FormsApplicationCafe;
            Debug.Assert( app != null, "Application is not FormsApplicationCafe");
            RealtimeEditService service = app.RealtimeEditService;

            RealtimeEditService.MonitoringState state = service.GetMonitoringState( component);

            switch( state )
            {
            case RealtimeEditService.MonitoringState.Converting:
                return ImageResource.SndEditConverting;

            case RealtimeEditService.MonitoringState.Transferring:
                return ImageResource.SndEditUpdating;

            case RealtimeEditService.MonitoringState.Completed:
                return ImageResource.SndEditSucceeded;

            case RealtimeEditService.MonitoringState.Error:
                return ImageResource.SndEditFailed;

            default:
                return null;
            }
#else

            int state = (int)item.GetConstValue("SndEditState").Value;
            switch (state)
            {
                case 0:
                    return ImageResource.SndEditConverting;

                case 1:
                    return ImageResource.SndEditUpdating;

                case 2:
                    return ImageResource.SndEditSucceeded;

                case 3:
                    return ImageResource.SndEditFailed;

                case 4:
                    return ImageResource.SndEditDeleting;

                default:
                    return null;
            }
#endif
        }
    }
}
