﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Utilities;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.ToolDevelopmentKit.Collections;

    using SFresrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    ///
    /// </summary>
    public partial class FindResultPanel : DockingPage, IToolWindowPage, ICommandTarget, IQueryCommandParameter, IPanel, IParameterPanelOperationExecutedListener
    {
        public const string PageName = "FindResult";

        private bool _filterUpdateLocked = false;

        private CommonListCtrl _ListCtrl = null;
        private CommonListAdapter _ListAdapter = null;
        private Component[] _ResultComponents = null;

        private ToolStripAdapter listMenuAdapter = null;

        private CommandBindingCollection commandBindings = new CommandBindingCollection();
        private TransactionPackStack transactionPackStack = new TransactionPackStack();

        private ToolStripMenuItem _HeaderContextMenuItem_AddPresetListColumns = null;
        private ToolStripMenuItem _HeaderContextMenuItem_ApplyPresetListColumns = null;

        private HashSet<string> excludeColumnNames = new HashSet<string>();

        public event EventHandler PreviewPlay;

        //
        private Dictionary<string, List<SoundSetBank>> soundSetBankDictionary = null;

        ///
        private ButtonPair<int>[] ButtonColorIndexPairs = null;

        /// <summary>
        ///
        /// </summary>
        public delegate CommonListCtrl ListControlCreateHandler();
        public static ListControlCreateHandler ListControlCreator
        {
            get;
            set;
        }

        /// <summary>
        /// スタティックコンストラクタ
        /// </summary>
        static FindResultPanel()
        {
            ListControlCreator = delegate ()
                {
                    return new CommonListCtrl();
                };
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FindResultPanel() : base(PageName)
        {
            InitializeComponent();

            Text = MessageResource.ToolWindowName_FindResult;
            Image = ImageResource.BitmapIconFindResultWindow.MakeNewTransparent
                (UIServiceBase.TransparentColor);

            DockVisible = false;

            // ツールバーのカラーボタンの設定です。
            this.tool_ColorRed.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorRed);
            this.tool_ColorOrange.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorOrange);
            this.tool_ColorYellow.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorYellow);
            this.tool_ColorGreen.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorGreen);
            this.tool_ColorSkyBlue.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorLightBlue);
            this.tool_ColorBlue.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorBlue);
            this.tool_ColorPurple.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorPurple);
            this.tool_ColorPink.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorPink);

            this.ButtonColorIndexPairs = new ButtonPair<int>[] {
                new ButtonPair<int>( tool_ColorNone,    0),
                new ButtonPair<int>( tool_ColorRed,     1),
                new ButtonPair<int>( tool_ColorOrange,  2),
                new ButtonPair<int>( tool_ColorYellow,  3),
                new ButtonPair<int>( tool_ColorGreen,   4),
                new ButtonPair<int>( tool_ColorSkyBlue, 5),
                new ButtonPair<int>( tool_ColorBlue,    6),
                new ButtonPair<int>( tool_ColorPurple,  7),
                new ButtonPair<int>( tool_ColorPink,    8),
            };

            //
            this._ListAdapter = new CommonListAdapter();
            this._ListAdapter.OperationExecuted += OnOperationExecuted;
            this._ListAdapter.PreviewPlay += OnPreviewPlay;
            this._ListAdapter.Setters.Add(ProjectParameterNames.SndEdit, SndEdit.SetSndEdit);
            this._ListAdapter.ListItemCreator = delegate (Component component)
                {
                    if (component is Instrument)
                    {
                        SoundSetBank[] soundSetBanks = null;

                        ImaginaryInstrument imaginaryInstrument = component as ImaginaryInstrument;
                        if (imaginaryInstrument != null)
                        {
                            string bankFilePath = imaginaryInstrument.BankFilePath.ToLower();
                            if (this.soundSetBankDictionary.ContainsKey(bankFilePath) != false)
                            {
                                soundSetBanks = this.soundSetBankDictionary[bankFilePath].ToArray();
                            }
                        }


                        return new ImaginaryInstrumentListItem(component, soundSetBanks);
                    }
                    else
                    {
                        CommonListItem item = new CommonListItem(component);
                        item.ConstValueGetters.Add("SndEditState", SndEdit.GetSndEditState);
                        return item;
                    }
                };

            //
            this.soundSetBankDictionary = new Dictionary<string, List<SoundSetBank>>();

            //
            this._ListCtrl = ListControlCreator();
            this._ListCtrl.Dock = DockStyle.Fill;
            this._ListCtrl.Parent = this.panel;
            this._ListCtrl.AllowDragItem = true;
            this._ListCtrl.AttachedContextMenuStrip = this.listMenu;
            this._ListCtrl.DropOnDragItem = false;
            this._ListCtrl.DecideFileDropPosition = false;
            this._ListCtrl.FollowSelectedItemToCaret = true;
            this._ListCtrl.InterlockSelectedItemAndSubItem = true;
            this._ListCtrl.MoveNewLineCaretLeftAndRight = true;
            this._ListCtrl.MoveNextLineCaretAfterEndEditByEnter = false;
            this._ListCtrl.StopCaretLeftAndRight = false;
            this._ListCtrl.SuspendTransaction = false;
            this._ListCtrl.HeaderHeight = 28;
            this._ListCtrl.HeaderCtrl.SortEnabled = true;
            this._ListCtrl.ItemsSource = this._ListAdapter;
            this._ListCtrl.SelectChanged += OnSelectChanged;
            this._ListCtrl.ItemDoubleClicked += OnListItemDoubleClicked;
            this._ListCtrl.QueryFileDropped += OnQueryFileDropped;
            this._ListCtrl.FileDropped += OnFileDropped;
            this._ListCtrl.Enter += OnEnter;
            this._ListCtrl.Leave += OnLeave;
            this._ListCtrl.EditBegan += OnEditBegan;
            this._ListCtrl.EditEnded += OnEditEnded;
            _HeaderContextMenuItem_AddPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_AddPresetListColumns);
            _HeaderContextMenuItem_AddPresetListColumns.Click +=
                OnHeaderMenuItemAddPresetListColumnsClick;
            _HeaderContextMenuItem_ApplyPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_ApplyPresetListColumns);
            _ListCtrl.HeaderCtrl.InsertContextMenuItems =
                new ToolStripMenuItem[] { _HeaderContextMenuItem_AddPresetListColumns,
                                          _HeaderContextMenuItem_ApplyPresetListColumns };
            _ListCtrl.HeaderCtrl.ContextMenuOpening += OnHeaderCtrlContextMenuOpening;

#if false
            //
            MultipleBankEditDialogInplaceEditor editor = null;
            editor = this._ListCtrl.InplaceEditors["BankDialog"] as
                MultipleBankEditDialogInplaceEditor;
            editor.Execute = ExecuteEditMultipleBank;
#endif

            //
            ((IDockingPage)this).DockStateChanged += OnDockStateChanged;
            ((IDockingPage)this).DockVisibleChanged += OnDockVisibleChanged;
            this.ActiveChanged += OnActiveChanged;
            this.WindowActiveChanged += OnWindowActiveChanged;

            this.FocusControl = _ListCtrl;

            //
            this.toolStripStatusLabel.Image =
                ImageResource.BitmapIconWarningFindResult.MakeNewTransparent
                (UIServiceBase.TransparentColor);

            // 編集不可セルの描画方法を変更します。
            this._ListCtrl.PartDrawers.Replace(new NAListPartDrawer());

            // 初期配置を設定する
            FloatingBounds = new Rectangle
                (-1, -1, Width + (SystemInformation.FrameBorderSize.Width * 2),
                  Height + SystemInformation.CaptionHeight +
                  SystemInformation.FrameBorderSize.Height);

            FormsApplication.Instance.ProjectService.Closed += OnProjectClosed;

            //
            ShowTransactionStackWarning(false);

            // 表示させたくないカラム名を設定します。
            // this.excludeColumnNames.Add( ProjectParameterNames.SoundSetItem.PreviewPlay);
            // this.excludeColumnNames.Add( ProjectParameterNames.SndEdit);
            // this.excludeColumnNames.Add( ProjectParameterNames.SndEditEnabled);

            //
            InitializeCommandBindings();

            // 行ヘッダを表示させる為に、カラム無しのHeaderSourceを設定します。
            SetResult(null);

            //
            ApplyOptionSettings();
        }

#if true
        /// <summary>
        /// マルチバンクの編集ダイアログを開きます。
        /// リストからの編集で呼ばれます。
        /// </summary>
        private void ExecuteEditMultipleBank(object sender, System.ComponentModel.CancelEventArgs e)
        {
            MultipleBankEditDialogInplaceEditor editor = sender as MultipleBankEditDialogInplaceEditor;
            CommonListItem item = editor.Item as CommonListItem;
            SequenceSound sequenceSound = item.Target as SequenceSound;

            SoundSet soundSet = GetSoundSet(item.Target);
            SoundSetDocument soundSetDocument = GetSoundSetDocument(soundSet);

            //
            string filePath = soundSetDocument.Resource.Key;
            if (!ProjectConfiguration.DocumentViews.ContainsKey(filePath))
            {
                return;
            }

            XmlSoundSetDocumentView configuration =
                ProjectConfiguration.DocumentViews[filePath] as XmlSoundSetDocumentView;

            IDictionary<string, XmlList> xmlLists = configuration.Lists.ExportDictionary();

            ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            foreach (XmlList xmlList in xmlLists.Values)
            {
                applier.Apply(xmlList);
            }

            //
            MultipleBankSettingsDialog dialog = new MultipleBankSettingsDialog
                (ProjectService,
                  soundSetDocument,
                  sequenceSound,
                  headerAdapters);

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                e.Cancel = true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected SoundProjectConfiguration ProjectConfiguration
        {
            get { return FormsApplication.Instance.ProjectConfiguration; }
        }
#endif

        /// <summary>
        /// 検索結果を設定します。
        /// </summary>
        public void SetResult(Component[] components)
        {
            List<HeaderColumnData> list = new List<HeaderColumnData>();
            bool editable = false;

            //
            this.soundSetBankDictionary.Clear();

            //
            if (components != null)
            {
                Instrument[] instruments = GetInstrument(components);
                if (instruments.Length > 0)
                {
                    //
                    foreach (SoundSetBank soundSetBank in ProjectService.SoundSetBanks)
                    {
                        string filePath = Path.GetFullPath(soundSetBank.FilePath).ToLower();

                        if (this.soundSetBankDictionary.ContainsKey(filePath) == false)
                        {
                            this.soundSetBankDictionary.Add
                                (filePath, new List<SoundSetBank>());
                        }

                        this.soundSetBankDictionary[filePath].Add(soundSetBank);
                    }

                    //
                    HeaderColumnData headerColumnData = GetHeaderColumnData
                        (ListTraits.ListName_SequenceSound,
                          ProjectParameterNames.SequenceSound.SoundSetBankReferences);
                    list.Add(headerColumnData);

                    AddHeaderColumnData(list, ListTraits.ListName_Instrument);

                    SetItemButtonEnabled(false);
                }
                else
                {
                    editable = true;

                    //
                    components = GetComponentsAtRelevanceComponent(components);

                    //
                    bool streamSoundContained = false;
                    bool streamSoundTrackContained = false;
                    bool waveSoundSetContained = false;
                    bool waveSoundContained = false;
                    bool sequenceSoundSetContained = false;
                    bool sequenceSoundContained = false;
                    bool bankContained = false;
                    bool waveArchiveContained = false;
                    bool groupContained = false;
                    bool playerContained = false;

                    foreach (Component component in components)
                    {
                        if (component is StreamSound && streamSoundContained == false)
                        {
                            streamSoundContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_StreamSound);
                        }
                        else if (component is StreamSoundTrack && streamSoundTrackContained == false)
                        {
                            streamSoundTrackContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_StreamSoundTrack);
                        }
                        else if (component is WaveSoundSet && waveSoundSetContained == false)
                        {
                            waveSoundSetContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_WaveSoundSet);
                        }
                        else if (component is WaveSound && waveSoundContained == false)
                        {
                            waveSoundContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_WaveSoundSetItem);
                        }
                        else if (component is SequenceSoundSet && sequenceSoundSetContained == false)
                        {
                            sequenceSoundSetContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_SequenceSoundSet);
                        }
                        else if (component is SequenceSound && sequenceSoundContained == false)
                        {
                            sequenceSoundContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_SequenceSound);
                        }
                        else if (component is SoundSetBank && bankContained == false)
                        {
                            bankContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_Bank);
                        }
                        else if (component is WaveArchive && waveArchiveContained == false)
                        {
                            waveArchiveContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_WaveArchive);
                        }
                        else if (component is Group && groupContained == false)
                        {
                            groupContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_Group);
                        }
                        else if (component is Player && playerContained == false)
                        {
                            playerContained = true;
                            AddHeaderColumnData(list, ListTraits.ListName_Player);
                        }
                    }

                    SetItemButtonEnabled(true);
                }
            }

            //
            this._ResultComponents = components;

            // コメント、ユーザーデータのカラムを最後尾に移動させます。
            HashSet<string> names = new HashSet<string> {
                ProjectParameterNames.UserParameter,
                ProjectParameterNames.UserParameter1,
                ProjectParameterNames.UserParameter2,
                ProjectParameterNames.UserParameter3,
                ProjectParameterNames.Comment,
                ProjectParameterNames.Comment1,
                ProjectParameterNames.Comment2,
                ProjectParameterNames.Comment3,
                ProjectParameterNames.Comment4,
                ProjectParameterNames.Comment5,
                ProjectParameterNames.Comment6,
                ProjectParameterNames.Comment7,
                ProjectParameterNames.Comment8,
                ProjectParameterNames.Comment9,
            };

            HeaderColumnData[] headerColumnDatas = list.ToArray();
            foreach (HeaderColumnData headerColumnData in headerColumnDatas)
            {
                if (names.Contains(headerColumnData.Name) != false)
                {
                    list.Remove(headerColumnData);
                    list.Add(headerColumnData);
                }
            }

            //
            ListHeaderAdapterBase adapter = new CommonListHeaderAdapter();
            adapter.Columns = list.ToArray();
            CommentColumnTextApplier.Apply(adapter);
            this._ListCtrl.HeaderSource = adapter;

            if (editable != false)
            {
                this._ListCtrl.ReadOnly = false;
                this._ListCtrl.AttachedContextMenuStrip = this.listMenu;
            }
            else
            {
                this._ListCtrl.ReadOnly = true;
                this._ListCtrl.AttachedContextMenuStrip = null;
            }

            //
            UpdateResult();
            ClearTransactions();
            this.UpdatePresetListColumnsSplitButton();
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateResult()
        {
            _ListAdapter.InterlockComponent = false;
            _ListAdapter.Items.Clear();

            if (_ResultComponents != null)
            {
                Component[] components = _ResultComponents
                    .Where(c => IsMatch(c))
                    .ToArray();

                this._ListAdapter.AddItems(components);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ClearTransactions()
        {
            this.transactionPackStack.Clear();

            ShowTransactionStackWarning(false);
        }

        /// <summary>
        ///
        /// </summary>
        public void CommandExecuted(Command command)
        {
            //
            if (command == EditCommands.Paste)
            {
                this._ListCtrl.Invalidate();
            }

            //
            if (command == ProjectCommands.ShowProjectSettings)
            {
                UpdateToolBarColorTexts();
            }
        }

        /// <summary>
        /// プロジェクト設定によりカラム名が変更された時に呼ばれます。
        /// </summary>
        public void ApplyColumnsText()
        {
            CommentColumnTextApplier.Apply(this._ListCtrl.HeaderSource);
            this._ListCtrl.InvalidateHeader();
        }

        /// <summary>
        /// オプション設定が変更された時に呼ばれます。
        /// </summary>
        public void ApplyOptionSettings()
        {
            this._ListCtrl.ApplyOptionConfigurations();
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdatedOptions()
        {
            this._ListCtrl.UpdatePartDrawerOptions();
        }

        /// <summary>
        ///
        /// </summary>
        public void ApplySettings(CommonListPanel.SettingKinds kind)
        {
            switch (kind)
            {
                case CommonListPanel.SettingKinds.ColorComment:
                    CommandTextReplacer.ReplaceColorCommentCommandTexts
                        (ProjectService, this.listMenuAdapter);
                    break;
            }
        }

        /// <summary>
        /// プリセット名からをプリセットを適用します。
        /// </summary>
        public void ApplyPresetListColumns(string presetName)
        {
            string listName = ListTraits.ListName_FindResult;

            ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
            headerAdapters[listName] = _ListCtrl.HeaderSource as ListHeaderAdapterBase;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.ApplyPresetListColumnsWithoutAdd(applier, listName, presetName);

            _ListCtrl.UpdateInfos();

            MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = presetName;
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            preset.SetCurrentPresetName(listName, presetName);
        }

        /// <summary>
        /// プリセットを更新します。
        /// </summary>
        public void UpdatePresetListColumns()
        {
            this.UpdatePresetListColumnsSplitButton();
        }

        ///--------------------------------
        /// <summary>
        /// クリップボードが更新された時に呼ばれる
        /// </summary>
        public void UpdatedClipboard()
        {
            _ListCtrl.ClearDashLine();
        }

        ///--------------------------------
        /// <summary>
        /// 再描画します。
        /// </summary>
        public void RefreshPanel()
        {
            _ListCtrl.Invalidate();
        }
        /// <summary>
        /// 指定コマンドを実行できるコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            return (this.commandBindings.Contains(command.ID)) ? this : FindOtherTarget(command);
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (!this.commandBindings.Contains(command.ID)) { return CommandStatus.Unsupported; }
            return this.commandBindings[command.ID].QueryStatus(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        bool ICommandTarget.Execute(Command command)
        {
            if (!this.commandBindings.Contains(command.ID)) { return false; }
            bool result = this.commandBindings[command.ID].Execute(command);

            if (result == true && command == EditCommands.CopyCell)
            {
                _ListCtrl.SetDashLineBySelectedSubItem();
            }

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                case CommandParameterNames.TargetDocuments:
                case CommandParameterNames.TargetComponents:
                case "ColumnNames":
                case "Instance":
                    return true;
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                    return TargetComponentService;

                case CommandParameterNames.TargetDocuments:
                    return TargetDocuments;

                case CommandParameterNames.TargetComponents:
                    return TargetComponents;

                case "ColumnNames":
                    return this._ListCtrl.ItemsName
                        .Where(s => s != "RowHeader")
                        .ToArray();

                case "Instance":
                    return this;
            }
            throw new KeyNotFoundException();
        }

        /// <summary>
        /// アクティブになった時に呼ばれます。
        /// </summary>
        void IPanel.Activated()
        {
            SetSelectedItemsToParameterPanel(this._ListCtrl);
            ParameterPanel.SetFindResultPanel(this);
        }

        /// <summary>
        ///
        /// </summary>
        void IPanel.Deactivated()
        {
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelOperationExecutedListener.OperationExecuted(OperationExecutedEventArgs e)
        {
            SetToTransaction(e.Operation as SetParameterOperation);
        }

        /// <summary>
        /// 指定コマンドを実行できる他のコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        protected virtual ICommandTarget FindOtherTarget(Command command)
        {
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool ValidateValueForBatchEdit(Component component, string name, string text, object value)
        {
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual string ConvertText(string name, string text)
        {
            return text;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual IParameterValue ConvertValue(string name, string text, IParameterValue value)
        {
            return value;
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            //
            _toolStream.Image = ImageResource.BitmapIconStreamSound;
            this.tool_StreamSoundTrack.Image = ImageResource.BitmapIconStreamSoundTrack;
            _toolWaveSet.Image = ImageResource.BitmapIconWaveSoundSet;
            _toolWave.Image = ImageResource.BitmapIconWaveSound;
            _toolSequenceSet.Image = ImageResource.BitmapIconSequenceSoundSet;
            _toolSequence.Image = ImageResource.BitmapIconSequenceSound;
            _toolBank.Image = ImageResource.BitmapIconBank;
            _toolWaveArchive.Image = ImageResource.BitmapIconWaveArchive;
            _toolGroup.Image = ImageResource.BitmapIconGroup;
            _toolPlayer.Image = ImageResource.BitmapIconPlayer;
            //_toolInstrument.Image = ImageResource.BitmapIconInstrument;

            //
            UpdateToolBarColorTexts();

            //
            InitializeMenu();
            UpdatedOptions();
        }

        /// <summary>
        ///
        /// </summary>
        private SoundProjectService ProjectService
        {
            get
            {
                return FormsApplication.Instance.ProjectService;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditBegan(object sender, EventArgs e)
        {
            ApplicationBase.Instance.DisableCommandKeyProcess();
            MainWindow.EnableAllCommands(false);
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditEnded(object sender, EventArgs e)
        {
            ApplicationBase.Instance.EnableCommandKeyProcess();
            UpdateStatusBar(_ListCtrl);
            MainWindow.EnableAllCommands(true);
            MainWindow.RefreshPanels();
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateStatusBar(CommonListCtrl listCtrl)
        {
            CommonListItem[] items = null;
            string message = String.Empty;

            if (listCtrl != null)
            {
                items = listCtrl.GetSelectedItems();
                if (items.Length == 1)
                {
                    message = items[0].FilePath;
                }
            }

            MainWindow.UpdateStatusText(message);
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateToolBarColorTexts()
        {
            foreach (ButtonPair<int> pair in ButtonColorIndexPairs)
            {
                string text = String.Empty;

                if (ProjectService.Project != null)
                {
                    switch (pair.Value)
                    {
                        case 0: text = ProjectService.Project.ColorComment0; break;
                        case 1: text = ProjectService.Project.ColorComment1; break;
                        case 2: text = ProjectService.Project.ColorComment2; break;
                        case 3: text = ProjectService.Project.ColorComment3; break;
                        case 4: text = ProjectService.Project.ColorComment4; break;
                        case 5: text = ProjectService.Project.ColorComment5; break;
                        case 6: text = ProjectService.Project.ColorComment6; break;
                        case 7: text = ProjectService.Project.ColorComment7; break;
                        case 8: text = ProjectService.Project.ColorComment8; break;
                        default: Debug.Assert(false, "Illigal index"); break;
                    }
                }

                if (text == String.Empty)
                {
                    text = System.Text.RegularExpressions.Regex.Replace
                        (CommandTextReplacer.DefaultColorComments[pair.Value],
                          "\\(&.+\\)", String.Empty);
                }

                pair.Button.ToolTipText = text;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Bitmap CreateToolBarBitmap(Bitmap bitmap)
        {
            return bitmap.MakeNewTransparent(UIServiceBase.TransparentColor);
        }

        /// <summary>
        /// PanelBase クラスにコマンドを関連付けます。
        /// </summary>
        /// <param name="binding">コマンドバインディング。</param>
        private void BindCommand(string commandId, CommandBinding.QueryStatusHandler queryStatusHandler, CommandBinding.ExecuteHandler executeHandler)
        {
            CommandBinding binding = new CommandBinding
                (this, commandId, queryStatusHandler, executeHandler);
            if (null == binding) { throw new ArgumentNullException("binding"); }
            this.commandBindings.Add(binding);
        }

        ///
        private void BindCommand(CommandHandler commandHandler)
        {
            CommandBinding binding = new CommandBinding(this, commandHandler);
            if (null == binding) { throw new ArgumentNullException("binding"); }
            this.commandBindings.Add(binding);
        }

        /// <summary>
        ///
        /// </summary>
        private ComponentService TargetComponentService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        /// <summary>
        ///
        /// </summary>
        private Document[] TargetDocuments
        {
            get
            {
                SoundDocument[] documents = TargetComponents
                    .Select(c => GetDocument(c))
                    .Where(d => d != null)
                    .ToArray();
                return documents;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Component[] TargetComponents
        {
            get
            {
                return this._ListCtrl.GetItemSelecteds().Keys
                    .Cast<ComponentListItem>()
                    .Select(i => i.Target)
                    .ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ParameterPanel ParameterPanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        /// <summary>
        /// パラメータパネルに選択されているアイテムを設定します。
        /// </summary>
        private void SetSelectedItemsToParameterPanel(ListCtrl listCtrl)
        {
            List<Component> list = new List<Component>();
            List<Instrument> instList = new List<Instrument>();

            foreach (ComponentListItem item in listCtrl.SelectedItems)
            {
                if (item.Target is Sound || item.Target is StreamSoundTrack)
                {
                    list.Add(item.Target);
                }
                else if (item.Target is Instrument)
                {
                    instList.Add(item.Target as Instrument);
                }
            }

#if true
            if (list.Count > 0)
            {
                ParameterPanel.Items = list.ToArray();
                return;
            }
#endif

            CommonTabPage page = MainWindow.ActivePage;
            if (page != null)
            {
                CommonPanel panel = page.Panel;
                if (list.Count > 0)
                {
#if false
                    if (panel is SoundSetPanel == true)
                    {
                        Component current = (panel as SoundSetPanel).CurrentItem;
                        if (this.IsSameRoot(current, list[0]) == true)
                        {
                            ParameterPanel.Items = list.ToArray();
                            return;
                        }
                    }
#endif
                }
                else if (instList.Count == 1)
                {
                    if (panel is BankPanel == true)
                    {
                        Bank bank = (panel as BankPanel).Bank;
                        Instrument instrument = this.GetContainInstrument(bank, instList[0]);
                        if (instrument != null)
                        {
                            ParameterPanel.Instrument = instrument;
                            return;
                        }
                    }
                }
            }
            ParameterPanel.Items = null;
            ParameterPanel.Instrument = null;
        }

        private bool IsSameRoot(Component component, Component target)
        {
            bool contain = false;

            if (component != null && target != null)
            {
                while (component.Parent != null && component.Parent is SoundSet == false)
                {
                    component = component.Parent;
                }
                while (target.Parent != null && target.Parent is SoundSet == false)
                {
                    target = target.Parent;
                }
                if (component == target)
                {
                    contain = true;
                }
            }

            return contain;
        }

        private Instrument GetContainInstrument(Bank bank, Instrument target)
        {
            Instrument instrument = null;

            if (bank != null)
            {
                if (bank.Children != null && 0 < bank.Children.Count)
                {
                    foreach (Component inst in bank.Children)
                    {
                        if (inst.Name == target.Name)
                        {
                            instrument = inst as Instrument;
                            break;
                        }
                    }
                }
            }

            return instrument;
        }

        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (this.listMenuAdapter != null)
            {
                return;
            }

            this.listMenuAdapter = ToolStripAdapter.FromToolStrip
                (FormsApplication.Instance.CommandService, this.listMenu);
            this.listMenuAdapter.CommandTarget = FormsApplication.Instance;
            this.listMenuAdapter.CommandExecuted += OnCommandExecuted;

            //ApplySettings( SettingKinds.ColorComment);
            BuildListCommandUI();
        }

        /// <summary>
        // ストリームサウンドが含まれていた場合にはストリームサウンドトラックを追加します。
        /// </summary>
        private Component[] GetComponentsAtRelevanceComponent(Component[] components)
        {
            HashSet<Component> hashSet = new HashSet<Component>();
            List<Component> list = new List<Component>();

            foreach (Component component in components)
            {
                if (hashSet.Contains(component) == false)
                {
                    list.Add(component);
                    hashSet.Add(component);
                }

                //
                if (component is StreamSound)
                {
                    foreach (Component childComponent in component.Children)
                    {
                        if (hashSet.Contains(childComponent) == false)
                        {
                            list.Add(childComponent);
                            hashSet.Add(childComponent);
                        }
                    }
                }
            }
            return list.ToArray();
        }

        /// <summary>
        /// コマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnCommandExecuted(object sender, CommandEventArgs e)
        {
            //CommandExecuted(e.Command);
            this._ListCtrl.Invalidate();

            MainWindow.RequestNotifyCommandExecuted(e.Command);
            MainWindow.BuildCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUIs()
        {
            BuildListCommandUI();
            MainWindow.BuildCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildListCommandUI()
        {
            if (this.listMenuAdapter != null)
            {
                this.listMenuAdapter.BuildUI();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Instrument[] GetInstrument(Component[] components)
        {
            return components
                .Where(c => c is Instrument)
                .Cast<Instrument>()
                .ToArray();
        }

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        private void InitializeCommandBindings()
        {
            BindCommand(EditCommands.Undo.ID,
                         QueryStatusUndo, ExecuteUndo);
            BindCommand(EditCommands.Redo.ID,
                         QueryStatusRedo, ExecuteRedo);

            BindCommand(new OpenItemFolderHandler(this));
            BindCommand(new ExecuteItemFolderHandler(this));
            BindCommand(new OpenBankItemHandler(this));

            BindCommand(new CopyHandler(this));
            BindCommand(EditCommands.CopyCell.ID,
                         QueryStatusCopyCell, ExecuteCopyCell);
            BindCommand(EditCommands.Paste.ID,
                         QueryStatusPaste, ExecutePaste);

            BindCommand(PreviewCommands.Play.ID,
                         QueryStatusPlay, ExecutePlay);

            BindCommand(EditCommands.SelectAll.ID,
                         QueryStatusSelectAll, ExecuteSelectAll);

            BindCommand(EditCommands.EnableItem.ID,
                         QueryStatusEnableItem, ExecuteEnableItem);
            BindCommand(EditCommands.DisableItem.ID,
                         QueryStatusDisableItem, ExecuteDisableItem);
            BindCommand(EditCommands.SetItemColor.ID,
                         QueryStatusSetItemColor, ExecuteSetItemColor);

            BindCommand(EditCommands.BatchEdit.ID,
                         QueryStatusBatchEdit, ExecuteBatchEdit);
        }

        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusUndo(Command command)
        {
            if (this._ListCtrl.ReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return (this.transactionPackStack.CanUndo() != false ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusRedo(Command command)
        {
            if (this._ListCtrl.ReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return (this.transactionPackStack.CanRedo() != false ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExecuteUndo(Command command)
        {
            this.transactionPackStack.Undo();
            this._ListCtrl.Invalidate();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExecuteRedo(Command command)
        {
            this.transactionPackStack.Redo();
            this._ListCtrl.Invalidate();
            return true;
        }

        /// <summary>
        /// プレビュー再生ができる Componentを取得します。
        /// </summary>
        private Component[] GetPreviewPlayTargetComponents()
        {
            return TargetComponents
                .Where(c => c is Sound)
                .ToArray();
        }

        /// <summary>
        /// "プレビュー再生"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusPlay(Command command)
        {
            return GetPreviewPlayTargetComponents().Length == 1 ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "プレビュー再生"を実行します。
        /// </summary>
        private bool ExecutePlay(Command command)
        {
            Component[] components = GetPreviewPlayTargetComponents();

            if (components.Length == 1)
            {
                ExecutePreviewPlay(components[0]);
            }
            return true;
        }

        /// <summary>
        /// "すべて選択"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusSelectAll(Command command)
        {
            return CanSelectAll() ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "すべて選択"を実行します。
        /// </summary>
        private bool ExecuteSelectAll(Command command)
        {
            SelectAll();
            return true;
        }

        /// <summary>
        /// カラー設定ができる Componentを取得します。
        /// </summary>
        private Component[] GetSetToItemColorTargetComponents()
        {
            return TargetComponents
                .Where(c => !(c is GroupItem))
                .ToArray();
        }

        /// <summary>
        /// "アイテムカラー"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusSetItemColor(Command command)
        {
            if (this._ListCtrl.ReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanSetItemColor() != false ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "アイテムカラー"を実行します。
        /// </summary>
        private bool ExecuteSetItemColor(Command command)
        {
            IQueryCommandParameter parameters = command.Parameters;
            if (parameters.ContainsParameter(CommandParameterNames.ColorIndex) == false)
            {
                return false;
            }

            try
            {
                int value = int.Parse
                    ((string)parameters.GetParameter(CommandParameterNames.ColorIndex));
                SetItemColor(GetSetToItemColorTargetComponents(), value);
                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// 有効/無効にできる Componentを取得します。
        /// </summary>
        private Component[] GetSetToEnableTargetComponents()
        {
            return TargetComponents
                .ToArray();
        }

        /// <summary>
        /// "アイテムを有効にする"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusEnableItem(Command command)
        {
            if (this._ListCtrl.ReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanEnableItem(true) != false ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "アイテムを有効にする"を実行します。
        /// </summary>
        private bool ExecuteEnableItem(Command command)
        {
            EnableItem(GetSetToEnableTargetComponents(), true);
            return true;
        }

        /// <summary>
        /// "アイテムを無効にする"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusDisableItem(Command command)
        {
            if (this._ListCtrl.ReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanEnableItem(false) != false ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "アイテムを無効にする"を実行します。
        /// </summary>
        private bool ExecuteDisableItem(Command command)
        {
            EnableItem(GetSetToEnableTargetComponents(), false);
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private class BatchEditItemInfo
        {
            private List<string> names = new List<string>();

            public ComponentListItem Item { get; set; }
            public List<string> Names
            {
                get
                {
                    return this.names;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private class BatchEditInfo : List<BatchEditItemInfo>
        {
            private SoundSet soundSet = null;
            private SoundSetDocument soundSetDocument = null;

            public BatchEditInfo(SoundSet soundSet, SoundSetDocument soundSetDocument)
            {
                this.soundSet = soundSet;
                this.soundSetDocument = soundSetDocument;
            }

            public SoundSet SoundSet
            {
                get
                {
                    return this.soundSet;
                }
            }
            public SoundSetDocument SoundSetDocument
            {
                get
                {
                    return this.soundSetDocument;
                }
            }
        }

        /// <summary>
        /// "一括編集"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusBatchEdit(Command command)
        {
            if (this._ListCtrl.ReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanBatchEdit() ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "一括編集"が可能なのか調べます。
        /// </summary>
        private bool CanBatchEdit()
        {
            ListItemSelectedDictionary selectedItems = this._ListCtrl.GetItemSelecteds();
            bool can = false;

            foreach (ComponentListItem item in selectedItems.Keys)
            {
                if (item.Target is GroupItem)
                {
                    return false;
                }

                ListItemSelectedState state = selectedItems[item];
                if (state.SubSelected.Count > 0)
                {
                    can = true;
                }
            }
            return can;
        }

        /// <summary>
        /// "一括編集"の実行
        /// </summary>
        private bool ExecuteBatchEdit(Command command)
        {
            // 一括編集の対象の情報を作成します。
            Dictionary<SoundSet, BatchEditInfo> dictionary =
                new Dictionary<SoundSet, BatchEditInfo>();

            ListItemSelectedDictionary selectedItems = this._ListCtrl.GetItemSelecteds();
            ListItemSelectedState state = null;

            foreach (ComponentListItem item in selectedItems.Keys)
            {
                BatchEditItemInfo itemInfo = new BatchEditItemInfo();
                itemInfo.Item = item;

                state = selectedItems[item];
                foreach (string name in state.SubSelected)
                {
                    itemInfo.Names.Add(name);
                }

                //
                SoundSet soundSet = GetSoundSet(item.Target);
                if (dictionary.ContainsKey(soundSet) == false)
                {
                    SoundSetDocument soundSetDocument = GetSoundSetDocument(soundSet);
                    dictionary.Add(soundSet, new BatchEditInfo(soundSet, soundSetDocument));
                }

                BatchEditInfo info = dictionary[soundSet];
                info.Add(itemInfo);
            }

            //
            BatchEditDialog dialog = null;

            dialog = new BatchEditDialog();
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            if (dialog.ValueText == String.Empty)
            {
                return false;
            }

            //
            {
                List<UserTransaction2> list = new List<UserTransaction2>();

                foreach (KeyValuePair<SoundSet, BatchEditInfo> pair in dictionary)
                {
                    UserTransaction2 transaction = new UserTransaction2(String.Empty);

                    SoundSetDocument soundSetDocument = GetSoundSetDocument(pair.Key);
                    Operation[] operations = BatchEdit2
                        (dialog.EffectKind, dialog.ValueText,
                          soundSetDocument, pair.Value);

                    if (operations.Length > 0)
                    {
                        foreach (Operation operation in operations)
                        {
                            transaction.Operations.Add(operation);
                        }

                        //transaction.SoundSetDocuments.Add( soundSetDocument);
                        transaction.Document = soundSetDocument;
                        soundSetDocument.OperationHistory.AddTransaction(transaction);

                        list.Add(transaction);
                    }
                }

                PushTransactionPackStack(list.ToArray());
            }

            return true;
        }

        /// <summary>
        /// "貼り付け"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusPaste(Command command)
        {
            if (this._ListCtrl.ReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return (CanPaste() ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "貼り付け"を実行します。
        /// </summary>
        private bool ExecutePaste(Command command)
        {
            if (Paste() == false)
            {
                return false;
            }

            //ProjectService.SendUpdateSoundProjectItemBindingEvent();
            return true;
        }

        /// <summary>
        /// "セルのコピー"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusCopyCell(Command command)
        {
            bool result = ClipboardService.CanCopyCell
                (this._ListCtrl.ItemsName,
                  this._ListCtrl.GetItemSelecteds());

            return (result != false ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "セルのコピー"を実行します。
        /// </summary>
        private bool ExecuteCopyCell(Command command)
        {
            ClipboardService.CopyCell(this._ListCtrl.ItemsName,
                                       this._ListCtrl.GetItemSelecteds());
            return true;
        }

        /// <summary>
        /// プレビュー再生を行います。
        /// </summary>
        private void ExecutePreviewPlay(Component component)
        {
            if (PreviewPlay != null)
            {
                PreviewPlay(component, new EventArgs());
            }
        }

        /// <summary>
        /// オペレーションが実行された後に呼ばれます。
        /// </summary>
        private void OnOperationExecuted(object sender, OperationExecutedEventArgs e)
        {
            SetToTransaction(e.Operation as SetParameterOperation);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetToTransaction(SetParameterOperation operation)
        {
            if (operation == null)
            {
                return;
            }

            Component component = FindComponent(operation.TargetDictionary,
                                                 operation.ParameterKey);
            SoundSet soundSet = GetSoundSet(component);

            SoundSetDocument soundSetDocument =
                FormsApplication.Instance.DocumentService.Documents
                .Where(d => d is SoundSetDocument)
                .Cast<SoundSetDocument>()
                .Where(d => d.SoundSet == soundSet)
                .First();

            UserTransaction2 transaction = new UserTransaction2(String.Empty);
            transaction.Document = soundSetDocument;

            soundSetDocument.OperationHistory.BeginTransaction(transaction);
            soundSetDocument.OperationHistory.AddOperation(operation);
            soundSetDocument.OperationHistory.EndTransaction();

            //
            PushTransactionPackStack(new UserTransaction2[] { transaction });

            //
            MainWindow.RefreshPanels(soundSetDocument);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOpeningListMenu(object sender, System.ComponentModel.CancelEventArgs e)
        {
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlay(object sender, EventArgs e)
        {
            Component component = sender as Component;
            ExecutePreviewPlay(component);
        }

        /// <summary>
        ///
        /// </summary>
        private void PushTransactionPackStack(UserTransaction2[] transactions)
        {
            if (transactions.Length <= 0)
            {
                return;
            }

            TransactionPack pack = new TransactionPack(transactions);
            this.transactionPackStack.Push(pack);

            ShowTransactionStackWarning(true);
        }

        /// <summary>
        /// 指定した ParameterDictionaryを保持する Componentをリストから検索します。
        /// </summary>
        private Component FindComponent(IParameterDictionary parameterDictionary, string name)
        {
            foreach (ComponentListItem item in this._ListAdapter.Items)
            {
                if (item.Target.Parameters == parameterDictionary)
                {
                    return item.Target;
                }
            }

            // リストの Component.Parametersに該当しない場合には、指定パラメータ名によって
            // 検索する場所を変えて、再度、検索を行います。
            switch (name)
            {
                case ProjectParameterNames.TargetName:
                    // 参照サンドセットバンク
                    {
                        foreach (ComponentListItem item in this._ListAdapter.Items)
                        {
                            ObservableList<ComponentReference> references =
                                item.Target.Parameters[ProjectParameterNames.SequenceSound.SoundSetBankReferences].Value as ObservableList<ComponentReference>;

                            if (references != null)
                            {
                                foreach (ComponentReference reference in references)
                                {
                                    if (reference.Parameters == parameterDictionary)
                                    {
                                        return item.Target;
                                    }
                                }
                            }
                        }
                    }
                    break;
            }

            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSet GetSoundSet(Component component)
        {
            while (component.Parent != null)
            {
                if (component.Parent is SoundSet)
                {
                    return component.Parent as SoundSet;
                }
                component = component.Parent;
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private SoundDocument GetDocument(Component component)
        {
            // 現状は SoundSetDocumentのみ対応します。
            if (component is Instrument)
            {
                return null;
            }

            return GetSoundSetDocument(GetSoundSet(component));
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSetDocument GetSoundSetDocument(SoundSet soundSet)
        {
            SoundSetDocument soundSetDocument =
                FormsApplication.Instance.DocumentService.Documents
                .Where(d => d is SoundSetDocument)
                .Cast<SoundSetDocument>()
                .Where(d => d.SoundSet == soundSet)
                .First();
            return soundSetDocument;
        }

        /// <summary>
        ///
        /// </summary>
        private Operation[] BatchEdit2(EffectKind effectKind, string valueText, SoundSetDocument soundSetDocument, BatchEditInfo info)
        {
            List<Operation> list = new List<Operation>();

            foreach (BatchEditItemInfo itemInfo in info)
            {
                foreach (string name in itemInfo.Names)
                {
                    Operation[] operations = BatchEditUtility.SetEffect
                        (itemInfo.Item, name, effectKind, valueText,
                          (BatchEditUtility.ConvertTextHandler)
                          delegate (string n, string text)
                              {
                                  return ConvertText(n, text);
                              },
                          (BatchEditUtility.ConvertValueHandler)
                          delegate (string n, string text, IParameterValue value)
                              {
                                  return ConvertValue(n, text, value);
                              },
                          (BatchEditUtility.ValidateHandler)
                          delegate (Component component, string n, string text, object value)
                              {
                                  return ValidateValueForBatchEdit
                                  (component, n, text, value);
                              });

                    if (operations != null)
                    {
                        list.AddRange(operations);
                    }
                }
            }
            return list.ToArray();
        }

#if false
        /// <summary>
        ///
        /// </summary>
        private Transaction BatchEdit( EffectKind effectKind, string valueText, SoundSetDocument soundSetDocument, BatchEditInfo info)
        {
            OperationHistory operationHistory = soundSetDocument.OperationHistory;
            Operation operation = null;

            UserTransaction transaction = new UserTransaction();

            foreach( BatchEditItemInfo itemInfo in info )
            {
                foreach( string name in itemInfo.Names )
                {
                    if((operation = SetEffect
                        ( itemInfo.Item, name, effectKind, valueText)) != null )
                    {
                        operationHistory.AddOperation(operation);
                    }
                }
            }
            return transaction;
        }


        /// <summary>
        ///
        /// </summary>
        private Operation SetEffect(ComponentListItem targetItem, string name, EffectKind effectKind, string text)
        {
            Component component = null;
            Operation operation = null;
            IParameterValue value = null;
            decimal sourceValue;
            decimal destValue;
            string answerText = null;
            ValidationResult result = null;
            object parsedValue = null;
            CommonListItem item = targetItem as CommonListItem;

            Debug.Assert(item != null, "Item is not CommonListItem");
            component = item.GetTargetByName(name);

            if (component == null ||
                component.Parameters.ContainsKey(name) == false)
            {
                return null;
            }

            if (component is GroupItem)
            {
                return null;
            }

            try
            {
                value = item.GetValue(name);

                text = ConvertText(name, text);
                value = ConvertValue(name, text, value);

                //
                switch (effectKind)
                {
                    case EffectKind.Set:
                        answerText = text;
                        parsedValue = value.ParseValue(answerText);
                        break;

                    case EffectKind.Addition:
                        sourceValue = decimal.Parse(value.ToString());
                        destValue = decimal.Parse(text);
                        answerText = (sourceValue + destValue).ToString();
                        parsedValue = value.ParseValue(answerText);
                        break;

                    case EffectKind.Multiplication:
                        sourceValue = decimal.Parse(value.ToString());
                        destValue = decimal.Parse(text);
                        answerText = (sourceValue * destValue).ToString();

                        try { parsedValue = value.ParseValue(answerText); }
                        catch
                        {
                            answerText = answerText.Substring(0, answerText.IndexOf("."));
                            parsedValue = value.ParseValue(answerText);
                        }
                        break;
                }

                //
                if (name == "OriginalKey")
                {
                    answerText = parsedValue.ToString();
                    parsedValue = text;
                }

                //
                if( ValidateValueForBatchEdit( component, name, answerText, parsedValue) == false )
                {
                    return null;
                }

                //
                result = value.ValidateValue(parsedValue);
                if (result.IsValid == false)
                {
                    return null;
                }

                //
                switch( name )
                {
                case ProjectParameterNames.SequenceSound.SoundSetBankReferences:
                    ObservableList<ComponentReference> reference =
                        (ObservableList<ComponentReference>)( component.Parameters[name].Value);
                    operation = new SetParameterOperation
                        ( reference[0].Parameters, ProjectParameterNames.TargetName,
                          (object)answerText);
                    break;

                default:
                    operation = new SetParameterOperation
                        ( component.Parameters, name, answerText);
                    break;
                }
                bool executeResult = operation.Execute();

                if (name == ProjectParameterNames.SndEdit && executeResult == true)
                {
                    if (parsedValue is bool == true)
                    {
                        if ((bool)parsedValue == true)
                        {
                            FormsApplication.Instance.RealtimeEditService.AddMonitoringTarget(component);
                        }
                        else
                        {
                            FormsApplication.Instance.RealtimeEditService.RemoveMonitoringTarget(component);
                        }
                    }
                }
            }

            catch { operation = null; }
            return operation;
        }
#endif

        /// <summary>
        /// "貼り付け"が実行可能なのか調べる
        /// </summary>
        private bool CanPaste()
        {
            //セルの貼り付けが可能なのか？
            if (ClipboardService.CanPasteCell(this._ListCtrl.ItemsName,
                                               this._ListCtrl.GetItemSelecteds()) != false)
            {
                return true;
            }

            //貼り付けが可能なのか？
            return ClipboardService.CanPaste(this._ListAdapter.ItemsOwner);
        }

        /// <summary>
        /// "貼り付け"の実行
        /// </summary>
        private bool Paste()
        {
            // 貼り付けが可能か？
            if (ClipboardService.CanPaste(this._ListAdapter.ItemsOwner) != false)
            {
                // 検索結果リストでは貼り付けはできません。
                return true;
            }

            // セルへの貼り付けが可能か？
            if (ClipboardService.CanPasteCell(this._ListCtrl.ItemsName,
                                               this._ListCtrl.GetItemSelecteds()) != false)
            {
                //
                TransactionDictionary dictionary =
                    ClipboardService.PasteCellForFind(this._ListCtrl.ItemsName,
                                                       this._ListAdapter.Items.ToArray(),
                                                       this._ListCtrl.GetItemSelecteds());
                if (dictionary == null)
                {
                    return false;
                }

                List<UserTransaction2> list = new List<UserTransaction2>();
                foreach (KeyValuePair<Document, UserTransaction2> pair in dictionary)
                {
                    UserTransaction2 transaction = pair.Value;
                    Document document = pair.Key;
                    transaction.Document = document;
                    document.OperationHistory.AddTransaction(transaction);

                    list.Add(transaction);
                }

                if (list.Count > 0)
                {
                    PushTransactionPackStack(list.ToArray());
                }

                return true;
            }

            return false;
        }

        /// <summary>
        /// "すべて選択"が実行可能か調べます。
        /// </summary>
        private bool CanSelectAll()
        {
            return this._ListCtrl != null ? true : false;
        }

        /// <summary>
        /// "すべて選択"を実行します。
        /// </summary>
        private void SelectAll()
        {
            if (CanSelectAll() == false)
            {
                return;
            }

            this._ListCtrl.SelectAll();
        }

        /// <summary>
        /// アイテムカラーを実行できるのか調べます。
        /// </summary>
        private bool CanSetItemColor()
        {
            return GetSetToItemColorTargetComponents().Length > 0 ? true : false;
        }

        /// <summary>
        /// アイテムカラーを実行します。
        /// </summary>
        private void SetItemColor(Component[] components, int value)
        {
            if (CanSetItemColor() == false)
            {
                return;
            }

            ExecuteOperation
                (components,
                  delegate (Component component)
                      {
                          Operation operation = new SetParameterOperation
                              (component.Parameters, ProjectParameterNames.ColorIndex, value);
                          return operation;
                      });
        }

        /// <summary>
        /// アイテムを有効/無効にできるのか調べます。
        /// </summary>
        private bool CanEnableItem(bool value)
        {
            // valueを引数として受け取っていますが、有効/無効で違いが無いので
            // 今は使っていません。
            return GetSetToEnableTargetComponents().Length > 0 ? true : false;
        }

        /// <summary>
        /// アイテムを有効/無効にします。
        /// </summary>
        private void EnableItem(Component[] components, bool value)
        {
            if (CanEnableItem(value) == false)
            {
                return;
            }

            ExecuteOperation
                (components,
                  delegate (Component component)
                      {
                          Operation operation = new SetParameterOperation
                              (component.Parameters, ProjectParameterNames.IsEnabled, value);
                          return operation;
                      });
        }

        /// <summary>
        ///
        /// </summary>
        private delegate Operation GetOperationHandler(Component component);

        /// <summary>
        ///
        /// </summary>
        private void ExecuteOperation(Component[] components, GetOperationHandler handler)
        {
            //
            TransactionDictionary dictionary = new TransactionDictionary();
            foreach (Component component in components)
            {
                Document document = GetDocument(component);
                if (dictionary.ContainsKey(document) == false)
                {
                    dictionary.Add(document, new UserTransaction2(String.Empty));
                }
            }

            //
            foreach (Component component in components)
            {
                Document document = GetDocument(component);
                UserTransaction2 transaction = dictionary[document];

                Operation operation = handler(component);
                operation.Execute();

                transaction.Operations.Add(operation);
            }

            //
            foreach (KeyValuePair<Document, UserTransaction2> pair in dictionary)
            {
                Document document = pair.Key;
                UserTransaction2 transaction = pair.Value;

                transaction.Document = document;
                document.OperationHistory.AddTransaction(transaction);
            }

            //
            UserTransaction2[] transactions = dictionary.Values
                .ToArray();
            if (transactions.Length > 0)
            {
                PushTransactionPackStack(transactions);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool ContainsHeaderColumn(List<HeaderColumnData> list, string name)
        {
            foreach (HeaderColumnData column in list)
            {
                if (column.Name == name)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExcludeHeaderColumn(string name)
        {
            return this.excludeColumnNames.Contains(name);
        }

        /// <summary>
        ///
        /// </summary>
        private void AddHeaderColumnData(List<HeaderColumnData> list, string name)
        {
            ListHeaderAdapterBase adapter =
                FormsApplication.Instance.UIService.DefaultHeaderAdapters[name];

            foreach (HeaderColumnData column in adapter.Columns)
            {
                if (ExcludeHeaderColumn(column.Name) == false &&
                    ContainsHeaderColumn(list, column.Name) == false)
                {
                    list.Add(column);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private HeaderColumnData GetHeaderColumnData(string name, string columnName)
        {
            ListHeaderAdapterBase adapter =
                FormsApplication.Instance.UIService.DefaultHeaderAdapters[name];

            foreach (HeaderColumnData column in adapter.Columns)
            {
                if (column.Name == columnName)
                {
                    return column;
                }
            }
            return null;
        }


        ///--------------------------------
        /// <summary>
        /// ファイルがドロップできるかどうか判断する時に呼ばれる
        /// </summary>
        private void OnQueryFileDropped(object sender, FileDroppedEventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDropped(object sender, FileDroppedEventArgs e)
        {
            MainWindow.DropFiles(e.FilePaths);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSetAllButtonCheck(object sender, EventArgs e)
        {
            SetButtonCheck(true);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnResetAllButtonCheck(object sender, EventArgs e)
        {
            SetButtonCheck(false);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetButtonCheck(bool b)
        {
            _filterUpdateLocked = true;

            _toolStream.Checked = b;
            this.tool_StreamSoundTrack.Checked = b;
            _toolWaveSet.Checked = b;
            _toolWave.Checked = b;
            _toolSequenceSet.Checked = b;
            _toolSequence.Checked = b;
            _toolBank.Checked = b;
            _toolWaveArchive.Checked = b;
            _toolGroup.Checked = b;
            _toolPlayer.Checked = b;

            this.tool_ColorNone.Checked = b;
            this.tool_ColorRed.Checked = b;
            this.tool_ColorOrange.Checked = b;
            this.tool_ColorYellow.Checked = b;
            this.tool_ColorGreen.Checked = b;
            this.tool_ColorSkyBlue.Checked = b;
            this.tool_ColorBlue.Checked = b;
            this.tool_ColorPurple.Checked = b;
            this.tool_ColorPink.Checked = b;

            _filterUpdateLocked = false;

            UpdateFilter();
        }

        /// <summary>
        ///
        /// </summary>
        private void SetItemButtonEnabled(bool b)
        {
            this._toolStream.Enabled = b;
            this.tool_StreamSoundTrack.Enabled = b;
            this._toolWaveSet.Enabled = b;
            this._toolWave.Enabled = b;
            this._toolSequenceSet.Enabled = b;
            this._toolSequence.Enabled = b;
            this._toolBank.Enabled = b;
            this._toolWaveArchive.Enabled = b;
            this._toolGroup.Enabled = b;
            this._toolPlayer.Enabled = b;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnFilterCheckedChanged(object sender, EventArgs e)
        {
            UpdateFilter();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnProjectClosed(object sender, EventArgs e)
        {
            SetResult(null);
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowTransactionStackWarning(bool visible)
        {
            this.statusStrip.Visible = visible;
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateFilter()
        {
            if (_filterUpdateLocked)
            {
                return;
            }

            UpdateResult();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private class ButtonPair<T>
        {
            private ToolStripButton button = null;
            private T value;

            public ButtonPair(ToolStripButton button, T value)
            {
                this.button = button;
                this.value = value;
            }

            public ToolStripButton Button
            {
                get { return this.button; }
            }

            public T Value
            {
                get { return this.value; }
            }

        }

        ///
        private class ButtonTypePair : ButtonPair<Type>
        {
            public ButtonTypePair(ToolStripButton button, Type type) : base(button, type)
            {
            }
        }

        ///
        private bool IsMatch(Component component)
        {
            ButtonTypePair[] buttonTypePair = new ButtonTypePair[] {
                new ButtonTypePair( _toolStream,      typeof(StreamSound)),
                new ButtonTypePair( this.tool_StreamSoundTrack, typeof(StreamSoundTrack)),
                new ButtonTypePair( _toolWaveSet,     typeof(WaveSoundSet)),
                new ButtonTypePair( _toolWave,        typeof(WaveSound)),
                new ButtonTypePair( _toolSequenceSet, typeof(SequenceSoundSet)),
                new ButtonTypePair( _toolSequence,    typeof(SequenceSound)),
                new ButtonTypePair( _toolBank,        typeof(SoundSetBank)),
                new ButtonTypePair( _toolWaveArchive, typeof(WaveArchive)),
                new ButtonTypePair( _toolGroup,       typeof(Group)),
                new ButtonTypePair( _toolPlayer,      typeof(Player)),
                //new ButtonTypePair( _toolInstrument,  typeof(ImaginaryInstrument)),
            };

            //
            if (this.tool_EnabledItem.Checked != false &&
                component.IsEnabled == false)
            {
                return false;
            }

            //
            foreach (ButtonPair<int> pair in ButtonColorIndexPairs)
            {
                if (pair.Button.Checked == false &&
                    component.ColorIndex == pair.Value)
                {
                    return false;
                }
            }

            //
            if (component is Instrument)
            {
                return true;
            }

            // アイテムの種類によるフィルタ処理です。
            foreach (ButtonTypePair pair in buttonTypePair)
            {
                if (pair.Button.Checked != false &&
                    HasAncestor(pair.Value, component.GetType()) != false)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private bool HasAncestor(Type ancestor, Type target)
        {
            do
            {
                if (ancestor == target)
                {
                    return true;
                }
                target = target.BaseType;
            } while (target != null);
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get
            {
                return FormsApplication.Instance.UIService.MainWindow;
            }
        }

        /// <summary>
        /// リストの選択状態が変更された時に呼ばれます
        /// </summary>
        private void OnSelectChanged(object sender, EventArgs e)
        {
            MainWindow.BuildCommandUI();
            SetSelectedItemsToParameterPanel(_ListCtrl);
            UpdateStatusBar(_ListCtrl);
            ParameterPanel.SetFindResultPanel(this);
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnListItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ComponentListItem item = e.Item as ComponentListItem;
            Component component = item.Target;

            if (item is ImaginaryInstrumentListItem == true)
            {
                ImaginaryInstrumentListItem instItem = item as ImaginaryInstrumentListItem;
                component = instItem.ImaginaryInstrument;
            }

            MainWindow.ShowPageByComponent(component);
        }

        ///--------------------------------
        /// <summary>
        /// リスト項目関連
        /// </summary>
        private void OnHeaderMenuItemAddPresetListColumnsClick(object sender, EventArgs e)
        {
            string listName = ListTraits.ListName_FindResult;
            ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
            headerAdapters[listName] = _ListCtrl.HeaderSource as ListHeaderAdapterBase;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.AddPresetListColumns(listName, applier);
            this.UpdatePresetListColumnsSplitButton();
        }

        private void OnHeaderMenuItemApplyPresetListColumnsClick(object sender, EventArgs e)
        {
            ToolStripMenuItem menuItem = sender as ToolStripMenuItem;
            string listName = ListTraits.ListName_FindResult;
            ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
            headerAdapters[listName] = _ListCtrl.HeaderSource as ListHeaderAdapterBase;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.ApplyPresetListColumnsWithoutAdd(applier, listName, menuItem.Text);
            _ListCtrl.UpdateInfos();
        }

        private void OnHeaderCtrlContextMenuOpening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            bool enabled = (_ListCtrl.HeaderSource as ListHeaderAdapterBase).Columns.Length > 1;
            _HeaderContextMenuItem_AddPresetListColumns.Enabled = enabled;
            _HeaderContextMenuItem_ApplyPresetListColumns.Enabled = enabled;

            if (enabled == true)
            {
                string listName = ListTraits.ListName_FindResult;

                ListHeaderHelper.CreateMeunApplyPresetListColumns(_HeaderContextMenuItem_ApplyPresetListColumns, listName, OnHeaderMenuItemApplyPresetListColumnsClick);
            }
        }

        private void OnEnter(object sender, EventArgs e)
        {
            this.UpdatePresetListColumnsSplitButton();
        }

        private void OnLeave(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        private void UpdatePresetListColumnsSplitButton()
        {
            if (this.Active == true || _ListCtrl.HeaderSource.Items.Count <= 1)
            {
                MainWindow.ClearPresetListColumnsOnSplitButton();
            }
            else
            {
                string listName = ListTraits.ListName_FindResult;
                PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
                MainWindow.PresetListColumnsPresetNamesOnSplitButton = preset.GetPresetNames(listName);
                MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = preset.GetCurrentPresetName(listName);
            }
        }

        private void OnDockStateChanged(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        private void OnDockVisibleChanged(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        private void OnActiveChanged(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        private void OnWindowActiveChanged(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }
    }

    /// <summary>
    /// 編集不可能セルの描画を行うクラスです。
    /// </summary>
    public class NAListPartDrawer : ListPartDrawer
    {
        private static Brush DisableTextBrush = SystemBrushes.GrayText;
        private static StringFormat Format = new StringFormat();
        //private static Color BackgroundColor;

        /// <summary>
        ///
        /// </summary>
        static NAListPartDrawer()
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DrawInternal(ListDrawDescriptor desc)
        {
            ListColumnStyle style = desc.ColumnStyle;

            switch (style.Name)
            {
                case ProjectParameterNames.Sound.ActorPlayer:
                    Format.LineAlignment = StringAlignment.Far;
                    Format.Alignment = StringAlignment.Far;
                    break;

                default:
                    if (style.SubItemType == typeof(string))
                    {
                        Format.LineAlignment = StringAlignment.Far;
                        Format.Alignment = StringAlignment.Near;
                    }
                    else if (style.SubItemType == typeof(bool))
                    {
                        Format.LineAlignment = StringAlignment.Far;
                        Format.Alignment = StringAlignment.Center;
                    }
                    else
                    {
                        Format.LineAlignment = StringAlignment.Far;
                        Format.Alignment = StringAlignment.Far;
                    }
                    break;
            }

            desc.ContentDrawer.DrawString
                (desc, NotAvailable.Text,
                  desc.Font, DisableTextBrush, desc.Bounds, Format);
        }
    }
}
