﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Profile;

    public delegate bool IsValidateWaveEncodingValueDelegate(WaveEncoding waveEncoding);

    /// <summary>
    ///
    /// </summary>
    public partial class SampleMapPanel : UserControl, IInnerBankPanel, SampleMapCtrl.IContextMenuHandler
    {
        private WaveEncodingItem[] waveEncodingItems = { new WaveEncodingItem(WaveEncoding.Adpcm),
                                                         new WaveEncodingItem(WaveEncoding.Pcm16),
                                                         new WaveEncodingItem(WaveEncoding.Pcm8), };
        private WaveEncodingItem[] _waveEncodingItemsWithoutPcm8 = { new WaveEncodingItem(WaveEncoding.Adpcm),
                                                                     new WaveEncodingItem(WaveEncoding.Pcm16), };

        private SampleMapAdapter _SampleMapAdapter = null;
        private Instrument _Instrument = null;
        private BankPanel _BankPanel = null;
        private bool _ReadOnly = false;
        private int _oldIndex = 0;

        public event OperationExecutedEventHandler OperationExecuted;
        public event NoteEventHandler NoteOn;
        public event NoteEventHandler NoteOff;
        public IsValidateWaveEncodingValueDelegate IsValidateWaveEncodingValueDelegate;

        /// <summary>
        ///
        /// </summary>
        public SampleMapPanel(BankPanel bankPanel)
        {
            InitializeComponent();

            _BankPanel = bankPanel;
            _SampleMapAdapter = new SampleMapAdapter();
            _SampleMapAdapter.Setters.Add(ProjectParameterNames.FilePath, this.FilePathSetter);
            _SampleMapAdapter.OperationHistory = _BankPanel.BankDocument.OperationHistory;

            _SampleMapCtrl.ItemsSource = _SampleMapAdapter;
            _SampleMapCtrl.GetFilePath = this.GetFilePath;
            this._SampleMapCtrl.KeyRegionDrawer = new KeyRegionWithWaveReferenceCountDrawer();
            this._SampleMapCtrl.VelocityRegionDrawer = new VelocityRegionWithWaveReferenceCountDrawer();
            this._SampleMapCtrl.ModifyDrawDescriptor = ModifyDrawDescriptor;

            _SampleMapAdapter.Updated += OnUpdated;

            //_SampleMapCtrl.Resize += OnResizeKeyboard;
            _SampleMapCtrl.Widths = _KeyboardCtrl.GetKeyboardWidthArray();
            _SampleMapCtrl.SelectChanged += OnSelectChangedSampleMap;
            _SampleMapCtrl.ItemsSourceChanged += OnItemsSourceChanged;
            _SampleMapCtrl.Dropped += OnDropped;
            _SampleMapCtrl.ContextMenuHandler = this;

            comboBox_WaveEncoding.Items.Clear();
            comboBox_WaveEncoding.Items.AddRange(this.waveEncodingItems);
            comboBox_WaveEncoding.SelectedIndexChanged += OnWaveEncodingChanged;
            comboBox_WaveEncoding.DropDown += OnDropDownWaveEncoding;
            comboBox_WaveEncoding.DropDownClosed += OnDropDownClosedWaveEncoding;

            velocitySelectorControl.VelocityChanged += OnVelocityChanged;

            originalKeyController.MapGraduationWidth(_KeyboardCtrl.GetKeyboardWidthArray());
            originalKeyController.OriginalKey = -1;
            originalKeyController.KeyChanged += OnOriginalKeyChanged;

            _KeyboardCtrl.KeyOn += OnNoteOn;
            _KeyboardCtrl.KeyOff += OnNoteOff;
            _KeyboardCtrl.MouseDownBefore += OnMouseDownBefore;

            foreach (Control ctrl in this.Controls)
            {
                ctrl.Leave += OnLeave;
                ctrl.Enter += OnEnter;
            }

            this.ProjectProfileService.WaveReferenceCountMeasurer.RequireRecalculated +=
                OnRequireRecalculated;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnRequireRecalculated(object sender, EventArgs e)
        {
            this._SampleMapCtrl.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private void ModifyDrawDescriptor(SampleMapDrawDescriptor desc, KeyRegionInfo keyRegionInfo, VelocityRegionInfo velocityRegionInfo)
        {
            ProjectProfileService.MeasureWaveReferenceCount.ReferenceCountData data =
                ProjectProfileService.WaveReferenceCountMeasurer.GetReferenceCountData
                (this._BankPanel.FilePath, this._Instrument.ProgramNo);

            if (data != null)
            {
                int count = data.GetCount(velocityRegionInfo.KeyRegionInfo.Minimum,
                                           velocityRegionInfo.KeyRegionInfo.Maximum,
                                           velocityRegionInfo.Minimum,
                                           velocityRegionInfo.Maximum);

                desc.Text = String.Format("{0} ({1}{2})",
                                           Path.GetFileName(velocityRegionInfo.FilePath),
                                           count,
                                           data.Incorrect == true ? "*" : String.Empty);

                ExtraDrawDescriptor extraDesc = new ExtraDrawDescriptor();
                extraDesc.FilePath = velocityRegionInfo.FilePath;
                extraDesc.ProgramNo = this._Instrument.ProgramNo;
                extraDesc.KeyMinimum = keyRegionInfo.Minimum;
                extraDesc.KeyMaximum = keyRegionInfo.Maximum;
                extraDesc.VelocityMinimum = velocityRegionInfo.Minimum;
                extraDesc.VelocityMaximum = velocityRegionInfo.Maximum;

                extraDesc.Count = count;
                extraDesc.TotalCount = data.TotalCount;

                desc.UserData = extraDesc;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void OnClosed()
        {
            if (_SampleMapAdapter != null)
            {
                _SampleMapAdapter.Dispose();
            }

            this.ProjectProfileService.WaveReferenceCountMeasurer.RequireRecalculated -=
                OnRequireRecalculated;
        }

        /// <summary>
        ///
        /// </summary>
        public SampleMapCtrl SampleMapCtrl
        {
            get
            {
                return _SampleMapCtrl;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public KeyboardControl KeyboardCtrl
        {
            get
            {
                return _KeyboardCtrl;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Instrument Instrument
        {
            get
            {
                return _Instrument;
            }
            set
            {
                _Instrument = value;

                if (_Instrument != null && _Instrument.Children != null)
                {
                    _SampleMapAdapter.SetupSampleMap(_Instrument.Children);
                }
                else
                {
                    _SampleMapAdapter.SetupSampleMap(null);
                }
                UpdatePanel();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int GetProgramNo()
        {
            int programNo = -1;

            if (_Instrument != null)
            {
                programNo = _Instrument.ProgramNo;
            }

            return programNo;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int GetOriginalKey()
        {
            int originalKey = -1;

            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                originalKey = velRegions[0].OriginalKey;
            }

            return originalKey;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int GetVelocityValue()
        {
            return velocitySelectorControl.SelectedVelocity;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetVelocityValue(int velocity)
        {
            velocitySelectorControl.SelectedVelocity = velocity;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void RedrawControls()
        {
            this.UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.Shown()
        {
            SelectChangedSampleMap();
            UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.UpdateReadOnly(bool readOnly)
        {
            _ReadOnly = readOnly;
            _SampleMapCtrl.ReadOnly = readOnly;
            originalKeyController.Enabled = !readOnly;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.CommandExecuted(Command command)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.UndoExecuted()
        {
            _SampleMapAdapter.Update();
            UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.RedoExecuted()
        {
            _SampleMapAdapter.Update();
            UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ParameterPanel ParameterPanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetValue(Component component, string name, object value)
        {
            if (component.Parameters.ContainsKey(name) == false)
            {
                return;
            }

            SetParameterOperation operation =
                new SetParameterOperation(component.Parameters, name, value);
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnOperationExecuted(Operation operation)
        {
            OnOperationExecuted(new OperationExecutedEventArgs(operation));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnOperationExecuted(OperationExecutedEventArgs e)
        {
            if (OperationHistory != null)
            {
                OperationHistory.AddOperation(e.Operation);
            }

            if (OperationExecuted != null)
            {
                OperationExecuted(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            _SampleMapCtrl.Widths = _KeyboardCtrl.GetKeyboardWidthArray();
            _SampleMapCtrl.UpdateInfos();
            _SampleMapCtrl.Invalidate();

            originalKeyController.MapGraduationWidth(_KeyboardCtrl.GetKeyboardWidthArray());
            originalKeyController.Invalidate();

            _KeyboardCtrl.Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private OperationHistory OperationHistory
        {
            get { return _BankPanel.BankDocument.OperationHistory; }
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsUpdateingShowStatus
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        private ProjectProfileService ProjectProfileService
        {
            get
            {
                return FormsApplication.Instance.ProjectProfileService;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool FilePathSetter(IAdapterManipulator adapter, Component targetComponent, string name, object value)
        {
            if (ProjectFilePathUtility.Inquire(adapter, targetComponent, name, value) == true)
            {
                return true;
            }

            if (targetComponent is VelocityRegion)
            {
                string filePath = (string)value;
                VelocityRegion velocityRegion = targetComponent as VelocityRegion;
                OperationHistory.BeginTransaction();
                SetValue(velocityRegion, ProjectParameterNames.FilePath, value);
                if (OriginalKeyCreator.HasOriginalKeyFromFileName(filePath) == true)
                {
                    SetValue(velocityRegion,
                             ProjectParameterNames.VelocityRegion.OriginalKey,
                             OriginalKeyCreator.FromFile(filePath));
                }
                OperationHistory.EndTransaction();

                return true;
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private string GetFilePath(string filePath)
        {
            string title = SoundFoundation.Resources.MessageResource.DialogTitle_SampleMap_ReferenceWaveFile_Text;
            string result = null;

            if (WaveFileQuester.QueryLoad(this, title, null, filePath, out result) == false)
            {
                result = null;
            }
            return result;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnResizeKeyboard(object sender, EventArgs e)
        {
            _SampleMapCtrl.Widths = _KeyboardCtrl.GetKeyboardWidthArray();
            _SampleMapCtrl.Invalidate();

            originalKeyController.MapGraduationWidth(_KeyboardCtrl.GetKeyboardWidthArray());
            originalKeyController.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnDropped(object sender, SampleMapCtrl.DroppedEventArgs e)
        {
            string filePath = e.FilePath;

            MainWindow.Activate();

            try
            {
                WaveFileReader.CreateInstance(filePath);
                e.Cancel = false;
            }
            catch
            {
                string message = Resources.MessageResource.Message_CanNotUseUnsupportedFileWithAAC;

                TextDisplayMessageBox dialog = new TextDisplayMessageBox
                    (message, filePath,
                      TextDisplayMessageBoxStyle.OKButton);
                dialog.ShowDialog();

                e.Cancel = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnItemsSourceChanged(object sender, EventArgs e)
        {
            UpdatePanel();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectChangedSampleMap(object sender, EventArgs e)
        {
            SelectChangedSampleMap();
        }

        /// <summary>
        ///
        /// </summary>
        private void SelectChangedSampleMap()
        {
            if (this._BankPanel.IsSelectedSampleMapPanel == false)
            {
                return;
            }

            VelocityRegion[] velRegions = SelectedVelocityRegions;
            string filePath = string.Empty;

            ParameterPanel.SetBankPanel(_BankPanel);

            if (velRegions.Length == 1)
            {
                ParameterPanel.Instrument = velRegions[0].Parent.Parent as Instrument;
                ParameterPanel.VelocityRegion = velRegions[0];
                filePath = velRegions[0].FilePath;
            }
            else
            {
                ParameterPanel.VelocityRegion = null;
            }

            MainWindow.UpdateStatusText(filePath);
            UpdatePanel();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnUpdated(object sender, EventArgs e)
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new MethodInvoker(UpdatePanel));
            }
            else
            {
                this.UpdatePanel();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdatePanel()
        {
            MainWindow.BuildCommandUI();
            _SampleMapCtrl.UpdateInfos();
            UpdateShowStatus();
            _BankPanel.UpdateInfos();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateShowStatus()
        {
            if (this.IsUpdateingShowStatus == true)
            {
                return;
            }
            try
            {
                IsUpdateingShowStatus = true;
                bool comboBox_WaveEncoding_HaveFocus = comboBox_WaveEncoding.Focused;
                IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;

                textBox_KeyRegionMin.Text = string.Empty;
                textBox_KeyRegionMax.Text = string.Empty;
                textBox_VelocityRegionMin.Text = string.Empty;
                textBox_VelocityRegionMax.Text = string.Empty;
                textBox_VelocityRegionMin.ReadOnly = true;
                textBox_VelocityRegionMax.ReadOnly = true;
                textBox_OriginalKey.Text = string.Empty;
                textBox_OriginalKey.ReadOnly = true;
                comboBox_WaveEncoding.Items.Clear();
                if (comboBox_WaveEncoding_HaveFocus == false)
                {
                    comboBox_WaveEncoding.Enabled = false;
                }
                originalKeyController.OriginalKey = -1;
                this.ErrorProvider.SetError(this.comboBox_WaveEncoding, null);

                if (velRegions.Length == 0)
                {
                }
                else if (velRegions.Length == 1)
                {
                    comboBox_WaveEncoding.Items.AddRange(this.waveEncodingItems);
                    IKeyRegion key = velRegions[0].Parent;
                    textBox_KeyRegionMin.Text = key.Minimum.ToString();
                    textBox_KeyRegionMax.Text = key.Maximum.ToString();
                    textBox_VelocityRegionMin.Text = velRegions[0].Minimum.ToString();
                    textBox_VelocityRegionMax.Text = velRegions[0].Maximum.ToString();
                    if (_ReadOnly == false)
                    {
                        textBox_VelocityRegionMin.ReadOnly = (velRegions[0].Minimum == 0);
                        textBox_VelocityRegionMax.ReadOnly = (velRegions[0].Maximum == 127);

                        textBox_OriginalKey.ReadOnly = false;

                        comboBox_WaveEncoding.Enabled = true;
                    }
                    textBox_OriginalKey.Text =
                        KeyNoteConverter.ToNote(velRegions[0].OriginalKey);
                    comboBox_WaveEncoding.SelectedItem = WaveEncodingToSelectItem(velRegions[0].WaveEncoding);
                    if (this.IsValidateWaveEncodingValueDelegate != null &&
                        this.IsValidateWaveEncodingValueDelegate(velRegions[0].WaveEncoding) == false)
                    {
                        this.ErrorProvider.SetError(this.comboBox_WaveEncoding, MessageResource.Message_UnsupportedWaveEncoding);
                    }

                    originalKeyController.OriginalKey =
                        ((VelocityRegion)(velRegions[0].Target)).OriginalKey;
                    VelocityRegion target = velRegions[0].Target as VelocityRegion;
                    if (target != null)
                    {
                        if (target.WaveFile == null)
                        {
                            if (string.IsNullOrEmpty(target.FilePath) == false)
                            {
                                WaveFile waveFile = null;
                                try
                                {
                                    using (WaveFileReader reader = WaveFileReader.CreateInstance(target.FilePath))
                                    {
                                        waveFile = reader.Open(target.FilePath);
                                    }
                                }
                                catch
                                {
                                    waveFile = null;
                                }
                                target.WaveFile = waveFile;
                            }
                        }
                    }
                }
                else if (velRegions.Length > 1)
                {
                    bool commonParentFlag = true;
                    IKeyRegion parent = velRegions[0].Parent;
                    foreach (IVelocityRegion vel in velRegions)
                    {
                        if (vel.Parent != parent)
                        {
                            commonParentFlag = false;
                            break;
                        }
                    }

                    if (commonParentFlag == true)
                    {
                        textBox_KeyRegionMin.Text = parent.Minimum.ToString();
                        textBox_KeyRegionMax.Text = parent.Maximum.ToString();
                    }
                }
            }
            finally
            {
                this.IsUpdateingShowStatus = false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public VelocityRegion[] SelectedVelocityRegions
        {
            get
            {
                return _SampleMapCtrl.SelectedVelocityRegions
                    .Cast<ComponentVelocityRegion>()
                    .Select(i => (VelocityRegion)i.Target).ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ExistBlankKeyRegion
        {
            get
            {
                if (_SampleMapAdapter != null)
                {
                    return _SampleMapAdapter.ExistBlankKeyRegion;
                }

                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private WaveEncodingItem WaveEncodingToSelectItem(WaveEncoding encoding)
        {
            foreach (WaveEncodingItem item in this.comboBox_WaveEncoding.Items)
            {
                if (item.WaveEncoding == encoding)
                {
                    return item;
                }
            }

            return this.comboBox_WaveEncoding.SelectedItem as WaveEncodingItem;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int LoopTypeToSelectIndex(LoopType loopType)
        {
            switch (loopType)
            {
                case LoopType.InFile: return 0;
                case LoopType.Manual: return 1;
                case LoopType.None: return 2;
                default: return 0; // LoopType.InFile
            }
        }
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private LoopType SelectIndexToLoopType(int index)
        {
            switch (index)
            {
                case 0: return LoopType.InFile;
                case 1: return LoopType.Manual;
                case 2: return LoopType.None;
                default: return LoopType.InFile;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelocityRegionMin(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                OnVelocityRegionMin(sender, EventArgs.Empty);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelocityRegionMin(object sender, EventArgs e)
        {
            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                int min = 0;
                if (int.TryParse(textBox_VelocityRegionMin.Text, out min) == true)
                {
                    IVelocityRegion velRegion = velRegions[0];
                    if (velRegion.Minimum != min)
                    {
                        IVelocityRegion prev =
                            _SampleMapAdapter.GetPrevVelocityRegion(velRegion);
                        if (prev == null)
                        {
                            if (min < 0)
                            {
                                goto Return;
                            }
                        }
                        else
                        {
                            if (min <= prev.Minimum)
                            {
                                goto Return;
                            }
                            else if (velRegion.Maximum < min)
                            {
                                goto Return;
                            }
                        }
                        velRegion.Minimum = min;
                        UpdatePanel();
                    }
                }
                Return:
                textBox_VelocityRegionMin.Text = velRegions[0].Minimum.ToString();
                textBox_VelocityRegionMin.SelectAll();
                textBox_VelocityRegionMin.Update();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelocityRegionMax(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                OnVelocityRegionMax(sender, EventArgs.Empty);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelocityRegionMax(object sender, EventArgs e)
        {
            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                int max = 0;
                if (int.TryParse(textBox_VelocityRegionMax.Text, out max) == true)
                {
                    IVelocityRegion velRegion = velRegions[0];
                    if (velRegion.Maximum != max)
                    {
                        IVelocityRegion next =
                            _SampleMapAdapter.GetNextVelocityRegion(velRegion);
                        if (next == null)
                        {
                            if (127 < max)
                            {
                                goto Return;
                            }
                        }
                        else
                        {
                            if (next.Maximum <= max)
                            {
                                goto Return;
                            }
                            else if (max < velRegion.Minimum)
                            {
                                goto Return;
                            }
                        }
                        velRegion.Maximum = max;
                        UpdatePanel();
                    }
                }
                Return:
                textBox_VelocityRegionMax.Text = velRegions[0].Maximum.ToString();
                textBox_VelocityRegionMax.SelectAll();
                textBox_VelocityRegionMax.Update();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelocityRegionOriginalKey(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                OnVelocityRegionOriginalKey(sender, EventArgs.Empty);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnVelocityRegionOriginalKey(object sender, EventArgs e)
        {
            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                int key = KeyNoteConverter.ToKey(textBox_OriginalKey.Text);
                if (0 <= key)
                {
                    IVelocityRegion velRegion = velRegions[0];
                    if (velRegion.OriginalKey != key)
                    {
                        velRegion.OriginalKey = key;
                        UpdatePanel();
                    }
                }
            }
            textBox_OriginalKey.SelectAll();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnWaveEncodingChanged(object sender, EventArgs e)
        {
            if (this.IsUpdateingShowStatus == true)
            {
                return;
            }

            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                WaveEncoding waveEncoding = (comboBox_WaveEncoding.SelectedItem as WaveEncodingItem).WaveEncoding;
                IVelocityRegion velRegion = velRegions[0];
                if (velRegion.WaveEncoding != waveEncoding)
                {
                    velRegion.WaveEncoding = waveEncoding;
                    UpdatePanel();
                }
            }
        }

        private void OnDropDownWaveEncoding(object sender, EventArgs e)
        {
            // 編集時には Pcm8 を選択できないようにします。
            // （Siglo では Pcm8 は非サポートです）
            var index = this.comboBox_WaveEncoding.SelectedIndex;
            this.comboBox_WaveEncoding.Items.Clear();
            this.comboBox_WaveEncoding.Items.AddRange(_waveEncodingItemsWithoutPcm8);
            if (index < this.comboBox_WaveEncoding.Items.Count)
            {
                this.comboBox_WaveEncoding.SelectedIndex = index;
            }
            _oldIndex = index;
        }

        private void OnDropDownClosedWaveEncoding(object sender, EventArgs e)
        {
            // 編集終了時には Pcm8 を表示するために元に戻します。
            // （Pcm8 は過去のデータを表示するためだけにあります）
            int index = this.comboBox_WaveEncoding.SelectedIndex;
            this.comboBox_WaveEncoding.Items.Clear();
            this.comboBox_WaveEncoding.Items.AddRange(this.waveEncodingItems);
            if (index < 0)
            {
                index = _oldIndex;
            }
            this.comboBox_WaveEncoding.SelectedIndex = index;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnVelocityChanged(object sender, EventArgs e)
        {
            label_SelectVelocityRegion.Text = velocitySelectorControl.SelectedVelocity.ToString();
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnOriginalKeyChanged(object sender, EventArgs e)
        {
            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                SetValue(velRegions[0].Target,
                          ProjectParameterNames.VelocityRegion.OriginalKey,
                          originalKeyController.OriginalKey);
                UpdatePanel();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public delegate void NoteEventHandler(object s, NoteEventArgs e);
        public class NoteEventArgs : EventArgs
        {
            public int Key { get; set; }
            public NoteEventArgs(int key)
            {
                Key = key;
            }
        }

        private void OnNoteOn(object sender, KeyboardControl.KeyboardEventArgs e)
        {
            if (NoteOn != null)
            {
                NoteOn(this, new NoteEventArgs(e.Key));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnNoteOff(object sender, KeyboardControl.KeyboardEventArgs e)
        {
            if (NoteOff != null)
            {
                NoteOff(this, new NoteEventArgs(e.Key));
            }
        }

        /// <summary>
        /// MouseDown の前に実行されます。
        /// </summary>
        private void OnMouseDownBefore(object sender, KeyboardControl.MouseDownBeforeEventArgs e)
        {
            if ((e.Button & MouseButtons.Left) != 0 &&
                FormsApplication.Instance.IsErrorAudioDevice == true)
            {
                FormsApplication.Instance.ShowWarningMessageCanNotInitializeAudioDevice();

                e.Cancel = true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnLeave(object sender, EventArgs e)
        {
            _SampleMapCtrl.GrayOut = true;
            _SampleMapCtrl.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEnter(object sender, EventArgs e)
        {
            _SampleMapCtrl.GrayOut = false;
            _SampleMapCtrl.Invalidate();
        }

#if false
        /// <summary>
        ///
        /// </summary>
        private void OnLoopTypeComboBoxSelectedIndexChanged(object sender, EventArgs e)
        {
            if (this.IsUpdateingShowStatus == true)
            {
                return;
            }

            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                VelocityRegion target = velRegions[0].Target as VelocityRegion;
                if (target != null)
                {
                    LoopType loopType = SelectIndexToLoopType(LoopTypeComboBox.SelectedIndex);
                    if (target.LoopType != loopType)
                    {
                        SetValue(target, ProjectParameterNames.LoopType, loopType);
                        UpdatePanel();
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnLoopStartTextBoxValidated(object sender, EventArgs e)
        {
            UpdatePanel();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnLoopEndTextBoxValidated(object sender, EventArgs e)
        {
            UpdatePanel();
        }
#endif
        /// <summary>
        ///
        /// </summary>
        private void OnLoopFrameTextBoxKeyDown(object sender, KeyEventArgs e)
        {
            if (this.IsUpdateingShowStatus == true)
            {
                return;
            }

            if (e.KeyCode == Keys.Enter)
            {
                CheckLoopFrameRange(sender as TextBox);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool CheckLoopFrameRange(TextBox textBox)
        {
            IVelocityRegion[] velRegions = _SampleMapCtrl.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                VelocityRegion target = velRegions[0].Target as VelocityRegion;
                if (target != null && target.WaveFile != null && textBox != null)
                {
                    int loopFrame = int.Parse(textBox.Text);
                    if (loopFrame < 0 ||
                        (int)(target.WaveFile.FrameCount) < loopFrame)
                    {
                        OutOfRangeValidationResult result =
                            new OutOfRangeValidationResult(0.ToString(), target.WaveFile.FrameCount.ToString(), string.Empty);
                        MessageBox.Show(result.ToString());
                        return false;
                    }
                }
            }

            return true;
        }

        void SampleMapCtrl.IContextMenuHandler.ExecuteItemFolder(SampleMapCtrl control)
        {
            string filePath = control.SelectedVelocityRegions
                .Select(it => it.FilePath)
                .SingleOrDefault();

            if (!string.IsNullOrEmpty(filePath))
            {
                CommandHandlers.ExecuteItemFolderHandler.Execute(filePath);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    static class RegionDrawer
    {
        public static void Draw(SampleMapDrawDescriptor drawDesc, Color baseColor, bool isVelocityRegion)
        {
            Graphics gc = drawDesc.Gc;
            Rectangle bounds = drawDesc.Bounds;

            if (drawDesc.Blank == false)
            {
                gc.FillRectangle(new SolidBrush(baseColor), bounds);

                //
                if (isVelocityRegion == true)
                {
                    ExtraDrawDescriptor extraDesc = drawDesc.UserData as ExtraDrawDescriptor;
                    if (extraDesc != null &&
                        extraDesc.Count > 0 &&
                        extraDesc.TotalCount > 0)
                    {
                        float ratio = (float)extraDesc.Count / (float)extraDesc.TotalCount;
                        int stage = 5;
                        int range = 255 / stage;

                        int alpha = ((int)(255.0F * ratio) / range + 1) * range;
                        if (alpha > 255)
                        {
                            alpha = 255;
                        }

                        Brush brush = new SolidBrush(Color.FromArgb(alpha, Color.Red));
                        gc.FillRectangle(brush, bounds);
                    }
                }
            }

            //
            if (drawDesc.Selected == true)
            {
                Color color;

                if (drawDesc.Focused == true)
                {
                    color = SystemColors.Highlight;
                }
                else
                {
                    color = Color.FromArgb(192,
                                            (SystemColors.ControlDark.R * 160 / 255) +
                                            (Color.White.R * (255 - 160) / 255),
                                            (SystemColors.ControlDark.G * 160 / 255) +
                                            (Color.White.G * (255 - 160) / 255),
                                            (SystemColors.ControlDark.B * 160 / 255) +
                                            (Color.White.B * (255 - 160) / 255));
                }

                Brush brush = new SolidBrush(Color.FromArgb(127, color));
                gc.FillRectangle(brush, bounds);
            }

            //
            if (drawDesc.Modify == true || drawDesc.Drop == true)
            {
                Brush brush = new SolidBrush
                    (Color.FromArgb(80,
                                      (SystemColors.WindowText.R +
                                        SystemColors.HighlightText.R) / 2,
                                      (SystemColors.WindowText.G +
                                        SystemColors.HighlightText.G) / 2,
                                      (SystemColors.WindowText.B +
                                        SystemColors.HighlightText.B) / 2));

                gc.FillRectangle(brush, bounds);
            }

            //
            if (drawDesc.Blank == false)
            {
                if (isVelocityRegion == true)
                {
                    Brush brush = (drawDesc.Selected == true ?
                                    SystemBrushes.HighlightText :
                                    SystemBrushes.WindowText);
                    gc.DrawString(drawDesc.Text, drawDesc.Font, brush, bounds);
                }

                drawDesc.Gc.DrawRectangle
                    (Pens.Black, bounds.X, bounds.Y, bounds.Width - 1, bounds.Height - 1);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class KeyRegionWithWaveReferenceCountDrawer : IKeyRegionDrawer
    {
        public void Draw(SampleMapDrawDescriptor drawDesc)
        {
            RegionDrawer.Draw(drawDesc, Color.FromArgb(255, 0, 255, 255), false);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class VelocityRegionWithWaveReferenceCountDrawer : IVelocityRegionDrawer
    {
        public void Draw(SampleMapDrawDescriptor drawDesc)
        {
            RegionDrawer.Draw(drawDesc, SystemColors.Window, true);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ExtraDrawDescriptor
    {
        public string FilePath { get; set; }
        public int ProgramNo { get; set; }
        public int KeyMinimum { get; set; }
        public int KeyMaximum { get; set; }
        public int VelocityMinimum { get; set; }
        public int VelocityMaximum { get; set; }

        public int Count { get; set; }
        public int TotalCount { get; set; }
    }

    /// <summary>
    /// 圧縮形式の選択アイテムクラス
    /// </summary>
    public class WaveEncodingItem
    {
        public WaveEncodingItem(WaveEncoding waveEncoding)
        {
            this.WaveEncoding = waveEncoding;
        }

        public override string ToString()
        {
            return this.WaveEncoding.ToText();
        }

        public WaveEncoding WaveEncoding
        {
            get; private set;
        }
    }
}
