﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundFoundation.Operations;

    public partial class ProjectPanel : CommonPanel, IQueryCommandParameter
    {
        // コマンドバー管理
        private ToolStripAdapter _projectTreeMenuAdapter;

        private ProjectTreeAdapter _ProjectAdapter = new ProjectTreeAdapter();

        private SoundProjectDocument _ProjectDocument = null;

        public event EventHandler SelectChanged;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ProjectPanel()
        {
            InitializeComponent();
            InitializeCommandBindings();
            InitializeMenu();

            var bitmap = ImageResource.BitmapIconLoupe;
            bitmap.MakeTransparent();
            pictureBox_Search.Image = bitmap;

            bitmap = ImageResource.BitmapIconDeleteFilterText;
            bitmap.MakeTransparent();
            button_DeleteFilterText.Image = bitmap;

            _ProjectCtrl.Hide();
            _ProjectCtrl.Images["Expand"] = ImageResource.BitmapIconSoundSet;
            _ProjectCtrl.Images["Collapse"] = ImageResource.BitmapIconStreamSoundFolder;
            _ProjectCtrl.SelectChanged += OnSelectChanged;
            _ProjectCtrl.EditBegan += OnEditBegan;
            _ProjectCtrl.EditEnded += OnEditEnded;
            _ProjectCtrl.QueryFileDropped += OnQueryFileDropped;
            _ProjectCtrl.FileDropped += OnFileDropped;

            _projectTreeMenuAdapter.CommandExecuted += (s, e) =>
            {
                if (e.Command.ID == ProjectTreeCommands.OutputType.ID)
                {
                    ProjectCtrl.Invalidate();
                }
            };

            AllowDrop = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void AttachProject(SoundProjectDocument projectDocument)
        {
            _ProjectDocument = projectDocument;

            _ProjectAdapter.AttachProject(Project);

            _ProjectCtrl.ItemsSource = _ProjectAdapter;
            _ProjectCtrl.ReadOnly = this.ReadOnly;

            ApplyProjectConfiguration();
            _ProjectCtrl.Show();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DetachProject()
        {
            ExtractProjectConfiguration();

            _ProjectDocument = null;

            _ProjectAdapter.AttachProject(null);
            _ProjectAdapter.OperationHistory = null;

            _ProjectCtrl.ItemsSource = null;
            _ProjectCtrl.ReadOnly = false;
            _ProjectCtrl.Hide();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void Activated()
        {
            ParameterPanel.Instrument = null;
            ParameterPanel.VelocityRegion = null;
            ParameterPanel.Item = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ProjectCtrl ProjectCtrl
        {
            get { return _ProjectCtrl; }
        }

        ///--------------------------------
        /// <summary>
        /// プロジェクトが読み取り専用かどうか取得します。
        /// </summary>
        public bool ReadOnly
        {
            get
            {
                if (this.ProjectDocument == null)
                {
                    return false;
                }
                else
                {
                    return this.ProjectDocument.IsReadOnly;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SoundProject Project
        {
            get { return ProjectDocument.Project; }
        }

        private SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SoundProjectDocument ProjectDocument
        {
            get
            {
                return _ProjectDocument as SoundProjectDocument;
            }
        }

        ///--------------------------------
        /// <summary>
        /// アクティブなツリーコントロールを取得します。
        /// </summary>
        protected ProjectCtrl ActiveTreeCtrl
        {
            get
            {
                if (_ProjectCtrl.ContainsFocus) { return _ProjectCtrl; }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override SoundDocument TargetDocument
        {
            get { return ProjectDocument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component[] TargetComponents
        {
            get
            {
                Component[] components = null;

                if (null != ActiveTreeCtrl)
                {
                    components = ActiveTreeCtrl.GetComponentsBySelectedItem();
                }

                return components != null ? components : new Component[0];
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateProjectCtrl()
        {
            _ProjectCtrl.Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void CommandExecuted(Command command)
        {
            UpdateProjectCtrl();
            BuildCommandUI();
            MainWindow.RedrawPanels();

            if (command == FileCommands.NewProject)
            {
                this._ProjectAdapter.UpdateEmptyNodesExpander();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void OnClosed()
        {
            ApplicationBase.Instance.ProjectConfiguration.Saving -= OnProjectConfigurationSaving;

            base.OnClosed();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ApplySoundSetConfigurations()
        {
            _ProjectAdapter.UpdateNodeVisibility();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ParameterPanel ParameterPanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundProjectConfiguration ProjectConfiguration
        {
            get { return ApplicationBase.Instance.ProjectConfiguration; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentService TargetComponentService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                case CommandParameterNames.TargetDocuments:
                case CommandParameterNames.TargetComponents:
                case CommandParameterNames.InsertParentComponent:
                    return true;
            }

            return false;
        }

        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                    return TargetComponentService;

                case CommandParameterNames.TargetDocuments:
                    if (null == TargetDocument) { return new SoundDocument[0]; }
                    return new SoundDocument[] { TargetDocument };

                case CommandParameterNames.TargetComponents:
                    return TargetComponents;

                case CommandParameterNames.InsertParentComponent:
                    if (ActiveTreeCtrl != null)
                    {
                        var selectedItems = ActiveTreeCtrl.SelectedItems;
                        if (selectedItems.Length > 0)
                        {
                            return selectedItems[0].Target;
                        }
                    }
                    return null;
            }

            throw new KeyNotFoundException();
        }

        protected override void OnLoad(EventArgs e)
        {
            if (ApplicationBase.Instance != null)
            {
                ApplicationBase.Instance.ProjectConfiguration.Saving += OnProjectConfigurationSaving;
                base.OnLoad(e);
            }
        }

        ///--------------------------------
        /// <summary>
        /// コントロールが入力されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragEnter(DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
            {
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragDrop(DragEventArgs e)
        {
            string[] filePaths = null;

            filePaths = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            MainWindow.DropFiles(filePaths);
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップできるかどうか判断する時に呼ばれる
        /// </summary>
        private void OnQueryFileDropped(object sender, FileDroppedEventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDropped(object sender, FileDroppedEventArgs e)
        {
            MainWindow.DropFiles(e.FilePaths);
        }

        ///--------------------------------
        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (null != _projectTreeMenuAdapter) { return; }
            if (null == ApplicationBase.Instance) { return; }

            _projectTreeMenuAdapter = ToolStripAdapter.FromToolStrip
                (ApplicationBase.Instance.CommandService, _projectTreeMenu);
            _projectTreeMenuAdapter.CommandTarget = ApplicationBase.Instance;
            _projectTreeMenuAdapter.CommandExecuted += OnProjectTreeCommandExecuted;

            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        /// コマンドの関連付けを行います。
        /// </summary>
        private void InitializeCommandBindings()
        {
            // ファイル関連コマンドハンドラ
            BindCommand(new CommandBinding(this, new OpenItemFolderHandler(this)));

            //
            BindCommand(new CommandBinding(this, new UndoHandler(this)));
            BindCommand(new CommandBinding(this, new RedoHandler(this)));
            BindCommand(new CommandBinding(this, new ProjectPanelDeleteHandler(this)));

            BindCommand(new CommandBinding(this, ProjectTreeCommands.Rename.ID,
                                             QueryStatusRename, ExecuteRename));
            BindCommand(new CommandBinding(this, ProjectTreeCommands.HideEmptyNode.ID,
                                             QueryStatusHideEmptyNode, ExecuteHideEmptyNode));
            BindCommand(new CommandBinding(this, ProjectTreeCommands.AllHideEmptyNode.ID,
                                             QueryStatusAllHideEmptyNode, ExecuteAllHideEmptyNode));
            BindCommand(new CommandBinding(this, ProjectTreeCommands.CollapseAll.ID,
                                           QueryStatusCollapseAll, ExecuteCollapseAll));

            BindCommand(new CommandBinding(this, ProjectTreeCommands.OutputType.ID,
                                           QueryStatusOutputType, ExecuteOutputType));

            BindCommand(new CommandBinding(this, new AddBookmarkHandler(this)));

            BindCommand(new CommandBinding(this, ProjectTreeCommands.AddNewFolder.ID,
                QueryStatusAddNewFolder, ExecuteAddNewFolder));

            // プロジェクト関連コマンドハンドラ
            BindCommand(new CommandBinding
                         (this, new QueryAndAddNewSoundSetHandler(this)));
            BindCommand(new CommandBinding
                         (this, new QueryAndAddExistingSoundSetHandler(this)));
        }

        ///--------------------------------
        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUI()
        {
            if (null != _projectTreeMenuAdapter)
            {
                _projectTreeMenuAdapter.BuildUI();
            }
        }

        /// <summary>
        /// プロジェクト設定を適用します。
        /// </summary>
        private void ApplyProjectConfiguration()
        {
            if (!ProjectConfiguration.DocumentViews.ContainsKey(_ProjectDocument.Resource.Key))
            {
                return;
            }

            XmlSoundProjectDocumentView xmlView =
                    ProjectConfiguration.DocumentViews[_ProjectDocument.Resource.Key]
                    as XmlSoundProjectDocumentView;

            ApplyTreeNodeConfiguration(xmlView.ProjectTreeNodes, _ProjectCtrl.ItemInfos[0].Item as ProjectTreeItem);

            _ProjectCtrl.UpdateInfos();
        }

        /// <summary>
        /// ノード設定を適用します。
        /// </summary>
        /// <param name="xmlTreeNodes">ノード設定。</param>
        /// <param name="item">対象ツリーアイテム。</param>
        private void ApplyTreeNodeConfiguration(XmlTreeNodes xmlTreeNodes, ProjectTreeItem item)
        {
            if (null == xmlTreeNodes) { throw new ArgumentNullException("xmlTreeNodes"); }
            if (null == item) { throw new ArgumentNullException("item"); }

            IDictionary<string, XmlTreeNode> treeNodes = xmlTreeNodes.CreateDictionary();

            foreach (ProjectTreeItem childItem in item.Children)
            {

                if (!treeNodes.ContainsKey(childItem.Target.Name)) { continue; }

                XmlTreeNode xmlTreeNode = treeNodes[childItem.Target.Name];
                childItem.Expanded = xmlTreeNode.Expanded;

                if (null != xmlTreeNode.TreeNodes)
                {
                    ApplyTreeNodeConfiguration(xmlTreeNode.TreeNodes, childItem);
                }

            }
        }

        /// <summary>
        /// プロジェクト設定を抽出します。
        /// </summary>
        private void ExtractProjectConfiguration()
        {
            if (null == ProjectConfiguration) { return; }
            if (null == _ProjectDocument) { return; }
            if (null == _ProjectDocument.Resource) { return; }
            if (!ProjectConfiguration.DocumentViews.ContainsKey(_ProjectDocument.Resource.Key))
            {
                return;
            }

            XmlSoundProjectDocumentView xmlView =
                ApplicationBase.Instance.ProjectConfiguration.DocumentViews[_ProjectDocument.Resource.Key]
                as XmlSoundProjectDocumentView;

            ExtractTreeNodeConfiguration(_ProjectCtrl.ItemInfos[0].Item as ProjectTreeItem,
                                          xmlView.ProjectTreeNodes);
        }

        /// <summary>
        /// ノード設定を抽出します。
        /// </summary>
        /// <param name="item">対象ツリーアイテム。</param>
        /// <param name="xmlTreeNodes">出力先のノード設定。</param>
        private void ExtractTreeNodeConfiguration(ProjectTreeItem item, XmlTreeNodes xmlTreeNodes)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            if (null == xmlTreeNodes) { throw new ArgumentNullException("xmlTreeNodes"); }

            List<XmlTreeNode> xmlTreeNodesWork = new List<XmlTreeNode>();

            foreach (ProjectTreeItem childItem in item.Children)
            {

                XmlTreeNode xmlChildNode = new XmlTreeNode()
                {
                    Name = childItem.Target.Name,
                    Expanded = childItem.Expanded,
                };

                if (0 < childItem.Children.Count)
                {
                    xmlChildNode.TreeNodes = new XmlTreeNodes();
                    ExtractTreeNodeConfiguration(childItem, xmlChildNode.TreeNodes);
                }

                xmlTreeNodesWork.Add(xmlChildNode);

            }

            if (0 < xmlTreeNodesWork.Count)
            {
                xmlTreeNodes.TreeNode = xmlTreeNodesWork.ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        /// "サウンドセット名の変更"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusRename(Command command)
        {
            if (null == TargetDocument ||
                 null == TargetComponents) { return CommandStatus.SupportedAndVisible; }

            if (this.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            Component[] components = TargetComponents;

            if (components.Length != 1)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return components[0] is SoundSet ?
                    CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "サウンドセット名の変更"の実行
        /// </summary>
        private bool ExecuteRename(Command command)
        {
            Component[] components = TargetComponents;

            if (components.Length != 1)
            {
                return false;
            }

            return ActiveTreeCtrl.BeginEditBySelectedItem();
        }

        ///--------------------------------
        /// <summary>
        /// "空のノードを隠す"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusHideEmptyNode(Command command)
        {
            SoundSet soundSet = null;
            CommandStatus status;

            if ((soundSet = SingleTargetSoundSet) == null)
            {
                return CommandStatus.Supported;
            }

            status = CommandStatus.SupportedAndEnabledAndVisible;
            if (soundSet.HideEmptyNode != false)
            {
                status |= CommandStatus.Checked;
            }

            return status;
        }

        ///--------------------------------
        /// <summary>
        /// "空のノードを隠す"の実行
        /// </summary>
        private bool ExecuteHideEmptyNode(Command command)
        {
            SoundSet soundSet = null;

            if ((soundSet = SingleTargetSoundSet) == null)
            {
                return false;
            }

            _ProjectAdapter.HideEmptyNode(soundSet, !soundSet.HideEmptyNode);
            return true;
        }

        /// <summary>
        /// "すべて空のノードを隠す"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusAllHideEmptyNode(Command command)
        {
            if (TargetComponents == null ||
                TargetComponents.Length <= 0 ||
                TargetComponents[0] is SoundProject == false)
            {
                return CommandStatus.Supported;
            }

            if (IsEnabledHideEmptyNode == false)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible | CommandStatus.Checked;
        }

        /// <summary>
        /// "すべて空のノードを隠す"の実行
        /// </summary>
        private bool ExecuteAllHideEmptyNode(Command command)
        {
            var hide = IsEnabledHideEmptyNode == true ? false : true;
            ProjectService.SoundSets
                .ForEach(s => s.HideEmptyNode = hide);

            _ProjectAdapter.UpdateNodeVisibility();
            return true;
        }

        private bool IsEnabledHideEmptyNode
        {
            get
            {
                return ProjectService.SoundSets
                    .All(s => s.HideEmptyNode == true);
            }
        }

        ///--------------------------------
        /// <summary>
        /// "すべて折りたたむ"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusCollapseAll(Command command)
        {
            if (null == ActiveTreeCtrl) { return CommandStatus.SupportedAndVisible; }

            Component[] components = ActiveTreeCtrl.GetComponentsBySelectedItem();
            return (null != components && 0 < components.Length) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "すべて折りたたむ"の実行
        /// </summary>
        private bool ExecuteCollapseAll(Command command)
        {
            CollapseAll();
            return true;
        }

        /// <summary>
        /// すべてのノードを折りたたみます。。
        /// </summary>
        private void CollapseAll()
        {
            foreach (ITreeItemInfo itemInfo in _ProjectCtrl.ItemInfos)
            {
                if (null == itemInfo.Item.Parent) { continue; }
                itemInfo.Item.Expanded = false;
            }

            _ProjectCtrl.UpdateInfos();
        }

        /// <summary>
        /// フォルダの追加が実行可能か調べます。
        /// </summary>
        private CommandStatus QueryStatusAddNewFolder(Command command)
        {
            if (this.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            if (TargetComponents.Length > 0)
            {
                if (TargetComponents.Length > 1 ||
                    IsContainerComponent(TargetComponents[0]) == false)
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// フォルダの追加を実行します。
        /// </summary>
        private bool ExecuteAddNewFolder(Command command)
        {
            var parentComponent = TargetComponents.Length > 0 ? TargetComponents[0] : Project;
            var folder = new FolderComponent() { Name = CreateUniqueNameInComponentChildren(parentComponent) };
            parentComponent.Children.Add(folder);

            ProjectCtrl.BeginEdit(_ProjectAdapter.FindTreeItem(folder));

            ProjectDocument.SetDirty();
            return true;
        }

        private bool IsContainerComponent(Component component)
        {
            return component is SoundProject || component is FolderComponent ? true : false;
        }

        private string CreateUniqueNameInComponentChildren(Component parentComponent)
        {
            var names = parentComponent.Children
                .Cast<Component>()
                .Select(c => c.Name);
            var hashSet = new HashSet<string>(names);
            var baseName = "NewFolder";

            if (hashSet.Contains(baseName) == false)
            {
                return baseName;
            }

            var index = 0;
            while (true)
            {
                var name = string.Format($"{baseName}_{index}");
                if (hashSet.Contains(name) == false)
                {
                    return name;
                }
                index++;
            }
        }

        /// <summary>
        /// サウンドアーカイブの出力設定が実行可能か調べます。
        /// </summary>
        private CommandStatus QueryStatusOutputType(Command command)
        {
            if (this.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            var components = TargetComponents;
            if (components.All(c => c is SoundSet) == false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            if (components.Length == 1)
            {
                var projectService = FormsApplication.Instance.ProjectService;
                var outputType = projectService.GetSoundArchiveOutputType(components.First() as SoundSet);
                var index = SoundArchiveOutputTypeToIndex(outputType);
                if (index == GetCommandIndexParameter(command))
                {
                    return CommandStatus.SupportedAndEnabledAndVisible | CommandStatus.Checked;
                }
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecuteOutputType(Command command)
        {
            var projectService = FormsApplication.Instance.ProjectService;
            var outputType = IndexToSoundArchiveOutputType(GetCommandIndexParameter(command));

            TargetComponents
                .OfType<SoundSet>()
                .ForEach(c => projectService.SetSoundArchiveOutputType(c, outputType));

            ProjectDocument.SetDirty();
            return true;
        }

        private int GetCommandIndexParameter(Command command)
        {
            Debug.Assert(command.Parameters.ContainsParameter("Index") == true);
            return int.Parse((string)command.Parameters.GetParameter("Index"));
        }

        private SoundArchiveOutputTypes IndexToSoundArchiveOutputType(int index)
        {
            switch (index)
            {
                case 0: return SoundArchiveOutputTypes.None;
                case 1: return SoundArchiveOutputTypes.SoundArchive;
                case 2: return SoundArchiveOutputTypes.AddonSoundArchive;
            }
            Debug.Assert(false);
            return SoundArchiveOutputTypes.SoundArchive;
        }

        private int SoundArchiveOutputTypeToIndex(SoundArchiveOutputTypes outputType)
        {
            switch (outputType)
            {
                case SoundArchiveOutputTypes.None: return 0;
                case SoundArchiveOutputTypes.SoundArchive: return 1;
                case SoundArchiveOutputTypes.AddonSoundArchive: return 2;
            }
            Debug.Assert(false);
            return -1;
        }

        private void UpdateProjectNodesVisivility()
        {
            _ProjectAdapter.FilterText = textBox_FilterText.Text;
            _ProjectAdapter.UpdateNodeVisibility();
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSet SingleTargetSoundSet
        {
            get
            {
                Component[] components = TargetComponents;

                if (components.Length != 1 ||
                    !(components[0] is SoundSet))
                {
                    return null;
                }

                return components[0] as SoundSet;
            }
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            ExtractProjectConfiguration();
        }

        ///--------------------------------
        /// <summary>
        /// プロジェクトツリーコマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnProjectTreeCommandExecuted(object sender, CommandEventArgs e)
        {
            if (e.IsExecuted != false)
            {
                if (e.Command == ProjectCommands.AddNewSoundSet ||
                    e.Command == ProjectCommands.AddExistingSoundSet)
                {
                    _ProjectCtrl.UpdateInfos();
                }

                if (e.Command == ProjectCommands.ShowProjectSettings)
                {
                    MainWindow.UpdateCommandTexts();
                }
            }

            MainWindow.BuildCommandUI();
            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectChanged(object sender, EventArgs e)
        {
            MainWindow.BuildCommandUI();
            MainWindow.ClearPresetListColumnsOnSplitButton();
            BuildCommandUI();

            if (SelectChanged != null)
            {
                SelectChanged(sender, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditBegan(object sender, EventArgs e)
        {
            ApplicationBase.Instance.DisableCommandKeyProcess();
            MainWindow.EnableAllCommands(false);
            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditEnded(object sender, EventArgs e)
        {
            ApplicationBase.Instance.EnableCommandKeyProcess();
            MainWindow.EnableAllCommands(true);
            BuildCommandUI();
        }

        private void OnFilterTextChanged(object sender, EventArgs e)
        {
            UpdateProjectNodesVisivility();
        }

        private void OnClickDeleteFilterText(object sender, EventArgs e)
        {
            textBox_FilterText.Text = string.Empty;
        }
    }
}
