﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Utilities;
    using NintendoWare.SoundFoundation.Windows.CommandBars;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.ToolDevelopmentKit.Collections;

    using CoreResources = SoundFoundation.Resources;
    using SFresrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    ///
    /// </summary>
    public partial class FindResultPanel2 : CommonListPanel, IQueryCommandParameter, IParameterPanelOperationExecutedListener, IEditableControlContainer
    {
        private const string ParameterName_Name = "Name";
        private const string ParameterName_FilePath = "FilePath";
        private const string ParameterName_Comment = "Comment";
        private const string ParameterName_ItemReference = "ItemReference";
        private const string ParameterName_StartPosition = "StartPosition";
        private const string ParameterName_CaseSensitive = "CaseSensitive";
        private const string ParameterName_FindExactMatch = "FindExactMatch";

        private const string FilterName_EnabledItem = "EnabledItem";
        private const string FilterName_StreamSound = "StreamSound";
        private const string FilterName_StreamSoundTrack = "StreamSoundTrack";
        private const string FilterName_WaveSoundSet = "WaveSoundSet";
        private const string FilterName_WaveSound = "WaveSound";
        private const string FilterName_SequenceSoundSet = "SequenceSoundSet";
        private const string FilterName_SequenceSound = "SequenceSound";
        private const string FilterName_Bank = "Bank";
        private const string FilterName_WaveArchive = "WaveArchive";
        private const string FilterName_Group = "Group";
        private const string FilterName_Player = "Player";
        private const string FilterName_ColorNone = "ColorNone";
        private const string FilterName_ColorRed = "ColorRed";
        private const string FilterName_ColorOrange = "ColorOrange";
        private const string FilterName_ColorYellow = "ColorYellow";
        private const string FilterName_ColorGreen = "ColorGreen";
        private const string FilterName_ColorSkyBlue = "ColorSkyBlue";
        private const string FilterName_ColorBlue = "ColorBlue";
        private const string FilterName_ColorPurple = "ColorPurple";
        private const string FilterName_ColorPink = "ColorPink";
        private const string FilterName_ShowAll = "ShowAll";
        private const string FilterName_HideAll = "HideAll";

        private ListPartTextDrawer textDrawer = null;
        private ListPartCommonRowHeaderDrawer rowHeaderDrawer = null;

        private bool unsearched = true;

        /// <summary>
        ///
        /// </summary>
        private class ComboBoxRangeItem
        {
            private string text = null;
            private FindArgs.Ranges range = FindArgs.Ranges.None;

            public ComboBoxRangeItem(string text, FindArgs.Ranges range)
            {
                this.text = text;
                this.range = range;
            }

            public override string ToString()
            {
                return this.text;
            }

            public FindArgs.Ranges Range
            {
                get { return this.range; }
            }
        }

        ///
        private bool filterUpdateLocked = false;
        private FindArgs.TargetDocumentKinds targetDocumentKind = FindArgs.TargetDocumentKinds.SoundSet;
        private string[] targetSoundSetNames = null;

        ///
        private CommonListCtrl listCtrl = null;
        private CommonListAdapter listAdapter = null;
        private Component[] resultComponents = null;

        private ToolStripAdapter listMenuAdapter = null;

        private HashSet<string> excludeColumnNames = new HashSet<string>();
        private int titleNumber = -1;
        private string titleName = MessageResource.TabText_FindResult;

        /// カラムヘッダを作成するか？のフラグです。
        private bool createHeaderColumnData = true;

        ///
        private Dictionary<string, List<SoundSetBankBase>> soundSetBankDictionary = null;

        ///
        private ButtonPair<int>[] ButtonColorIndexPairs = null;

        ///
        private TransactionPackStack transactionPackStack = new TransactionPackStack();

        // TODO: 未実装
        private Dictionary<Component, ListCtrlState> _ListCtrlState = new Dictionary<Component, ListCtrlState>();
        //private bool enabledSaveListState = true;

        private ListHeaderAdapterDictionary _headerAdapters = new ListHeaderAdapterDictionary();
        //private IDictionary<string, XmlList> _xmlLists;

        private ToolStripMenuItem _HeaderContextMenuItem_AddPresetListColumns = null;
        private ToolStripMenuItem _HeaderContextMenuItem_ApplyPresetListColumns = null;

        public event EventHandler PreviewPlay;
        public event EventHandler PreviewMute;
        public event EventHandler PreviewSoloPlay;

        public event EventHandler SelectChangedListItem;

        /// <summary>
        ///
        /// </summary>
        public delegate CommonListCtrl ListControlCreateHandler();
        public static ListControlCreateHandler ListControlCreator
        {
            get;
            set;
        }

        ///
        private delegate string GetStringMethodInvoker();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FindResultPanel2()
        {
            InitializeComponent();

#if DEBUG
            // WindowsForm デザイナ対応
            // this.DesignMode はコンストラクタ内で設定されないようなので ProjectService で判断します。
            if (ProjectService == null)
            {
                return;
            }
#endif

            // ツールバーのカラーボタンの設定です。
            this.tool_ColorRed.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorRed);
            this.tool_ColorOrange.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorOrange);
            this.tool_ColorYellow.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorYellow);
            this.tool_ColorGreen.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorGreen);
            this.tool_ColorSkyBlue.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorLightBlue);
            this.tool_ColorBlue.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorBlue);
            this.tool_ColorPurple.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorPurple);
            this.tool_ColorPink.Image = CreateToolBarBitmap
                (ImageResource.BitmapIconItemColorPink);

            this.ButtonColorIndexPairs = new ButtonPair<int>[] {
                new ButtonPair<int>( tool_ColorNone,    0),
                new ButtonPair<int>( tool_ColorRed,     1),
                new ButtonPair<int>( tool_ColorOrange,  2),
                new ButtonPair<int>( tool_ColorYellow,  3),
                new ButtonPair<int>( tool_ColorGreen,   4),
                new ButtonPair<int>( tool_ColorSkyBlue, 5),
                new ButtonPair<int>( tool_ColorBlue,    6),
                new ButtonPair<int>( tool_ColorPurple,  7),
                new ButtonPair<int>( tool_ColorPink,    8),
            };

            // ツールバーのアイテムボタンの設定です。
            this.tool_Stream.Image = ImageResource.BitmapIconStreamSound;
            this.tool_StreamSoundTrack.Image = ImageResource.BitmapIconStreamSoundTrack;
            this.tool_WaveSet.Image = ImageResource.BitmapIconWaveSoundSet;
            this.tool_Wave.Image = ImageResource.BitmapIconWaveSound;
            this.tool_SequenceSet.Image = ImageResource.BitmapIconSequenceSoundSet;
            this.tool_Sequence.Image = ImageResource.BitmapIconSequenceSound;
            this.tool_Bank.Image = ImageResource.BitmapIconBank;
            this.tool_WaveArchive.Image = ImageResource.BitmapIconWaveArchive;
            this.tool_Group.Image = ImageResource.BitmapIconGroup;
            this.tool_Player.Image = ImageResource.BitmapIconPlayer;
            //this.tool_Instrument.Image = ImageResource.BitmapIconInstrument;

            UpdateToolBarColorTexts();

            //
            this.chk_FindExactMatch.Image =
                ImageResource.BitmapIconFindExactMatch.MakeNewTransparent
                (UIServiceBase.TransparentColor);

            this.chk_CaseSensitive.Image =
                ImageResource.BitmapIconCaseSensitive.MakeNewTransparent
                (UIServiceBase.TransparentColor);

            //
            this.toolLabel_ListFilter.Image =
                ImageResource.BitmapIconListFilter.MakeNewTransparent
                (UIServiceBase.TransparentColor);

            this.toolLabel_ItemFilter.Image =
                ImageResource.BitmapIconItemFilter.MakeNewTransparent
                (UIServiceBase.TransparentColor);

            // ステータスバーの設定です。
            this.toolStripStatusLabel.Image =
                ImageResource.BitmapIconWarningFindResult.MakeNewTransparent
                (UIServiceBase.TransparentColor);

            ShowTransactionStackWarning(false);

            // リストの設定です。
            this.listAdapter = new CommonListAdapter(this.ListDecorationEvaluator);
            this.listAdapter.OperationExecuted += OnOperationExecuted;
            this.listAdapter.PreviewPlay += OnPreviewPlay;
            this.listAdapter.PreviewMute += OnPreviewMute;
            this.listAdapter.PreviewSoloPlay += OnPreviewSoloPlay;
            this.listAdapter.Setters.Add(ProjectParameterNames.SndEdit, SndEdit.SetSndEdit);
            this.listAdapter.ListItemCreator = delegate (Component component)
                {
                    if (component is Instrument)
                    {
                        SoundSetBankBase[] soundSetBanks = null;

                        ImaginaryInstrument imaginaryInstrument = component as ImaginaryInstrument;
                        if (imaginaryInstrument != null)
                        {
                            string bankFilePath = imaginaryInstrument.BankFilePath.ToLower();
                            if (this.soundSetBankDictionary.ContainsKey(bankFilePath) == true)
                            {
                                soundSetBanks = this.soundSetBankDictionary[bankFilePath].ToArray();
                            }
                        }


                        return new ImaginaryInstrumentListItem(component, soundSetBanks);
                    }
                    else
                    {
                        //
                        CommonListItem item = new FindResultListItem(component);
                        item.ConstValueGetters.Add("SndEditState", SndEdit.GetSndEditState);

                        return item;
                    }
                };

            //
            this.soundSetBankDictionary = new Dictionary<string, List<SoundSetBankBase>>();

            //
            this.listCtrl = ListControlCreator();
            this.listCtrl.Dock = DockStyle.Fill;
            this.listCtrl.Parent = this.pnl_List;
            this.listCtrl.AllowDragItem = true;
            this.listCtrl.AttachedContextMenuStrip = this.listMenu;
            this.listCtrl.DropOnDragItem = false;
            this.listCtrl.DecideFileDropPosition = false;
            this.listCtrl.FollowSelectedItemToCaret = true;
            this.listCtrl.InterlockSelectedItemAndSubItem = true;
            this.listCtrl.MoveNewLineCaretLeftAndRight = true;
            this.listCtrl.MoveNextLineCaretAfterEndEditByEnter = false;
            this.listCtrl.StopCaretLeftAndRight = false;
            this.listCtrl.SuspendTransaction = false;
            this.listCtrl.HeaderHeight = 28;
            this.listCtrl.HeaderCtrl.SortEnabled = true;
            this.listCtrl.ItemsSource = this.listAdapter;
            this.listCtrl.SelectChanged += OnSelectChangedListItem;
            this.listCtrl.ItemDoubleClicked += OnListItemDoubleClicked;
            //this.listCtrl.QueryFileDropped += OnQueryFileDropped;
            //this.listCtrl.FileDropped += OnFileDropped;
            this.listCtrl.Enter += OnEnter;
            this.listCtrl.Leave += OnLeave;
            this.listCtrl.EditBegan += OnEditBegan;
            this.listCtrl.EditEnded += OnEditEnded;

            this.listCtrl.ContentDrawer = new FindResultListContentDrawer();
            this.listCtrl.PartDrawers.Replace(new FindResultListPartCheckBoxDrawer());

            foreach (ListColumnStyle columnStyle in FindResultSettings.ExtendListColumnStyles)
            {
                this.listCtrl.ColumnStyles.Add(columnStyle);
            }

            _HeaderContextMenuItem_AddPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_AddPresetListColumns);
            _HeaderContextMenuItem_AddPresetListColumns.Click +=
                OnHeaderMenuItemAddPresetListColumnsClick;
            _HeaderContextMenuItem_ApplyPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_ApplyPresetListColumns);
            this.listCtrl.HeaderCtrl.InsertContextMenuItems =
                new ToolStripMenuItem[] { _HeaderContextMenuItem_AddPresetListColumns,
                                          _HeaderContextMenuItem_ApplyPresetListColumns };
            this.listCtrl.HeaderCtrl.ContextMenuOpening += OnHeaderCtrlContextMenuOpening;

            //
            InitializeCommandBindings();

            ApplySoundSetConfiguration();
            UpdatedOptions();
            UpdateReadOnly();
            UpdateTitle();
            UpdateTargetSoundSetNames();
            UpdateUI();

            //
            ProjectService.Opened += OnProjectOpened;
            ProjectService.Closed += OnProjectClosed;

            //
            MultipleBankEditDialogInplaceEditor editor = null;

            editor = listCtrl.InplaceEditors["BankDialog"] as
                MultipleBankEditDialogInplaceEditor;
            editor.Execute = ExecuteEditMultipleBank;

            // 編集不可セルの描画方法を変更します。
            this.listCtrl.PartDrawers.Replace(new NAListPartDrawer());

            this.SelectChangedListItem += this.OnSelectChanged;

            this.textDrawer = new ListPartTextDrawer();
            this.rowHeaderDrawer = new ListPartCommonRowHeaderDrawer();

            this.listCtrl.PartDrawers.Replace(this.textDrawer);
            this.listCtrl.PartDrawers.Replace(this.rowHeaderDrawer);

            // 非表示カラムを登録します。
            if (AppConfiguration.EnabledAAC == false)
            {
                excludeColumnNames.Add(ListTraits.ColumnName_ChannelNo);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void IEditableControlContainer.CancelEdit()
        {
            this.listCtrl.CancelEdit();
        }

        /// <summary>
        ///
        /// </summary>
        public FindArgs.TargetDocumentKinds TargetDocumentKind
        {
            get
            {
                return this.targetDocumentKind;
            }
            set
            {
                this.targetDocumentKind = value;

                bool enabled = true;
                switch (value)
                {
                    case FindArgs.TargetDocumentKinds.SoundSet:
                        break;

                    case FindArgs.TargetDocumentKinds.Bank:
                        enabled = false;
                        break;
                }

                this.btn_SelectSoundSet.Enabled = enabled;
                this.btn_SelectSoundSet.Visible = enabled;
                this.tbx_TargetSoundSet.Enabled = enabled;
                this.tbx_TargetSoundSet.Visible = enabled;
                this.chk_ItemReference.Enabled = enabled;
                this.chk_ItemReference.Visible = enabled;
                this.chk_StartPosition.Enabled = enabled;
                this.chk_StartPosition.Visible = enabled;

                SetItemButtonEnabled(enabled);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string TitleName
        {
            get
            {
                return this.titleName;
            }
            set
            {
                this.titleName = value;
                UpdateTitle();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int TitleNumber
        {
            get
            {
                return this.titleNumber;
            }
            set
            {
                this.titleNumber = value;
                UpdateTitle();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string Keyword
        {
            get
            {
                if (this.InvokeRequired == true)
                {
                    return (string)this.Invoke
                        ((GetStringMethodInvoker)delegate ()
                           {
                               return this.cmb_Keyword.Text;
                           });
                }
                else
                {
                    return this.cmb_Keyword.Text;
                }
            }
            set
            {
                this.cmb_Keyword.Text = value;
            }
        }

        public override CommonListCtrl[] ListControls
        {
            get
            {
                return new[] { this.listCtrl };
            }
        }

        /// <summary>
        /// セルの装飾に関する情報を提供します。
        /// </summary>
        protected CommonListDecorationEvaluator ListDecorationEvaluator { get; } =
            new CommonListDecorationEvaluator();

        /// <summary>
        /// 検索対象のパラメータ名を取得します。
        /// </summary>
        public string[] GetTargetParameterNames()
        {
            Dictionary<string, CheckBox> dictionary = GetTargetParameterCheckBoxs();
            List<string> list = new List<string>();

            foreach (KeyValuePair<string, CheckBox> pair in dictionary)
            {
                if (pair.Value.Checked == true)
                {
                    list.Add(pair.Key);
                }
            }
            return list.ToArray();
        }

        /// <summary>
        /// 検索対象をパラメータ名によって設定します。
        /// </summary>
        public void SetTargetParameterNames(string[] names)
        {
            Dictionary<string, CheckBox> dictionary = GetTargetParameterCheckBoxs();

            foreach (CheckBox checkBox in dictionary.Values)
            {
                checkBox.Checked = false;
            }

            foreach (string name in names)
            {
                if (dictionary.ContainsKey(name) == true)
                {
                    dictionary[name].Checked = true;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string[] GetTargetFilterNames()
        {
            Dictionary<string, ToolStripButton> dictionary = GetTargetFilterToolStripButtons();
            List<string> list = new List<string>();

            foreach (KeyValuePair<string, ToolStripButton> pair in dictionary)
            {
                if (pair.Value.Checked == true)
                {
                    list.Add(pair.Key);
                }
            }
            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        public void SetTargetFilterNames(string[] names)
        {
            Dictionary<string, ToolStripButton> dictionary = GetTargetFilterToolStripButtons();

            try
            {
                this.filterUpdateLocked = true;

                foreach (ToolStripButton button in dictionary.Values)
                {
                    button.Checked = false;
                }

                foreach (string name in names)
                {
                    if (dictionary.ContainsKey(name) == true)
                    {
                        dictionary[name].Checked = true;
                    }
                }
            }
            finally
            {
                this.filterUpdateLocked = false;
            }

            UpdateFilter();
        }

        /// <summary>
        /// 検索対象サウンドセットの名前を取得します。
        /// </summary>
        public string[] GetTargetSoundSetNames()
        {
            return this.targetSoundSetNames;
        }

        /// <summary>
        /// 検索対象サウンドセットを名前によって設定します。
        /// </summary>
        public void SetTargetSoundSetNames(string[] names)
        {
            if (names == null)
            {
                this.targetSoundSetNames = GetSoundSetNames();
            }
            else
            {
                HashSet<string> hashSet = new HashSet<string>();
                foreach (string name in GetSoundSetNames())
                {
                    hashSet.Add(name);
                }

                this.targetSoundSetNames = names
                    .Where(n => hashSet.Contains(n))
                    .ToArray();
            }

            UpdateUI();
        }

        /// <summary>
        /// アクティブになった時に呼ばれます。
        /// </summary>
        public override void Activated()
        {
            SetSelectedItemsToParameterPanel(this.listCtrl);
            ParameterPanel.SetFindResultPanel(this);
        }

        /// <summary>
        /// ディアクティブになった時に呼ばれます。
        /// </summary>
        public override void Deactivated()
        {
        }

        /// <summary>
        /// クリップボードが更新された時に呼ばれる
        /// </summary>
        public override void UpdatedClipboard()
        {
            this.listCtrl.ClearDashLine();
        }

        /// <summary>
        /// プロジェクト設定によりユーザーパラメータが変更された時に呼ばれます。
        /// </summary>
        public override void ApplyUserParameterColumns()
        {
            this.listCtrl.UpdateUserParameterColumnStyles();

            ListHeaderAdapterDictionary listHeader = new ListHeaderAdapterDictionary();
            listHeader.Add(ListTraits.ListName_FindResult2, this.listCtrl.HeaderSource as ListHeaderAdapterBase);

            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(listHeader);

            XmlList xmlList = new XmlList();
            xmlList.Name = ListTraits.ListName_FindResult2;
            xmlList.ListColumns = new XmlListColumns();
            xmlList.ListColumns.ListColumn = new XmlListColumn[0];

            applier.Extract(xmlList);

            applier.Apply(xmlList);

            this.listCtrl.UpdateInfos();
            this.listCtrl.Invalidate();
        }

        /// <summary>
        /// プロジェクト設定によりカラム名が変更された時に呼ばれます。
        /// </summary>
        public override void ApplyColumnsText()
        {
            ListHeaderAdapterDictionary listHeader = new ListHeaderAdapterDictionary();
            listHeader.Add(ListTraits.ListName_FindResult2, this.listCtrl.HeaderSource as ListHeaderAdapterBase);

            CommentColumnTextApplier.Apply(listHeader.Values.ToArray());
            this.listCtrl.InvalidateHeader();
        }

        /// <summary>
        /// オプション設定が変更された時に呼ばれます。
        /// </summary>
        public override void ApplyOptionSettings()
        {
            this.listCtrl.ApplyOptionConfigurations();
        }

        /// <summary>
        ///
        /// </summary>
        public override void UpdatedOptions()
        {
            this.listCtrl.UpdatePartDrawerOptions();
        }

        /// <summary>
        ///
        /// </summary>
        public override void ApplySettings(SettingKinds kind)
        {
            switch (kind)
            {
                case CommonListPanel.SettingKinds.ColorComment:
                    CommandTextReplacer.ReplaceColorCommentCommandTexts
                        (ProjectService, this.listMenuAdapter);
                    break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void CommandExecuted(Command command)
        {
            //
            if (command == EditCommands.CopyCell)
            {
                this.listCtrl.SetDashLineBySelectedSubItem();
            }

            //
            if (command == ToolCommands.ShowOptions ||
                command == ProjectCommands.ShowProjectSettings)
            {
                MainWindow.UpdatedOptions();
                UpdateToolBarColorTexts();
            }

            //
            if (command == EditCommands.EnableItem ||
                command == EditCommands.DisableItem)
            {
                MainWindow.ProjectPanel.UpdateProjectCtrl();
            }

            //
            RedrawControls();
            //MainWindow.RedrawPanels();
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        public override Component[] TargetComponents
        {
            get
            {
                return this.listCtrl.GetItemSelecteds().Keys
                    .Cast<ComponentListItem>()
                    .Select(i => i.Target)
                    .ToArray();
            }
        }

        /// <summary>
        /// 再描画します。
        /// </summary>
        public override void RefreshPanel()
        {
            this.listCtrl.Invalidate();
        }

        /// <summary>
        /// タイトルを更新します。
        /// </summary>
        public override void UpdateTitle()
        {
            if (this.InvokeRequired == true)
            {
                this.Invoke(new MethodInvoker
                             (delegate ()
                                  {
                                      UpdateTitleInternal();
                                  }));
            }
            else
            {
                UpdateTitleInternal();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void RedrawPanel()
        {
            RedrawControls();
        }

        /// <summary>
        /// アクティブになった時に呼ばれます。
        /// </summary>
        public override void ActivatePanel(bool resetSelection, bool mainList)
        {
            this.cmb_Keyword.Select();
        }

        /// <summary>
        ///
        /// </summary>
        public override void OnClosed()
        {
            base.OnClosed();
        }

        /// <summary>
        ///
        /// </summary>
        public override void ApplyProjectConfigurations()
        {
            ApplySoundSetConfiguration();
        }

        /// <summary>
        /// プリセット名からをプリセットを適用します。
        /// </summary>
        public override void ApplyPresetListColumns(string presetName)
        {
            string listName = ListTraits.ListName_FindResult;

            ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
            headerAdapters[listName] = this.listCtrl.HeaderSource as ListHeaderAdapterBase;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.ApplyPresetListColumnsWithoutAdd(applier, listName, presetName);

            this.listCtrl.UpdateInfos();

            MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = presetName;
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            preset.SetCurrentPresetName(listName, presetName);
        }

        /// <summary>
        /// プリセットを更新します。
        /// </summary>
        public override void UpdatePresetListColumns()
        {
            this.UpdatePresetListColumnsSplitButton();
        }

        /// <summary>
        /// カラムの状態を取得します。
        /// </summary>
        public XmlList GetHeaderColumnSetting()
        {
            ListHeaderAdapterDictionary listHeader = new ListHeaderAdapterDictionary();
            listHeader.Add(ListTraits.ListName_FindResult2, this.listCtrl.HeaderSource as ListHeaderAdapterBase);

            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(listHeader);

            XmlList xmlList = new XmlList();
            xmlList.Name = ListTraits.ListName_FindResult2;
            xmlList.ListColumns = new XmlListColumns();
            xmlList.ListColumns.ListColumn = new XmlListColumn[0];

            applier.Extract(xmlList);

            return xmlList;
        }

        /// <summary>
        /// カラムの状態を設定します。
        /// </summary>
        public void SetHeaderColumnSetting(XmlList xmlList)
        {
            ListHeaderAdapterDictionary listHeader = new ListHeaderAdapterDictionary();
            listHeader.Add(ListTraits.ListName_FindResult2, this.listCtrl.HeaderSource as ListHeaderAdapterBase);

            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(listHeader);

            applier.Apply(xmlList);

            this.listCtrl.UpdateInfos();
            this.listCtrl.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelOperationExecutedListener.OperationExecuted(OperationExecutedEventArgs e)
        {
            SetToTransaction(e.Operation as SetParameterOperation);
            //RedrawPanel();
        }

        /// <summary>
        ///
        /// </summary>
        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                case CommandParameterNames.TargetDocuments:
                case CommandParameterNames.TargetComponents:
                //case CommandParameterNames.InsertParentComponent:
                //case CommandParameterNames.InsertTargetComponent:
                case "ColumnNames":
                //case CaretItemSetter.CommandParameterOfListCtrl:
                case "Instance":
                    return true;
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                    return TargetComponentService;

                case CommandParameterNames.TargetDocuments:
                    return TargetDocuments;

                case CommandParameterNames.TargetComponents:
                    return TargetComponents;

                case "ColumnNames":
                    return this.listCtrl.ItemsName
                        .Where(s => s != "RowHeader")
                        .ToArray();

                case "Instance":
                    return this;
            }

            throw new KeyNotFoundException();
        }

        /// <summary>
        ///
        /// </summary>
        protected override ComponentService ComponentService
        {
            get { return ProjectService; }
        }

        /// <summary>
        ///
        /// </summary>
        private SoundProjectService ProjectService
        {
            get
            {
                if (FormsApplication.Instance == null)
                {
                    return null;
                }

                return FormsApplication.Instance.ProjectService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        /// <summary>
        ///
        /// </summary>
        private ParameterPanel ParameterPanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private ComponentService TargetComponentService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        /// <summary>
        ///
        /// </summary>
        private Document[] TargetDocuments
        {
            get
            {
                SoundDocument[] documents = TargetComponents
                    .Select(c => GetDocument(c))
                    .Where(d => d != null)
                    .ToArray();
                return documents;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Dictionary<string, CheckBox> GetTargetParameterCheckBoxs()
        {
            Dictionary<string, CheckBox> dictionary = new Dictionary<string, CheckBox>();
            dictionary.Add(ParameterName_Name, this.chk_Name);
            dictionary.Add(ParameterName_FilePath, this.chk_FilePath);
            dictionary.Add(ParameterName_Comment, this.chk_Comment);
            dictionary.Add(ParameterName_ItemReference, this.chk_ItemReference);
            dictionary.Add(ParameterName_StartPosition, this.chk_StartPosition);
            dictionary.Add(ParameterName_CaseSensitive, this.chk_CaseSensitive);
            dictionary.Add(ParameterName_FindExactMatch, this.chk_FindExactMatch);
            return dictionary;
        }

        /// <summary>
        ///
        /// </summary>
        private Dictionary<string, ToolStripButton> GetTargetFilterToolStripButtons()
        {
            Dictionary<string, ToolStripButton> dictionary = new Dictionary<string, ToolStripButton>();
            dictionary.Add(FilterName_EnabledItem, this.tool_EnabledItem);
            dictionary.Add(FilterName_StreamSound, this.tool_Stream);
            dictionary.Add(FilterName_StreamSoundTrack, this.tool_StreamSoundTrack);
            dictionary.Add(FilterName_WaveSoundSet, this.tool_WaveSet);
            dictionary.Add(FilterName_WaveSound, this.tool_Wave);
            dictionary.Add(FilterName_SequenceSoundSet, this.tool_SequenceSet);
            dictionary.Add(FilterName_SequenceSound, this.tool_Sequence);
            dictionary.Add(FilterName_Bank, this.tool_Bank);
            dictionary.Add(FilterName_WaveArchive, this.tool_WaveArchive);
            dictionary.Add(FilterName_Group, this.tool_Group);
            dictionary.Add(FilterName_Player, this.tool_Player);
            dictionary.Add(FilterName_ColorNone, this.tool_ColorNone);
            dictionary.Add(FilterName_ColorRed, this.tool_ColorRed);
            dictionary.Add(FilterName_ColorOrange, this.tool_ColorOrange);
            dictionary.Add(FilterName_ColorYellow, this.tool_ColorYellow);
            dictionary.Add(FilterName_ColorGreen, this.tool_ColorGreen);
            dictionary.Add(FilterName_ColorSkyBlue, this.tool_ColorSkyBlue);
            dictionary.Add(FilterName_ColorBlue, this.tool_ColorBlue);
            dictionary.Add(FilterName_ColorPurple, this.tool_ColorPurple);
            dictionary.Add(FilterName_ColorPink, this.tool_ColorPink);
            return dictionary;
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateTargetSoundSetNames()
        {
            this.targetSoundSetNames = GetSoundSetNames();
        }

        /// <summary>
        ///
        /// </summary>
        private string[] GetSoundSetNames()
        {
            return ProjectService.SoundSetDocuments
                .Select(d => d.SoundSet.Name)
                .ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateUI()
        {
            string text = String.Empty;
            if (this.targetSoundSetNames != null &&
                this.targetSoundSetNames.Length > 0)
            {
                string[] soundSetNames = GetSoundSetNames();
                if (soundSetNames.Length == this.targetSoundSetNames.Length)
                {
                    text = MessageResource.FindWindowText_Project;
                }
                else
                {
                    text = this.targetSoundSetNames[0];
                    for (int index = 1; index < this.targetSoundSetNames.Length; index++)
                    {
                        text = String.Format("{0}, {1}", text, this.targetSoundSetNames[index]);
                    }
                }
            }
            this.tbx_TargetSoundSet.Text = text;
        }

        /// <summary>
        /// コマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnCommandExecuted(object sender, CommandEventArgs e)
        {
            this.listCtrl.Invalidate();

            MainWindow.RequestNotifyCommandExecuted(e.Command);
            MainWindow.BuildCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnFilterCheckedChanged(object sender, EventArgs e)
        {
            UpdateFilter();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnProjectOpened(object sender, EventArgs e)
        {
            UpdateTargetSoundSetNames();
            UpdateUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnProjectClosed(object sender, EventArgs e)
        {
            SetResult(null);
        }

        /// <summary>
        /// コントロールがロードされると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnLoad(EventArgs e)
        {
#if DEBUG
            // WindowsForm デザイナ対応
            if (this.DesignMode == true)
            {
                return;
            }
#endif

            base.OnLoad(e);

            InitializeMenu();
            ApplicationBase.Instance.ProjectConfiguration.Saving += OnProjectConfigurationSaving;

            // Load直後のリストにフォーカスが渡るタイミングを検知する術がないので、
            // BeginInvoke でタイミングを遅らせる
            BeginInvoke(new MethodInvoker(BuildCommandUIs));
        }

        /// <summary>
        /// コントロールが入力されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            BuildListCommandUI();
            this.UpdatePresetListColumnsSplitButton();
        }

        /// <summary>
        /// リストの行ヘッダがダブルクリックされたら呼ばれます。
        /// </summary>
        protected virtual void ExecuteItem(ComponentListItem item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            if (item.Target is SoundSetBankBase)
            {
                SoundSetBankBase bank = item.Target as SoundSetBankBase;

                string validFilePath = new ReferenceFileFinder()
                {
                    Text = bank.FilePath,
                }.Validate(bank.FilePath);

                if (null == validFilePath) { return; }

                if (validFilePath != bank.FilePath)
                {
                    Operation operation = new SetParameterOperation
                        (item.Target.Parameters,
                          ProjectParameterNames.FilePath, validFilePath);
                    operation.Execute();
                }

                MainWindow.OpenFile(bank);
            }

            // グループアイテムでは、対象アイテムにジャンプします。(2012.5.31 aoyagi)
            if (item.Target is GroupItemBase)
            {
                GroupItemBase groupItem = item.Target as GroupItemBase;
                MainWindow.ShowPageByComponent(groupItem.Target);
            }
        }

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        private void InitializeCommandBindings()
        {
            BindCommand(EditCommands.Undo.ID,
                         QueryStatusUndo, ExecuteUndo);
            BindCommand(EditCommands.Redo.ID,
                         QueryStatusRedo, ExecuteRedo);

            BindCommand(new OpenItemFolderHandler(this));
            BindCommand(new ExecuteItemFolderHandler(this));
            BindCommand(new OpenBankItemHandler(this));

            BindCommand(new CopyHandler(this));
            BindCommand(EditCommands.CopyCell.ID,
                         QueryStatusCopyCell, ExecuteCopyCell);
            BindCommand(EditCommands.Paste.ID,
                         QueryStatusPaste, ExecutePaste);

            BindCommand(PreviewCommands.Play.ID,
                         QueryStatusPlay, ExecutePlay);

            BindCommand(EditCommands.SelectAll.ID,
                         QueryStatusSelectAll, ExecuteSelectAll);

            BindCommand(EditCommands.EnableItem.ID,
                         QueryStatusEnableItem, ExecuteEnableItem);
            BindCommand(EditCommands.DisableItem.ID,
                         QueryStatusDisableItem, ExecuteDisableItem);
            BindCommand(EditCommands.SetItemColor.ID,
                         QueryStatusSetItemColor, ExecuteSetItemColor);

            BindCommand(EditCommands.BatchEdit.ID,
                         QueryStatusBatchEdit, ExecuteBatchEdit);

            /// デフォルト非表示です。表示するには派生先で BindCommand で上書きしてください。
            BindCommand(EditCommands.UseWaveSoundWaveFile.ID, (c) => CommandStatus.Supported, (c) => true);
            BindCommand(EditCommands.UseWaveSoundResource.ID, (c) => CommandStatus.Supported, (c) => true);
        }

        /// <summary>
        ///
        /// </summary>
        protected override string ConvertText(string name, string text)
        {
            if (name == ProjectParameterNames.WaveArchive.LoadType)
            {
                if (text.ToLower() == "true")
                {
                    IParameterValue individual =
                        new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Individual);
                    text = individual.ToString();
                }
                else if (text.ToLower() == "false")
                {
                    IParameterValue whole =
                        new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Whole);
                    text = whole.ToString();
                }
            }

            return text;
        }

        /// <summary>
        ///
        /// </summary>
        protected override IParameterValue ConvertValue(
                                                        string name,
                                                        string text,
                                                        IParameterValue value
                                                        )
        {
            if (name == ProjectParameterNames.WaveArchive.LoadType)
            {
                IParameterValue individual =
                    new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Individual);
                if (text == individual.ToString())
                {
                    return individual;
                }

                IParameterValue whole =
                    new WaveArchiveLoadTypeParameterValue(WaveArchiveLoadType.Whole);
                if (text == whole.ToString())
                {
                    return whole;
                }
            }

            return value;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool IsSupportWaveFile(string filePath)
        {
            try
            {
                WaveFileReader.CreateInstance(filePath);
            }
            catch
            {
                return false;
            }
            return true;
        }

        /// <summary>
        /// ツールバー、カラーボタンのテキストを更新します。
        /// </summary>
        private void UpdateToolBarColorTexts()
        {
            foreach (ButtonPair<int> pair in ButtonColorIndexPairs)
            {
                string text = String.Empty;

                if (ProjectService.Project != null)
                {
                    switch (pair.Value)
                    {
                        case 0: text = ProjectService.Project.ColorComment0; break;
                        case 1: text = ProjectService.Project.ColorComment1; break;
                        case 2: text = ProjectService.Project.ColorComment2; break;
                        case 3: text = ProjectService.Project.ColorComment3; break;
                        case 4: text = ProjectService.Project.ColorComment4; break;
                        case 5: text = ProjectService.Project.ColorComment5; break;
                        case 6: text = ProjectService.Project.ColorComment6; break;
                        case 7: text = ProjectService.Project.ColorComment7; break;
                        case 8: text = ProjectService.Project.ColorComment8; break;
                        default: Debug.Assert(false, "Illigal index"); break;
                    }
                }

                if (text == String.Empty)
                {
                    text = System.Text.RegularExpressions.Regex.Replace
                        (CommandTextReplacer.DefaultColorComments[pair.Value],
                          "\\(&.+\\)", String.Empty);
                }

                pair.Button.ToolTipText = text;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Bitmap CreateToolBarBitmap(Bitmap bitmap)
        {
            return bitmap.MakeNewTransparent(UIServiceBase.TransparentColor);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSetAllButtonCheck(object sender, EventArgs e)
        {
            SetItemButtonCheck(true);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnResetAllButtonCheck(object sender, EventArgs e)
        {
            SetItemButtonCheck(false);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetItemButtonCheck(bool b)
        {
            try
            {
                this.filterUpdateLocked = true;

                this.tool_Stream.Checked = b;
                this.tool_StreamSoundTrack.Checked = b;
                this.tool_WaveSet.Checked = b;
                this.tool_Wave.Checked = b;
                this.tool_SequenceSet.Checked = b;
                this.tool_Sequence.Checked = b;
                this.tool_Bank.Checked = b;
                this.tool_WaveArchive.Checked = b;
                this.tool_Group.Checked = b;
                this.tool_Player.Checked = b;

                this.tool_ColorNone.Checked = b;
                this.tool_ColorRed.Checked = b;
                this.tool_ColorOrange.Checked = b;
                this.tool_ColorYellow.Checked = b;
                this.tool_ColorGreen.Checked = b;
                this.tool_ColorSkyBlue.Checked = b;
                this.tool_ColorBlue.Checked = b;
                this.tool_ColorPurple.Checked = b;
                this.tool_ColorPink.Checked = b;
            }
            finally
            {
                this.filterUpdateLocked = false;
            }

            UpdateFilter();
        }

        /// <summary>
        ///
        /// </summary>
        private void SetItemButtonEnabled(bool b)
        {
            this.tool_Stream.Enabled = b;
            this.tool_Stream.Visible = b;
            this.tool_StreamSoundTrack.Enabled = b;
            this.tool_StreamSoundTrack.Visible = b;
            this.tool_WaveSet.Enabled = b;
            this.tool_WaveSet.Visible = b;
            this.tool_Wave.Enabled = b;
            this.tool_Wave.Visible = b;
            this.tool_SequenceSet.Enabled = b;
            this.tool_SequenceSet.Visible = b;
            this.tool_Sequence.Enabled = b;
            this.tool_Sequence.Visible = b;
            this.tool_Bank.Enabled = b;
            this.tool_Bank.Visible = b;
            this.tool_WaveArchive.Enabled = b;
            this.tool_WaveArchive.Visible = b;
            this.tool_Group.Enabled = b;
            this.tool_Group.Visible = b;
            this.tool_Player.Enabled = b;
            this.tool_Player.Visible = b;

            this.separator_Item.Enabled = b;
            this.separator_Item.Visible = b;

            //
            this.toolLabel_ListFilter.Enabled = b;
            this.toolLabel_ListFilter.Visible = b;
            this.tool_CreateColumn.Enabled = b;
            this.tool_CreateColumn.Visible = b;
            this.separator_ListFilter.Enabled = b;
            this.separator_ListFilter.Visible = b;
        }

        /// <summary>
        /// リストの選択状態が変更された時に呼ばれます
        /// </summary>
        private void OnSelectChangedListItem(object sender, EventArgs e)
        {
            if (this.SelectChangedListItem != null)
            {
                this.SelectChangedListItem(sender, e);
            }
        }

        private void OnSelectChanged(object sender, EventArgs e)
        {
            MainWindow.BuildCommandUI();
            SetSelectedItemsToParameterPanel(this.listCtrl);
            UpdateStatusBar(this.listCtrl);
            ParameterPanel.SetFindResultPanel(this);
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnListItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ComponentListItem item = e.Item as ComponentListItem;
            Component component = item.Target;

            if (item is ImaginaryInstrumentListItem == true)
            {
                ImaginaryInstrumentListItem instItem = item as ImaginaryInstrumentListItem;
                component = instItem.ImaginaryInstrument;
            }

            MainWindow.ShowPageByComponent(component);
        }

        /// <summary>
        ///
        /// </summary>
        private bool ContainsHeaderColumn(List<HeaderColumnData> list, string name)
        {
            foreach (HeaderColumnData column in list)
            {
                if (column.Name == name)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExcludeHeaderColumn(string name)
        {
            return this.excludeColumnNames.Contains(name);
        }

        /// <summary>
        ///
        /// </summary>
        private void AddHeaderColumnData(List<HeaderColumnData> list, string name)
        {
            ListHeaderAdapterBase adapter =
                FormsApplication.Instance.UIService.DefaultHeaderAdapters[name];

            foreach (HeaderColumnData column in adapter.Columns)
            {
                if (ExcludeHeaderColumn(column.Name) == false &&
                    ContainsHeaderColumn(list, column.Name) == false)
                {
                    list.Add(column);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowHeaderColumns(IHeaderSource headerSource, string name)
        {
            ListHeaderAdapterBase adapter =
                FormsApplication.Instance.UIService.DefaultHeaderAdapters[name];

            foreach (HeaderColumnData column in adapter.Columns)
            {
                if (column.Visible == true)
                {
                    ShowHeaderColumn(headerSource, column.Name);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowHeaderColumn(IHeaderSource headerSource, string name)
        {
            IHeaderItem item = GetHeaderItem(headerSource, name);
            if (item != null)
            {
                item.Visible = true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private IHeaderItem GetHeaderItem(IHeaderSource headerSource, string name)
        {
            foreach (IHeaderItem item in headerSource.Items)
            {
                if (item.Name == name)
                {
                    return item;
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private HeaderColumnData GetHeaderColumnData(string name, string columnName)
        {
            ListHeaderAdapterBase adapter =
                FormsApplication.Instance.UIService.DefaultHeaderAdapters[name];

            foreach (HeaderColumnData column in adapter.Columns)
            {
                if (column.Name == columnName)
                {
                    return column;
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private Instrument[] GetInstrument(Component[] components)
        {
            return components
                .Where(c => c is Instrument)
                .Cast<Instrument>()
                .ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private class ButtonPair<T>
        {
            private ToolStripButton button = null;
            private T value;

            public ButtonPair(ToolStripButton button, T value)
            {
                this.button = button;
                this.value = value;
            }

            public ToolStripButton Button
            {
                get { return this.button; }
            }

            public T Value
            {
                get { return this.value; }
            }

        }

        /// <summary>
        ///
        /// </summary>
        private class ButtonTypePair : ButtonPair<Type>
        {
            public ButtonTypePair(ToolStripButton button, Type type) : base(button, type)
            {
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsMatch(Component component)
        {
            ButtonTypePair[] buttonTypePair = new ButtonTypePair[] {
                new ButtonTypePair( this.tool_Stream,      typeof(StreamSoundBase)),
                new ButtonTypePair( this.tool_StreamSoundTrack, typeof(StreamSoundTrackBase)),
                new ButtonTypePair( this.tool_WaveSet,     typeof(WaveSoundSetBase)),
                new ButtonTypePair( this.tool_Wave,        typeof(WaveSoundBase)),
                new ButtonTypePair( this.tool_SequenceSet, typeof(SequenceSoundSetBase)),
                new ButtonTypePair( this.tool_Sequence,    typeof(SequenceSoundBase)),
                new ButtonTypePair( this.tool_Bank,        typeof(SoundSetBankBase)),
                new ButtonTypePair( this.tool_WaveArchive, typeof(WaveArchiveBase)),
                new ButtonTypePair( this.tool_Group,       typeof(GroupBase)),
                new ButtonTypePair( this.tool_Player,      typeof(PlayerBase)),
                //new ButtonTypePair( this.tool_Instrument,  typeof(ImaginaryInstrument)),
            };

            //
            if (this.tool_EnabledItem.Checked == true &&
                component.IsEnabled == false)
            {
                return false;
            }

            //
            foreach (ButtonPair<int> pair in ButtonColorIndexPairs)
            {
                if (pair.Button.Checked == false &&
                    component.ColorIndex == pair.Value)
                {
                    return false;
                }
            }

            //
            if (component is Instrument)
            {
                return true;
            }

            // アイテムの種類によるフィルタ処理です。
            foreach (ButtonTypePair pair in buttonTypePair)
            {
                if (pair.Button.Checked == true &&
                    HasAncestor(pair.Value, component.GetType()) == true)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private bool HasAncestor(Type ancestor, Type target)
        {
            do
            {
                if (ancestor == target)
                {
                    return true;
                }
                target = target.BaseType;
            } while (target != null);
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public void Reloading()
        {
            SetResult(null);
        }

        /// <summary>
        ///
        /// </summary>
        public void Reloaded()
        {
            if (this.unsearched == true) // 一度も検索していない場合は検索しない。
            {
                return;
            }

            Find();
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateResult()
        {
            this.listAdapter.InterlockComponent = false;
            this.listAdapter.Items.Clear();

            if (this.resultComponents != null)
            {
                Component[] components = this.resultComponents
                    .Where(c => IsMatch(c))
                    .ToArray();

                this.listAdapter.AddItems(components);
            }
        }

        /// <summary>
        /// 検索結果を設定します。
        /// </summary>
        public void SetResult(Component[] components)
        {
            SetResult(components, this.createHeaderColumnData);
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowControl(Control control)
        {
            control.Enabled = true;
            control.Visible = true;
        }

        /// <summary>
        ///
        /// </summary>
        private void HideControl(Control control)
        {
            control.Enabled = false;
            control.Visible = false;
        }

        /// <summary>
        /// 検索結果を設定します。
        /// </summary>
        private void SetResult(Component[] components, bool create)
        {
            List<HeaderColumnData> list = new List<HeaderColumnData>();
            bool editable = false;

            //
            this.soundSetBankDictionary.Clear();

            //
            if (components != null)
            {
                if (this.targetDocumentKind == FindArgs.TargetDocumentKinds.Bank)
                {
                    //
                    foreach (SoundSetBankBase soundSetBank in ProjectService.SoundSetBanks)
                    {
                        string filePath = Path.GetFullPath(soundSetBank.FilePath).ToLower();

                        if (this.soundSetBankDictionary.ContainsKey(filePath) == false)
                        {
                            this.soundSetBankDictionary.Add
                                (filePath, new List<SoundSetBankBase>());
                        }

                        this.soundSetBankDictionary[filePath].Add(soundSetBank);
                    }

                    //
                    if (create == true)
                    {
                        GetHeaderColumnDataForInstrument(list, components);
                    }

                    SetItemButtonEnabled(false);
                    HideControl(this.chk_ItemReference);
                    HideControl(this.chk_StartPosition);
                }
                else
                {
                    editable = true;
                    components = GetComponentsAtRelevanceComponent(components);

                    if (create == true)
                    {
                        GetHeaderColumnDatasForSound(list, components);
                    }

                    SetItemButtonEnabled(true);
                    ShowControl(this.chk_ItemReference);
                    ShowControl(this.chk_StartPosition);
                }

                //
                Tuple<string, string>[] insertNames = new Tuple<string, string>[] {
                    Tuple.Create<string, string>
                        ( ProjectParameterNames.StreamSoundTrack.TrackNo,
                          ProjectParameterNames.Name ),

                    Tuple.Create<string, string>
                        ( ProjectParameterNames.StreamSoundTrack.PreviewSoloPlay,
                          ProjectParameterNames.SoundSetItem.PreviewPlay),

                    Tuple.Create<string, string>
                        ( ProjectParameterNames.StreamSoundTrack.PreviewMute,
                          ProjectParameterNames.SoundSetItem.PreviewPlay),

                    Tuple.Create<string, string>
                        ( ListTraits.ColumnName_ChannelNo,
                          ProjectParameterNames.StreamSoundTrack.TrackNo ),
                };

                foreach (Tuple<string, string> insertName in insertNames)
                {
                    HeaderColumnData data = list.Find
                        (delegate (HeaderColumnData d)
                             {
                                 return d.Name == insertName.Item1;
                             });
                    if (data == null)
                    {
                        continue;
                    }

                    int index = list.FindIndex
                        (delegate (HeaderColumnData d)
                             {
                                 return d.Name == insertName.Item2;
                             });
                    if (index < 0)
                    {
                        continue;
                    }

                    list.Remove(data);
                    if (index < list.Count)
                    {
                        list.Insert(index + 1, data);
                    }
                    else
                    {
                        list.Add(data);
                    }
                }

                // コメント、ユーザーデータのカラムを最後尾に移動させます。
                HashSet<string> names = new HashSet<string> {
                    ProjectParameterNames.UserParameter,
                    ProjectParameterNames.UserParameter1,
                    ProjectParameterNames.UserParameter2,
                    ProjectParameterNames.UserParameter3,
                    ProjectParameterNames.Comment,
                    ProjectParameterNames.Comment1,
                    ProjectParameterNames.Comment2,
                    ProjectParameterNames.Comment3,
                    ProjectParameterNames.Comment4,
                    ProjectParameterNames.Comment5,
                    ProjectParameterNames.Comment6,
                    ProjectParameterNames.Comment7,
                    ProjectParameterNames.Comment8,
                    ProjectParameterNames.Comment9,
                };

                HeaderColumnData[] headerColumnDatas = list.ToArray();
                foreach (HeaderColumnData headerColumnData in headerColumnDatas)
                {
                    if (names.Contains(headerColumnData.Name) == true)
                    {
                        list.Remove(headerColumnData);
                        list.Add(headerColumnData);
                    }
                }
            }

            //
            if (create == true)
            {
                ListHeaderAdapterBase adapter = new CommonListHeaderAdapter();
                adapter.Columns = list.ToArray();
                CommentColumnTextApplier.Apply(adapter);
                this.listCtrl.HeaderSource = adapter;
            }

            //
            if (editable == true)
            {
                this.listCtrl.ReadOnly = false;
                //this.listCtrl.AttachedContextMenuStrip = this.listMenu;
            }
            else
            {
                this.listCtrl.ReadOnly = true;
                this.listCtrl.AttachedContextMenuStrip = null;
            }

            //
            this.resultComponents = components;

            //
            UpdateResult();
            ClearTransactions();
            this.UpdatePresetListColumnsSplitButton();
        }

        /// <summary>
        ///
        /// </summary>
        private void GetHeaderColumnDataForInstrument(List<HeaderColumnData> list, Component[] components)
        {
            HeaderColumnData headerColumnData = GetHeaderColumnData
                (ListTraits.ListName_SequenceSound,
                  ProjectParameterNames.SequenceSound.SoundSetBankReference0);
            list.Add(headerColumnData);

            AddHeaderColumnData(list, ListTraits.ListName_Instrument);
        }

        /// <summary>
        ///
        /// </summary>
        private void GetHeaderColumnDatasForSound(List<HeaderColumnData> list, Component[] components)
        {
            //
            AddHeaderColumnData(list, ListTraits.ListName_StreamSound);
            AddHeaderColumnData(list, ListTraits.ListName_StreamSoundTrack);
            AddHeaderColumnData(list, ListTraits.ListName_WaveSoundSet);
            AddHeaderColumnData(list, ListTraits.ListName_WaveSoundSetItem);
            AddHeaderColumnData(list, ListTraits.ListName_SequenceSoundSet);
            AddHeaderColumnData(list, ListTraits.ListName_SequenceSound);
            AddHeaderColumnData(list, ListTraits.ListName_Bank);
            AddHeaderColumnData(list, ListTraits.ListName_WaveArchive);
            AddHeaderColumnData(list, ListTraits.ListName_Group);
            AddHeaderColumnData(list, ListTraits.ListName_Player);

            //
            foreach (FindResultSettings.ExtendHeaderColumnData data in
                     FindResultSettings.ExtendHeaderColumnDatas)
            {
                int index = list.FindIndex
                    (delegate (HeaderColumnData d)
                         {
                             return d.Name == data.PreviousColumnName;
                         });
                if (index >= 0 && index < list.Count)
                {
                    list.Insert(index + 1, data.HeaderColumnData.Clone());
                }
                else
                {
                    list.Add(data.HeaderColumnData.Clone());
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowUsedHeaderColumns(IHeaderSource headerSource, Component[] components)
        {
            if (components == null)
            {
                return;
            }

            bool streamSoundContained = !this.tool_Stream.Checked;
            bool streamSoundTrackContained = !this.tool_StreamSoundTrack.Checked;
            bool waveSoundSetContained = !this.tool_WaveSet.Checked;
            bool waveSoundContained = !this.tool_Wave.Checked;
            bool sequenceSoundSetContained = !this.tool_SequenceSet.Checked;
            bool sequenceSoundContained = !this.tool_Sequence.Checked;
            bool bankContained = !this.tool_Bank.Checked;
            bool waveArchiveContained = !this.tool_WaveArchive.Checked;
            bool groupContained = !this.tool_Group.Checked;
            bool playerContained = !this.tool_Player.Checked;

            //
            foreach (IHeaderItem item in headerSource.Items)
            {
                if (item.Name != "RowHeader")
                {
                    item.Visible = false;
                }
            }

            //
            foreach (Component component in components)
            {
                if (component is StreamSoundBase && streamSoundContained == false)
                {
                    streamSoundContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_StreamSound);
                }
                else if (component is StreamSoundTrackBase && streamSoundTrackContained == false)
                {
                    streamSoundTrackContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_StreamSoundTrack);

                    foreach (FindResultSettings.ExtendHeaderColumnData data in
                             FindResultSettings.ExtendHeaderColumnDatas)
                    {
                        ShowHeaderColumn(headerSource, data.HeaderColumnData.Name);
                    }
                }
                else if (component is WaveSoundSetBase && waveSoundSetContained == false)
                {
                    waveSoundSetContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_WaveSoundSet);
                }
                else if (component is WaveSoundBase && waveSoundContained == false)
                {
                    waveSoundContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_WaveSoundSetItem);
                }
                else if (component is SequenceSoundSetBase && sequenceSoundSetContained == false)
                {
                    sequenceSoundSetContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_SequenceSoundSet);
                }
                else if (component is SequenceSoundBase && sequenceSoundContained == false)
                {
                    sequenceSoundContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_SequenceSound);
                }
                else if (component is SoundSetBankBase && bankContained == false)
                {
                    bankContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_Bank);
                }
                else if (component is WaveArchiveBase && waveArchiveContained == false)
                {
                    waveArchiveContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_WaveArchive);
                }
                else if (component is GroupBase && groupContained == false)
                {
                    groupContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_Group);
                }
                else if (component is PlayerBase && playerContained == false)
                {
                    playerContained = true;
                    ShowHeaderColumns(headerSource, ListTraits.ListName_Player);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateFilter()
        {
            if (this.filterUpdateLocked == true)
            {
                return;
            }

            UpdateResult();
        }

        /// <summary>
        // ストリームサウンドが含まれていた場合にはストリームサウンドトラックを追加します。
        /// </summary>
        private Component[] GetComponentsAtRelevanceComponent(Component[] components)
        {
            HashSet<Component> hashSet = new HashSet<Component>();
            List<Component> list = new List<Component>();

            foreach (Component component in components)
            {
                if (hashSet.Contains(component) == false)
                {
                    list.Add(component);
                    hashSet.Add(component);
                }

                //
                if (component is StreamSoundBase)
                {
                    foreach (Component childComponent in component.Children)
                    {
                        if (hashSet.Contains(childComponent) == false)
                        {
                            list.Add(childComponent);
                            hashSet.Add(childComponent);
                        }
                    }
                }
            }
            return list.ToArray();
        }

        /// <summary>
        /// オペレーションが実行された後に呼ばれます。
        /// </summary>
        private void OnOperationExecuted(object sender, OperationExecutedEventArgs e)
        {
            SetToTransaction(e.Operation as SetParameterOperation);
        }

        /// <summary>
        ///
        /// </summary>
        private void SetToTransaction(SetParameterOperation operation)
        {
            if (operation == null)
            {
                return;
            }

            Component component = FindComponent(operation.TargetDictionary,
                                                 operation.ParameterKey);
            SoundSet soundSet = GetSoundSet(component);

            SoundSetDocument soundSetDocument =
                FormsApplication.Instance.DocumentService.Documents
                .Where(d => d is SoundSetDocument)
                .Cast<SoundSetDocument>()
                .Where(d => d.SoundSet == soundSet)
                .First();

            UserTransaction2 transaction = new UserTransaction2(String.Empty);
            transaction.Document = soundSetDocument;

            soundSetDocument.OperationHistory.BeginTransaction(transaction);
            soundSetDocument.OperationHistory.AddOperation(operation);
            soundSetDocument.OperationHistory.EndTransaction();

            //
            PushTransactionPackStack(new UserTransaction2[] { transaction });

            //
            MainWindow.RedrawPanels();

            //
            foreach (FindResultPanel2 panel in MainWindow.FindResultPanels)
            {
                if (panel != this)
                {
                    panel.ClearTransactions();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void PushTransactionPackStack(UserTransaction2[] transactions)
        {
            if (transactions.Length <= 0)
            {
                return;
            }

            TransactionPack pack = new TransactionPack(transactions);
            this.transactionPackStack.Push(pack);

            ShowTransactionStackWarning(true);
        }

        /// <summary>
        ///
        /// </summary>
        public void ClearTransactions()
        {
            this.transactionPackStack.Clear();

            ShowTransactionStackWarning(false);
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowTransactionStackWarning(bool visible)
        {
            this.statusStrip.Visible = visible;
        }

        /// <summary>
        /// 指定した ParameterDictionaryを保持する Componentをリストから検索します。
        /// </summary>
        private Component FindComponent(IParameterDictionary parameterDictionary, string name)
        {
            foreach (ComponentListItem item in this.listAdapter.Items)
            {
                if (item.Target.Parameters == parameterDictionary)
                {
                    return item.Target;
                }
            }

            // リストの Component.Parametersに該当しない場合には、指定パラメータ名によって
            // 検索する場所を変えて、再度、検索を行います。
            switch (name)
            {
                case ProjectParameterNames.TargetName:
                    // 参照サンドセットバンク
                    {
                        foreach (ComponentListItem item in this.listAdapter.Items)
                        {
                            ObservableList<ComponentReference> references =
                                item.Target.Parameters[ProjectParameterNames.SequenceSound.SoundSetBankReferences].Value as ObservableList<ComponentReference>;

                            if (references != null)
                            {
                                foreach (ComponentReference reference in references)
                                {
                                    if (reference.Parameters == parameterDictionary)
                                    {
                                        return item.Target;
                                    }
                                }
                            }
                        }
                    }
                    break;
            }

            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSet GetSoundSet(Component component)
        {
            while (component.Parent != null)
            {
                if (component.Parent is SoundSet)
                {
                    return component.Parent as SoundSet;
                }
                component = component.Parent;
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected SoundDocument GetDocument(Component component)
        {
            // 現状は SoundSetDocumentのみ対応します。
            if (component is Instrument)
            {
                return null;
            }

            return GetSoundSetDocument(GetSoundSet(component));
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSetDocument GetSoundSetDocument(SoundSet soundSet)
        {
            if (soundSet == null)
            {
                return null;
            }

            SoundSetDocument soundSetDocument =
                FormsApplication.Instance.DocumentService.Documents
                .Where(d => d is SoundSetDocument)
                .Cast<SoundSetDocument>()
                .Where(d => d.SoundSet == soundSet)
                .First();
            return soundSetDocument;
        }

        /// <summary>
        /// マルチバンクの編集ダイアログを開きます。
        /// リストからの編集で呼ばれます。
        /// </summary>
        private void ExecuteEditMultipleBank(object sender, System.ComponentModel.CancelEventArgs e)
        {
        }

        /// <summary>
        /// コマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnSoundListCommandExecuted(object sender, CommandEventArgs e)
        {
            CommandExecuted(e.Command);
            MainWindow.BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEnterd(object sender, EventArgs e)
        {
            ListCtrl listCtrl = sender as CommonListCtrl;
            SetSelectedItemsToParameterPanel(listCtrl);

            BuildCommandUIs();

            this.UpdatePresetListColumns();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEnter(object sender, EventArgs e)
        {
            //this.UpdatePresetListColumnsSplitButton();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnLeave(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnDirtyChanged(object sender, EventArgs e)
        {
            MainWindow.ProjectPanel.UpdateProjectCtrl();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOperationHistoryItemsChanged(object sender, EventArgs e)
        {
            if (e is TransactionStackEventArgs)
            {
                TransactionStackEventArgs args = e as TransactionStackEventArgs;
                if ((args.Transaction is UserTransaction2) == false)
                {
                    ClearTransactions();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOperationHistoryCurrentItemChanged(object sender, EventArgs e)
        {
            OnOperationHistoryItemsChanged(sender, e);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEditBegan(object sender, EventArgs e)
        {
            ApplicationBase.Instance.DisableCommandKeyProcess();
            MainWindow.EnableAllCommands(false);
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEditEnded(object sender, EventArgs e)
        {
            ApplicationBase.Instance.EnableCommandKeyProcess();
            UpdateStatusBar(ActiveListCtrl);
            MainWindow.EnableAllCommands(true);
            //MainWindow.RedrawPanels();
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateStatusBar(CommonListCtrl listCtrl)
        {
            CommonListItem[] items = null;
            string message = String.Empty;

            if (listCtrl != null)
            {
                items = listCtrl.GetSelectedItems();
                if (items.Length == 1)
                {
                    message = items[0].FilePath;
                }
            }

            MainWindow.UpdateStatusText(message);
        }

        /// <summary>
        /// パラメータパネルに選択されているアイテムを設定します。
        /// </summary>
        private void SetSelectedItemsToParameterPanel(ListCtrl listCtrl)
        {
            if (this.targetDocumentKind == FindArgs.TargetDocumentKinds.SoundSet)
            {
                SetSelectedSoundSetItemsToParameterPanel(listCtrl);
            }
            else if (this.targetDocumentKind == FindArgs.TargetDocumentKinds.Bank)
            {
                SetSelectedInstrumentsToParameterPanel(listCtrl);
            }
        }

        private void SetSelectedSoundSetItemsToParameterPanel(ListCtrl listCtrl)
        {
            List<Component> items = new List<Component>();

            foreach (ComponentListItem item in listCtrl.SelectedItems)
            {
                if (item.Target is Sound || item.Target is StreamSoundTrackBase)
                {
                    items.Add(item.Target);
                }
            }

            ParameterPanel.Items = items.Count > 0 ? items.ToArray() : null;
        }

        private void SetSelectedInstrumentsToParameterPanel(ListCtrl listCtrl)
        {
            List<Component> instruments = new List<Component>();

            foreach (ComponentListItem item in listCtrl.SelectedItems)
            {
                if (item.Target is Instrument)
                {
                    instruments.Add(item.Target);
                }
            }

            ParameterPanel.Instrument = instruments.Count > 0 ? (Instrument)instruments[0] : null;
            ParameterPanel.VelocityRegion = null;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ExecuteItem(e.Item as ComponentListItem);
        }

        /// <summary>
        ///
        /// </summary>
        private class ListCtrlState
        {
            private Dictionary<Component, ListItemSelectedState> _ListItemSelectedStateDictionary;

            public ListCtrlState(ListItemSelectedDictionary dictionary)
            {
                _ListItemSelectedStateDictionary = new Dictionary<Component, ListItemSelectedState>();

                foreach (IListItem item in dictionary.Keys)
                {
                    if (item is CommonListItem == true)
                    {
                        CommonListItem common = item as CommonListItem;
                        Component component = common.Target;
                        _ListItemSelectedStateDictionary[component] = dictionary[item];
                    }
                }
            }

            public Component CaretComponent { get; set; }
            public string CaretColumnName { get; set; }
            public int VerticalScrollBarValue { get; set; }
            public int HorizontalScrollBarValue { get; set; }
            public Dictionary<Component, ListItemSelectedState> ListItemSelectedStateDictionary
            {
                get
                {
                    return _ListItemSelectedStateDictionary;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SaveListsState()
        {
        }

        /// <summary>
        ///
        /// </summary>
        private void SaveListState(CommonListCtrl ctrl, Dictionary<Component, ListCtrlState> stateDictionary, Component currentItem)
        {
        }

        /// <summary>
        ///
        /// </summary>
        private void LoadListsState()
        {
        }

        /// <summary>
        ///
        /// </summary>
        private void LoadListState(CommonListCtrl ctrl, CommonListAdapter adapter, Dictionary<Component, ListCtrlState> listCtrlStateDictionary, Component currentItem)
        {
        }

        /// <summary>
        ///
        /// </summary>
        private void SetSelectedList(ListItemSelectedDictionary dictionary,
                                     Dictionary<Component, ListItemSelectedState> selectDictionary,
                                     ComponentListItemCollection items)
        {
            foreach (CommonListItem item in items)
            {
                if (selectDictionary.ContainsKey(item.Target) == true)
                {
                    if (selectDictionary[item.Target].Selected == true)
                    {
                        dictionary.Select(item, true);
                    }
                    foreach (string name in selectDictionary[item.Target].SubSelected)
                    {
                        dictionary.SubSelect(item, name, true);
                    }
                }
            }
        }

        private void SetCaret(CommonListCtrl ctrl, ComponentListItemCollection items,
                              Component component, string columnName)
        {
            if (string.IsNullOrEmpty(columnName) == true)
            {
                return;
            }

            foreach (CommonListItem item in items)
            {
                if (item.Target == component)
                {
                    ctrl.SetCaretWithoutFollow(item, columnName);
                    break;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ResetListSeletion()
        {
            this.listCtrl.GetItemSelecteds().Clear();
            this.listCtrl.SetCaret();
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSetItem FindItem(string targetName, Component parentComponent)
        {
            if (null == targetName) { throw new ArgumentNullException("targetName"); }
            if (null == parentComponent) { throw new ArgumentNullException("parentComponent"); }

            foreach (Component child in parentComponent.Children)
            {

                if (child.Name == targetName)
                {
                    return child as SoundSetItem;
                }

                SoundSetItem result = FindItem(targetName, child);
                if (null != result) { return result; }

            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateReadOnly()
        {
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateImage(SoundSetItem item)
        {
            ImageIndex = TabCtrl.IconType.Loupe;
        }

        /// <summary>
        /// タイトルを更新します。
        /// </summary>
        private void UpdateTitleInternal()
        {
            Title = String.Format("{0}({1})", TitleName, TitleNumber);
        }

        /// <summary>
        ///
        /// </summary>
        private void RedrawControls()
        {
            this.listCtrl.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusUndo(Command command)
        {
            if (this.listCtrl.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return (this.transactionPackStack.CanUndo() == true ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusRedo(Command command)
        {
            if (this.listCtrl.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return (this.transactionPackStack.CanRedo() == true ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExecuteUndo(Command command)
        {
            this.transactionPackStack.Undo();
            this.listCtrl.Invalidate();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExecuteRedo(Command command)
        {
            this.transactionPackStack.Redo();
            this.listCtrl.Invalidate();
            return true;
        }

        /// <summary>
        /// プレビュー再生ができる Componentを取得します。
        /// </summary>
        private Component[] GetPreviewPlayTargetComponents()
        {
            return TargetComponents
                .Where(c => c is Sound)
                .ToArray();
        }

        /// <summary>
        /// "プレビュー再生"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusPlay(Command command)
        {
            return GetPreviewPlayTargetComponents().Length == 1 ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "プレビュー再生"を実行します。
        /// </summary>
        private bool ExecutePlay(Command command)
        {
            Component[] components = GetPreviewPlayTargetComponents();

            if (components.Length == 1)
            {
                ExecutePreviewPlay(components[0]);
            }
            return true;
        }

        /// <summary>
        /// プレビュー再生を行います。
        /// </summary>
        private void ExecutePreviewPlay(Component component)
        {
            if (PreviewPlay != null)
            {
                PreviewPlay(component, new EventArgs());
            }
        }

        /// <summary>
        /// "すべて選択"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusSelectAll(Command command)
        {
            return CanSelectAll() ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "すべて選択"の実行
        /// </summary>
        private bool ExecuteSelectAll(Command command)
        {
            return SelectAll();
        }


        /// <summary>
        /// アイテムカラーを実行できるのか調べます。
        /// </summary>
        private bool CanSetItemColor()
        {
            return GetSetToItemColorTargetComponents().Length > 0 ? true : false;
        }

        /// <summary>
        /// アイテムカラーを実行します。
        /// </summary>
        private void SetItemColor(Component[] components, int value)
        {
            if (CanSetItemColor() == false)
            {
                return;
            }

            ExecuteOperation
                (components,
                  delegate (Component component)
                      {
                          Operation operation = new SetParameterOperation
                              (component.Parameters, ProjectParameterNames.ColorIndex, value);
                          return operation;
                      });
        }

        /// <summary>
        /// カラー設定ができる Componentを取得します。
        /// </summary>
        private Component[] GetSetToItemColorTargetComponents()
        {
            return TargetComponents
                .Where(c => !(c is GroupItemBase))
                .ToArray();
        }

        /// <summary>
        /// "アイテムカラー"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusSetItemColor(Command command)
        {
            if (this.listCtrl.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanSetItemColor() == true ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "アイテムカラー"を実行します。
        /// </summary>
        private bool ExecuteSetItemColor(Command command)
        {
            IQueryCommandParameter parameters = command.Parameters;
            if (parameters.ContainsParameter(CommandParameterNames.ColorIndex) == false)
            {
                return false;
            }

            try
            {
                int value = int.Parse
                    ((string)parameters.GetParameter(CommandParameterNames.ColorIndex));
                SetItemColor(GetSetToItemColorTargetComponents(), value);
                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// アイテムを有効/無効にできるのか調べます。
        /// </summary>
        private bool CanEnableItem(bool value)
        {
            // valueを引数として受け取っていますが、有効/無効で違いが無いので
            // 今は使っていません。
            return GetSetToEnableTargetComponents().Length > 0 ? true : false;
        }

        /// <summary>
        /// アイテムを有効/無効にします。
        /// </summary>
        private void EnableItem(Component[] components, bool value)
        {
            if (CanEnableItem(value) == false)
            {
                return;
            }

            ExecuteOperation
                (components,
                  delegate (Component component)
                      {
                          Operation operation = new SetParameterOperation
                              (component.Parameters, ProjectParameterNames.IsEnabled, value);
                          return operation;
                      });
        }

        /// <summary>
        /// 有効/無効にできる Componentを取得します。
        /// </summary>
        private Component[] GetSetToEnableTargetComponents()
        {
            return TargetComponents
                .ToArray();
        }

        /// <summary>
        /// "アイテムを有効にする"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusEnableItem(Command command)
        {
            if (this.listCtrl.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanEnableItem(true) == true ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "アイテムを有効にする"を実行します。
        /// </summary>
        private bool ExecuteEnableItem(Command command)
        {
            EnableItem(GetSetToEnableTargetComponents(), true);
            return true;
        }

        /// <summary>
        /// "アイテムを無効にする"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusDisableItem(Command command)
        {
            if (this.listCtrl.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanEnableItem(false) == true ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "アイテムを無効にする"を実行します。
        /// </summary>
        private bool ExecuteDisableItem(Command command)
        {
            EnableItem(GetSetToEnableTargetComponents(), false);
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected delegate Operation GetOperationHandler(Component component);

        /// <summary>
        ///
        /// </summary>
        protected void ExecuteOperation(Component[] components, GetOperationHandler handler)
        {
            //
            TransactionDictionary dictionary = new TransactionDictionary();
            foreach (Component component in components)
            {
                Document document = GetDocument(component);
                if (dictionary.ContainsKey(document) == false)
                {
                    dictionary.Add(document, new UserTransaction2(String.Empty));
                }
            }

            //
            foreach (Component component in components)
            {
                Document document = GetDocument(component);
                UserTransaction2 transaction = dictionary[document];

                Operation operation = handler(component);
                operation.Execute();

                transaction.Operations.Add(operation);
            }

            //
            foreach (KeyValuePair<Document, UserTransaction2> pair in dictionary)
            {
                Document document = pair.Key;
                UserTransaction2 transaction = pair.Value;

                transaction.Document = document;
                document.OperationHistory.AddTransaction(transaction);
            }

            //
            UserTransaction2[] transactions = dictionary.Values
                .ToArray();
            if (transactions.Length > 0)
            {
                PushTransactionPackStack(transactions);
            }
        }

        /// <summary>
        /// "貼り付け"が実行可能なのか調べる
        /// </summary>
        private new bool CanPaste()
        {
            //セルの貼り付けが可能なのか？
            if (ClipboardService.CanPasteCell(this.listCtrl.ItemsName,
                                               this.listCtrl.GetItemSelecteds()) == true)
            {
                return true;
            }

            //貼り付けが可能なのか？
            return ClipboardService.CanPaste(this.listAdapter.ItemsOwner);
        }

        /// <summary>
        /// "貼り付け"の実行
        /// </summary>
        private new bool Paste()
        {
            // 貼り付けが可能か？
            if (ClipboardService.CanPaste(this.listAdapter.ItemsOwner) == true)
            {
                // 検索結果リストでは貼り付けはできません。
                return true;
            }

            // セルへの貼り付けが可能か？
            if (ClipboardService.CanPasteCell(this.listCtrl.ItemsName,
                                               this.listCtrl.GetItemSelecteds()) == true)
            {
                //
                TransactionDictionary dictionary =
                    ClipboardService.PasteCellForFind(this.listCtrl.ItemsName,
                                                       this.listAdapter.Items.ToArray(),
                                                       this.listCtrl.GetItemSelecteds());
                if (dictionary == null)
                {
                    return false;
                }

                List<UserTransaction2> list = new List<UserTransaction2>();
                foreach (KeyValuePair<Document, UserTransaction2> pair in dictionary)
                {
                    UserTransaction2 transaction = pair.Value;
                    Document document = pair.Key;
                    transaction.Document = document;
                    document.OperationHistory.AddTransaction(transaction);

                    list.Add(transaction);
                }

                if (list.Count > 0)
                {
                    PushTransactionPackStack(list.ToArray());
                }

                return true;
            }

            return false;
        }

        /// <summary>
        /// "貼り付け"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusPaste(Command command)
        {
            if (this.listCtrl.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return (CanPaste() ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "貼り付け"を実行します。
        /// </summary>
        private bool ExecutePaste(Command command)
        {
            if (Paste() == false)
            {
                return false;
            }

            //ProjectService.SendUpdateSoundProjectItemBindingEvent();
            return true;
        }

        /// <summary>
        /// "セルのコピー"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusCopyCell(Command command)
        {
            bool result = ClipboardService.CanCopyCell
                (this.listCtrl.ItemsName,
                  this.listCtrl.GetItemSelecteds());

            return (result == true ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        /// <summary>
        /// "セルのコピー"の実行
        /// </summary>
        private bool ExecuteCopyCell(Command command)
        {
            ClipboardService.CopyCell(this.listCtrl.ItemsName,
                                       this.listCtrl.GetItemSelecteds());
            return true;
        }

        /// <summary>
        /// "項目を選択して貼り付け"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusPasteSpecial(Command command)
        {
            return CanPasteSpecial() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "項目を選択して貼り付け"の実行
        /// </summary>
        private bool ExecutePasteSpecial(Command command)
        {
            if (PasteSpecial() == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private class BatchEditItemInfo
        {
            private List<string> names = new List<string>();

            public ComponentListItem Item { get; set; }
            public List<string> Names
            {
                get
                {
                    return this.names;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private class BatchEditInfo : List<BatchEditItemInfo>
        {
            private SoundSet soundSet = null;
            private SoundSetDocument soundSetDocument = null;

            public BatchEditInfo(SoundSet soundSet, SoundSetDocument soundSetDocument)
            {
                this.soundSet = soundSet;
                this.soundSetDocument = soundSetDocument;
            }

            public SoundSet SoundSet
            {
                get
                {
                    return this.soundSet;
                }
            }
            public SoundSetDocument SoundSetDocument
            {
                get
                {
                    return this.soundSetDocument;
                }
            }
        }

        /// <summary>
        /// "一括編集"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusBatchEdit(Command command)
        {
            if (this.listCtrl.ReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanBatchEdit() ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "一括編集"が可能なのか調べます。
        /// </summary>
        private new bool CanBatchEdit()
        {
            ListItemSelectedDictionary selectedItems = this.listCtrl.GetItemSelecteds();
            bool can = false;

            foreach (ComponentListItem item in selectedItems.Keys)
            {
                if (item.Target is GroupItemBase)
                {
                    return false;
                }

                ListItemSelectedState state = selectedItems[item];
                if (state.SubSelected.Count > 0)
                {
                    can = true;
                }
            }
            return can;
        }

        /// <summary>
        ///
        /// </summary>
        private Operation[] BatchEdit2(EffectKind effectKind, string valueText, SoundSetDocument soundSetDocument, BatchEditInfo info)
        {
            List<Operation> list = new List<Operation>();

            foreach (BatchEditItemInfo itemInfo in info)
            {
                foreach (string name in itemInfo.Names)
                {
                    Operation[] operations = BatchEditUtility.SetEffect
                        (itemInfo.Item, name, effectKind, valueText,
                          (BatchEditUtility.ConvertTextHandler)
                          delegate (string n, string text)
                              {
                                  return ConvertText(n, text);
                              },
                          (BatchEditUtility.ConvertValueHandler)
                          delegate (string n, string text, IParameterValue value)
                              {
                                  return ConvertValue(n, text, value);
                              },
                          (BatchEditUtility.ValidateHandler)
                          delegate (Component component, string n, string text, object value)
                              {
                                  return ValidateValueForBatchEdit
                                  (component, n, text, value);
                              });

                    if (operations != null)
                    {
                        list.AddRange(operations);
                    }
                }
            }
            return list.ToArray();
        }

        /// <summary>
        /// "一括編集"の実行
        /// </summary>
        private bool ExecuteBatchEdit(Command command)
        {
            // 一括編集の対象の情報を作成します。
            Dictionary<SoundSet, BatchEditInfo> dictionary =
                new Dictionary<SoundSet, BatchEditInfo>();

            ListItemSelectedDictionary selectedItems = this.listCtrl.GetItemSelecteds();
            ListItemSelectedState state = null;

            foreach (ComponentListItem item in selectedItems.Keys)
            {
                BatchEditItemInfo itemInfo = new BatchEditItemInfo();
                itemInfo.Item = item;

                state = selectedItems[item];
                foreach (string name in state.SubSelected)
                {
                    itemInfo.Names.Add(name);
                }

                //
                SoundSet soundSet = GetSoundSet(item.Target);
                if (dictionary.ContainsKey(soundSet) == false)
                {
                    SoundSetDocument soundSetDocument = GetSoundSetDocument(soundSet);
                    dictionary.Add(soundSet, new BatchEditInfo(soundSet, soundSetDocument));
                }

                BatchEditInfo info = dictionary[soundSet];
                info.Add(itemInfo);
            }

            //
            BatchEditDialog dialog = null;

            dialog = new BatchEditDialog();
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            if (dialog.ValueText == String.Empty)
            {
                return false;
            }

            //
            {
                List<UserTransaction2> list = new List<UserTransaction2>();

                foreach (KeyValuePair<SoundSet, BatchEditInfo> pair in dictionary)
                {
                    UserTransaction2 transaction = new UserTransaction2(String.Empty);

                    SoundSetDocument soundSetDocument = GetSoundSetDocument(pair.Key);
                    Operation[] operations = BatchEdit2
                        (dialog.EffectKind, dialog.ValueText,
                          soundSetDocument, pair.Value);

                    if (operations.Length > 0)
                    {
                        foreach (Operation operation in operations)
                        {
                            transaction.Operations.Add(operation);
                        }

                        //transaction.SoundSetDocuments.Add( soundSetDocument);
                        transaction.Document = soundSetDocument;
                        soundSetDocument.OperationHistory.AddTransaction(transaction);

                        list.Add(transaction);
                    }
                }

                PushTransactionPackStack(list.ToArray());
            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private Component[] GetSoundSetItems()
        {
            List<Component> list = new List<Component>();

            foreach (SoundSetDocument soundSetDocument in ProjectService.SoundSetDocuments)
            {
                list.AddRange(GetSoundSetItems(soundSetDocument));
            }
            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private Component[] GetSoundSetItems(SoundSetDocument soundSetDocument)
        {
            List<Component> list = new List<Component>();
            SoundSet soundSet = soundSetDocument.SoundSet;

            foreach (Component component in soundSet.Children)
            {
                list.AddRange(component.Children);

                if (component is StreamSoundBase ||
                    component is WaveSoundSetPack ||
                    component is SequenceSoundSetPack)
                {
                    foreach (Component childComponent in component.Children)
                    {
                        list.AddRange(childComponent.Children);
                    }
                }
            }

            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private void ApplySoundSetConfiguration()
        {
        }

        private void ExtractSoundSetConfiguration()
        {
        }

        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (this.listMenuAdapter != null)
            {
                return;
            }

            this.listMenuAdapter = ToolStripAdapter.FromToolStrip
                (FormsApplication.Instance.CommandService, this.listMenu);
            this.listMenuAdapter.CommandTarget = FormsApplication.Instance;
            this.listMenuAdapter.CommandExecuted += OnCommandExecuted;

            //ApplySettings( SettingKinds.ColorComment);
            BuildListCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUIs()
        {
            BuildListCommandUI();
            MainWindow.BuildCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildListCommandUI()
        {
            if (null != this.listMenuAdapter)
            {
                this.listMenuAdapter.BuildUI();
            }
        }

        /// <summary>
        /// 検索用の引数を取得します。
        /// </summary>
        private FindArgs GetFindArgs()
        {
            FindArgs args = new FindArgs();

            switch (this.targetDocumentKind)
            {
                case FindArgs.TargetDocumentKinds.SoundSet:
                    args.Range = FindArgs.Ranges.SoundSet;
                    break;
                case FindArgs.TargetDocumentKinds.Bank:
                    args.Range = FindArgs.Ranges.Project;
                    break;
            }

            args.EnableLabel = this.chk_Name.Checked;
            args.EnableFilePath = this.chk_FilePath.Checked;
            args.EnableComment = this.chk_Comment.Checked;
            args.EnableItemReference = this.chk_ItemReference.Checked;
            args.EnableStartPosition = this.chk_StartPosition.Checked;
            args.CaseSensitive = this.chk_CaseSensitive.Checked;
            args.Text = Finder.ReplaceMetaStrings(this.cmb_Keyword.Text);
            args.TargetSoundSetNames = this.targetSoundSetNames;

            //
            if (this.chk_FindExactMatch.Checked == true)
            {
                args.Text = args.Text.Replace("*", ".*");
                args.Text = "^" + args.Text + "$";
            }
            else
            {
                args.Text = args.Text.Replace("*", "\\*");
            }

            return args;
        }

        /// <summary>
        /// 検索に使用した文字列を履歴として ComboBoxに記録します。
        /// </summary>
        private void AddTextToFindTextComboBox()
        {
            ComboBox comboBox = this.cmb_Keyword;

            string text = comboBox.Text;
            int index = -1;

            if (text == string.Empty)
            {
                return;
            }

            if ((index = comboBox.Items.IndexOf(text)) >= 0)
            {
                comboBox.Items.RemoveAt(index);
            }

            comboBox.Items.Insert(0, text);
            if (comboBox.Items.Count > 10)
            {
                comboBox.Items.RemoveAt(comboBox.Items.Count - 1);
            }

            comboBox.Text = text;
            comboBox.SelectionStart = 0;
            comboBox.SelectionLength = text.Length;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetReplaceArgumentDelegate(string column, ReplaceArgumentDelegate replaceArgumentDelegate)
        {
            if (this.listCtrl.ReplaceArgumentDelegates.ContainsKey(column) == true)
            {
                this.listCtrl.ReplaceArgumentDelegates.Remove(column);
            }

            this.listCtrl.ReplaceArgumentDelegates.Add(column, replaceArgumentDelegate);
        }

        public void Find()
        {
            Finder finder = new Finder();
            Component[] components = finder.Find
                (ProjectService,
                  this.targetDocumentKind,
                  GetFindArgs());

            SetResult(components);
            AddTextToFindTextComboBox();

            this.unsearched = false; // 一度でも検索したら false を設定する
        }

        private CommonListCtrl GetListCtrl()
        {
            CommonListCtrl listCtrl = ActiveListCtrl;
            if (listCtrl == null)
            {
                listCtrl = this.listCtrl;
            }

            return listCtrl;
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            ExtractSoundSetConfiguration();
        }

        private string GetListName(Component item, bool main)
        {
            string name = string.Empty;

            if (item is StreamSoundPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_StreamSound;
                }
                else
                {
                    name = ListTraits.ListName_StreamSoundTrack;
                }
            }
            else if (item is WaveSoundSetPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_WaveSoundSet;
                }
                else
                {
                    name = ListTraits.ListName_WaveSoundSetItem;
                }
            }
            else if (item is WaveSoundSetBase)
            {
                name = ListTraits.ListName_WaveSoundSetItem;
            }
            else if (item is SequenceSoundSetPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_SequenceSoundSet;
                }
                else
                {
                    name = ListTraits.ListName_SequenceSoundSetItem;
                }
            }
            else if (item is SequenceSoundSetBase)
            {
                name = ListTraits.ListName_SequenceSoundSetItem;
            }
            else if (item is SequenceSoundPack)
            {
                name = ListTraits.ListName_SequenceSound;
            }
            else if (item is SoundSetBankPack)
            {
                name = ListTraits.ListName_Bank;
            }
            else if (item is WaveArchivePack)
            {
                name = ListTraits.ListName_WaveArchive;
            }
            else if (item is PlayerPack)
            {
                name = ListTraits.ListName_Player;
            }
            else if (item is GroupPack)
            {
                if (main == true)
                {
                    name = ListTraits.ListName_Group;
                }
                else
                {
                    name = ListTraits.ListName_GroupItem;
                }
            }

            return name;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlay(object sender, EventArgs e)
        {
            if (PreviewPlay != null)
            {
                PreviewPlay(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewMute(object sender, EventArgs e)
        {
            if (PreviewMute != null)
            {
                PreviewMute(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewSoloPlay(object sender, EventArgs e)
        {
            if (PreviewSoloPlay != null)
            {
                PreviewSoloPlay(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOpening_soundListMenu(object sender, System.ComponentModel.CancelEventArgs e)
        {
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnFind(object sender, EventArgs e)
        {
            Find();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClick_SelectSoundSet(object sender, EventArgs e)
        {
            SoundSetSelectDialog dialog = new SoundSetSelectDialog
                (ProjectService, this.targetSoundSetNames);

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.targetSoundSetNames = dialog.SelectedSoundSets
                .Select(s => s.Name)
                .ToArray();

            UpdateUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnKeyDown_Keyword(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                Find();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCreateColumn(object sender, EventArgs e)
        {
            SetResult(this.resultComponents, true);
            ShowUsedHeaderColumns(this.listCtrl.HeaderSource, this.resultComponents);

            // リストの表示を更新させる為に、同じデータで再設定をしています。
            this.listCtrl.HeaderSource = this.listCtrl.HeaderSource;
        }

        ///--------------------------------
        /// <summary>
        /// リスト項目関連
        /// </summary>
        private void OnHeaderMenuItemAddPresetListColumnsClick(object sender, EventArgs e)
        {
            string listName = ListTraits.ListName_FindResult;
            ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
            headerAdapters[listName] = this.listCtrl.HeaderSource as ListHeaderAdapterBase;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.AddPresetListColumns(listName, applier);
            this.UpdatePresetListColumnsSplitButton();
        }

        private void OnHeaderMenuItemApplyPresetListColumnsClick(object sender, EventArgs e)
        {
            ToolStripMenuItem menuItem = sender as ToolStripMenuItem;
            string listName = ListTraits.ListName_FindResult;
            ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
            headerAdapters[listName] = this.listCtrl.HeaderSource as ListHeaderAdapterBase;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.ApplyPresetListColumnsWithoutAdd(applier, listName, menuItem.Text);
            this.listCtrl.UpdateInfos();
        }

        private void OnHeaderCtrlContextMenuOpening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            bool enabled = (this.listCtrl.HeaderSource as ListHeaderAdapterBase).Columns.Length > 1;
            _HeaderContextMenuItem_AddPresetListColumns.Enabled = enabled;
            _HeaderContextMenuItem_ApplyPresetListColumns.Enabled = enabled;

            if (enabled == true)
            {
                string listName = ListTraits.ListName_FindResult;

                ListHeaderHelper.CreateMeunApplyPresetListColumns(_HeaderContextMenuItem_ApplyPresetListColumns, listName, OnHeaderMenuItemApplyPresetListColumnsClick);
            }
        }

        private void UpdatePresetListColumnsSplitButton()
        {
            string listName = ListTraits.ListName_FindResult;
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            if (this.listCtrl.HeaderSource == null ||
                this.listCtrl.HeaderSource.Items == null ||
                this.listCtrl.HeaderSource.Items.Count <= 0)
            {
                MainWindow.PresetListColumnsPresetNamesOnSplitButton = null;
                MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = null;
            }
            else
            {
                MainWindow.PresetListColumnsPresetNamesOnSplitButton = preset.GetPresetNames(listName);
                MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = preset.GetCurrentPresetName(listName);
            }
        }
    }

    /// <summary>
    /// 編集不可能セルの描画を行うクラスです。
    /// </summary>
    public class NAListPartDrawer : ListPartDrawer
    {
        private static StringFormat Format = new StringFormat();

        /// <summary>
        ///
        /// </summary>
        static NAListPartDrawer()
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DrawInternal(ListDrawDescriptor desc)
        {
            ListColumnStyle style = desc.ColumnStyle;

            switch (style.Name)
            {
                case ProjectParameterNames.Sound.ActorPlayer:
                    Format.LineAlignment = StringAlignment.Far;
                    Format.Alignment = StringAlignment.Far;
                    break;

                default:
                    if (style.SubItemType == typeof(string))
                    {
                        Format.LineAlignment = StringAlignment.Far;
                        Format.Alignment = StringAlignment.Near;
                    }
                    else if (style.SubItemType == typeof(bool))
                    {
                        Format.LineAlignment = StringAlignment.Far;
                        Format.Alignment = StringAlignment.Center;
                    }
                    else
                    {
                        Format.LineAlignment = StringAlignment.Far;
                        Format.Alignment = StringAlignment.Far;
                    }
                    break;
            }

            var decorationEvaluator = ((desc.Item as CommonListItem).Adapter as CommonListAdapter).DecorationEvaluator;
            var isEnabled = decorationEvaluator.IsEnabled;
            try
            {
                decorationEvaluator.IsEnabled = false;
                desc.ContentDrawer.DrawString
                    (desc, "-", //NotAvailable.Text,
                      desc.Font, FindResultSettings.DisableTextBrush, desc.Bounds, Format);
            }
            finally
            {
                decorationEvaluator.IsEnabled = isEnabled;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class FindResultListPartCheckBoxDrawer : ListPartCheckBoxDrawer
    {
        /// <summary>
        ///
        /// </summary>
        protected override ButtonState GetButtonState(ListDrawDescriptor desc, ButtonState state)
        {
            if (desc.Item is FindResultListItem)
            {
                FindResultListItem item = desc.Item as FindResultListItem;

                if (desc.Name == ListTraits.ColumnName_IsResampleEnabled ||
                    desc.Name == ListTraits.ColumnName_IsDownMixEnabled)
                {
                    Component component = item.Target;

                    if (component is StreamSoundBase == true &&
                        component.Children.OfType<StreamSoundTrackBase>().Any(t => FileUtil.IsOpusFile(t.FilePath)) == true)
                    {
                        // opus ファイルの場合はダウンサンプルチェックボックスを無効にします。
                        return state | ButtonState.Inactive;
                    }
                }

                if (item.IsDisabledColorParameter(desc.Name) == true)
                {
                    return state | ButtonState.Inactive;
                }
            }

            return state;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class FindResultListContentDrawer : CommonListContentDrawer
    {
        /// <summary>
        ///
        /// </summary>
        public override void DrawString(ListDrawDescriptor desc, string text, Font font, Brush brush, Rectangle bounds, StringFormat format)
        {
            if (desc.Item is FindResultListItem)
            {
                FindResultListItem item = desc.Item as FindResultListItem;
                if (item.IsDisabledColorParameter(desc.Name) == true)
                {
                    brush = FindResultSettings.DisableTextBrush;
                }
            }
            base.DrawString(desc, text, font, brush, bounds, format);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class FindResultListItem : CommonListItem
    {

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FindResultListItem(Component component) : base(component)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsDisabledColorParameter(string name)
        {
            if (Target is StreamSoundTrackBase)
            {
                return FindResultSettings.ReferenceParentParameterNames.Contains(name);
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public override IParameterValue GetValue(string name)
        {
            if (Target is StreamSoundTrackBase)
            {
                if (FindResultSettings.ReplacedParameterNames.ContainsKey(name) == true)
                {
                    name = ReplaceParameterName(name);
                }
            }
            return base.GetValue(name);
        }

        /// <summary>
        ///
        /// </summary>
        public override IConstParameterValue GetConstValue(string name)
        {
            if (Target is StreamSoundTrackBase)
            {
                if (Target.Parent == null)
                {
                    return null;
                }

                if (FindResultSettings.ReferenceParentParameterNames.Contains(name) == true &&
                    FindResultSettings.ReplacedParameterNames.ContainsKey(name) == false)
                {

                    Component component = Target;
                    try
                    {
                        Target = Target.Parent;
                        name = ReplaceParameterName(name);
                        return base.GetConstValue(name);
                    }
                    finally
                    {
                        Target = component;
                    }
                }
            }

            name = ReplaceParameterName(name);
            return base.GetConstValue(name);
        }

        /// <summary>
        ///
        /// </summary>
        private string ReplaceParameterName(string name)
        {
            if (Target is StreamSoundTrackBase &&
                FindResultSettings.ReplacedParameterNames.ContainsKey(name) == true)
            {
                name = FindResultSettings.ReplacedParameterNames[name];
            }
            return name;
        }
    }
}
