﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;

    public partial class BookmarkPanel : CommonPanel, IQueryCommandParameter
    {
        private BookmarkAdapter bookmarkAdapter = new BookmarkAdapter();
        private ToolStripAdapter bookmarkTreeMenuAdapter;

        public BookmarkPanel()
        {
            InitializeComponent();
            InitializeCommandBindings();
            InitializeMenu();

            this.bookmarkCtrl.SelectChanged += OnSelectChanged;
            this.bookmarkCtrl.Hide();
        }

        public BookmarkService BookmarkService { get; set; }

        public void AttachBookmark(BookmarkService bookmarkService)
        {
            this.BookmarkService = bookmarkService;
            this.BookmarkService.BookmarkChanged += OnBookmarkChanged;
            this.BookmarkService.BookmarkAdded += OnBookmarkAdded;
            this.bookmarkAdapter.AttachBookmark(bookmarkService.BookmarkRoot);
            this.bookmarkCtrl.ItemsSource = this.bookmarkAdapter;
            this.bookmarkCtrl.Show();
        }

        public void DetachBookmark()
        {
            if (this.BookmarkService != null)
            {
                this.BookmarkService.Clear();
                this.BookmarkService.BookmarkChanged -= OnBookmarkChanged;
                this.BookmarkService.BookmarkAdded -= OnBookmarkAdded;
                this.BookmarkService = null;
            }
            this.bookmarkAdapter.AttachBookmark(null);
            this.bookmarkCtrl.ItemsSource = null;
            this.bookmarkCtrl.Hide();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component[] TargetComponents
        {
            get
            {
                Component[] components = null;

                if (null != this.bookmarkCtrl)
                {
                    components = this.bookmarkCtrl.GetComponentsBySelectedItem();
                }

                return components != null ? components : new Component[0];
            }
        }

        public void UpdateBookmark()
        {
            if (this.BookmarkService != null)
            {
                this.bookmarkAdapter.AttachBookmark(this.BookmarkService.BookmarkRoot);
                this.bookmarkAdapter.UpdateBookmark();
            }
        }

        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponents:
                    return true;
            }

            return false;
        }

        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponents:
                    return TargetComponents;
            }

            throw new KeyNotFoundException();
        }

        protected MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            this.BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        /// コマンドの関連付けを行います。
        /// </summary>
        private void InitializeCommandBindings()
        {
            BindCommand(new CommandBinding(this, EditCommands.Delete.ID, QueryBookmarkDelete, ExecuteBookmarkDelete));
            BindCommand(new CommandBinding(this, new CreateBookmarkFolderHandler(this)));
            BindCommand(new CommandBinding(this, new RenameBookmarkHandler(this)));
        }

        private void InitializeMenu()
        {
            if (ApplicationBase.Instance != null)
            {
                this.bookmarkTreeMenuAdapter = ToolStripAdapter.FromToolStrip
                    (ApplicationBase.Instance.CommandService, this.contextMenuStrip_bookmark);
                this.bookmarkTreeMenuAdapter.CommandTarget = ApplicationBase.Instance;
            }

            this.BuildCommandUI();
        }

        private void BuildCommandUI()
        {
            if (null != this.bookmarkTreeMenuAdapter)
            {
                this.bookmarkTreeMenuAdapter.BuildUI();
            }
        }

        private CommandStatus QueryBookmarkDelete(Command command)
        {
            Component[] components = this.bookmarkCtrl.GetComponentsBySelectedItem();

            if (components.Length == 1 &&
                this.BookmarkService != null &&
                this.BookmarkService.BookmarkRoot == components[0])
            {
                return CommandStatus.SupportedAndVisible;
            }
            else if (0 < components.Length)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteBookmarkDelete(Command command)
        {
            BookmarkComponent[] components = this.bookmarkCtrl.GetComponentsBySelectedItem();

            if (FormsApplication.Instance.UIService.ShowMessageBox
                (MessageResource.Message_ComfirmDeleteBookmark,
                  MessageResource.DialogTitle_DeleteBookmark,
                  AppMessageBoxButton.YesNo,
                  AppMessageBoxImage.Warning,
                  AppMessageBoxResult.Yes) != AppMessageBoxResult.Yes)
            {
                return false;
            }

            foreach (BookmarkComponent component in components)
            {
                this.BookmarkService.Delete(component);
            }

            return true;
        }

        private void OnSelectChanged(object sender, EventArgs e)
        {
            MainWindow.BuildCommandUI();
            MainWindow.ClearPresetListColumnsOnSplitButton();
            this.BuildCommandUI();
        }

        private void OnBookmarkChanged(object sender, EventArgs e)
        {
            this.bookmarkAdapter.UpdateBookmark();
        }

        private void OnBookmarkAdded(object sender, BookmarkAddedEventArgs e)
        {
            this.bookmarkAdapter.UpdateBookmark();
            BookmarkComponent bookmark = e.Bookmark;
            if (bookmark != null)
            {
                TreeItemSelectedDictionary selected = this.bookmarkCtrl.GetItemSelecteds();
                selected.Clear();
                ITreeItem item = this.bookmarkAdapter.FindBookmarkItem(bookmark);
                if (item != null)
                {
                    this.bookmarkCtrl.SetCaret(item);
                    selected.Selected(item, true);
                }
            }

            MainWindow.ActivateBookmarkToolPage();
        }
    }
}
