﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Data;
    using System.Drawing;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Resources;
    using Projects=NintendoWare.SoundFoundation.Projects;

    public partial class BookMarkListPanel : CommonPanel, IQueryCommandParameter
    {
        // コマンドバー管理
        private ToolStripAdapter bookMarkMenuAdapter;

        private BookMarkService bookMarkService = null;

        public BookMarkListPanel()
        {
            InitializeComponent();
            InitializeCommandBindings();
            InitializeMenu();
        }

        public BookMarkService BookMarkService
        {
            get
            {
                return this.bookMarkService;
            }
            set
            {
                if (this.bookMarkService != null)
                {
                    this.bookMarkService.ChangedBookMark -= this.OnChangedBookMark;
                }

                this.bookMarkService = value;

                if (this.bookMarkService != null)
                {
                    this.bookMarkService.ChangedBookMark += this.OnChangedBookMark;
                }

                this.UpdateBookMarkList();
            }
        }

        bool IQueryCommandParameter.ContainsParameter( string parameterName )
        {
            return false;
        }

        object IQueryCommandParameter.GetParameter( string parameterName )
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// コントロールが入力されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            BuildCommandUI();
        }

        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        /// コマンドの関連付けを行います。
        /// </summary>
        private void InitializeCommandBindings()
        {
            BindCommand(new CommandBinding(this, BookMarkCommands.Delete.ID,
                                           QueryStatusDelete, ExecuteDelete));
            BindCommand(new CommandBinding(this, BookMarkCommands.Rename.ID,
                                           QueryStatusRename, ExecuteRename));
        }

        ///--------------------------------
        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (null != this.bookMarkMenuAdapter) { return; }
            if (null == ApplicationBase.Instance) { return; }

            this.bookMarkMenuAdapter = ToolStripAdapter.FromToolStrip
                (ApplicationBase.Instance.CommandService, this.contextMenuStrip_BookMark);
            this.bookMarkMenuAdapter.CommandTarget = ApplicationBase.Instance;
            this.bookMarkMenuAdapter.CommandExecuted += OnBookMarkCommandExecuted;

            BuildCommandUI();
        }

        private void UpdateBookMarkList()
        {
            this.listView_BookMark.Items.Clear();
            this.listView_History.Items.Clear();

            if (this.bookMarkService != null)
            {
                foreach (string bookMarkName in this.bookMarkService.GetBookMarkNames())
                {
                    ListViewItem listViewItem = new ListViewItem(bookMarkName);
                    this.listView_BookMark.Items.Add(listViewItem);
                }
                foreach (string bookMarkName in this.bookMarkService.GetHistory())
                {
                    ListViewItem listViewItem = new ListViewItem(bookMarkName);
                    this.listView_History.Items.Add(listViewItem);
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUI()
        {
            if (null != this.bookMarkMenuAdapter)
            {
                this.bookMarkMenuAdapter.BuildUI();
            }
        }

        ///--------------------------------
        /// <summary>
        /// コマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnBookMarkCommandExecuted(object sender, CommandEventArgs e)
        {
            if (e.IsExecuted != false)
            {
                if (e.Command == BookMarkCommands.Rename)
                {

                }

                if (e.Command == BookMarkCommands.Delete)
                {

                }
            }

            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        /// "ブックマーク名の変更"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusRename(Command command)
        {
            if (listView_BookMark.SelectedItems.Count == 1)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "ブックマーク名の変更"の実行
        /// </summary>
        private bool ExecuteRename(Command command)
        {
            BookMarkNameSettingDialog dialog = new BookMarkNameSettingDialog();
            string bookMarkName = listView_BookMark.SelectedItems[0].Text;
            dialog.Text = MessageResource.Label_BookMarkRename;
            dialog.BookMarkName = bookMarkName;
        show:
            if (dialog.ShowDialog() == DialogResult.OK)
            {
                if (dialog.BookMarkName != bookMarkName)
                {
                    if (this.bookMarkService.Contains(dialog.BookMarkName) == true)
                    {
                        MessageBox.Show(MessageResource.Label_WarningSameNameBookMark);
                        goto show;
                    }
                    this.bookMarkService.Rename(bookMarkName, dialog.BookMarkName);
                    return true;
                }
            }

            return false;
        }


        ///--------------------------------
        /// <summary>
        /// "ブックマークの削除"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusDelete(Command command)
        {
            if (0 < listView_BookMark.SelectedItems.Count)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }
            return CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "ブックマークの削除"の実行
        /// </summary>
        private bool ExecuteDelete(Command command)
        {
            if (MessageBox.Show(MessageResource.Message_BookMarkDelete,
                                MessageResource.Label_BookMarkDelete,
                                MessageBoxButtons.OKCancel) == DialogResult.OK)
            {
                foreach (ListViewItem item in listView_BookMark.SelectedItems)
                {
                    string bookMarkName = item.Text;
                    this.bookMarkService.Remove(bookMarkName);
                }

                return true;
            }

            return false;
        }

        private void OnChangedBookMark(object sender, EventArgs e)
        {
            this.UpdateBookMarkList();
        }

        private void OnDoubleClick_BookMark(object sender, EventArgs e)
        {
            ListView listView = sender as ListView;

            if (listView != null && listView.SelectedItems.Count == 1)
            {
                ListViewItem item = listView.SelectedItems[0];
                string bookMarkName = item.Text;
                string soundSetName = this.bookMarkService.GetSoundSetName(bookMarkName);
                SoundSet soundSet = this.GetSoundSet(soundSetName);
                string nodeName = this.bookMarkService.GetNodeName(bookMarkName);
                Projects.Component component = this.GetComponent(soundSet, nodeName);

                if (soundSet != null && component != null)
                {
                    if (component is SoundSetBank)
                    {
                        SoundSetBank bank = component as SoundSetBank;

                        string filePath = new ReferenceFileFinder()
                            {
                                Text = bank.FilePath,
                            }.Validate(bank.FilePath);

                        if (null == filePath) { return; }

                        if (filePath != bank.FilePath)
                        {
                            SoundSetDocument soundSetDocument =
                                MainWindow.GetSoundSetDocument(soundSet);

                            Operation operation =
                                new SetParameterOperation(component.Parameters,
                                                          ProjectParameterNames.FilePath, filePath);
                            operation.Execute();
                            soundSetDocument.OperationHistory.AddOperation(operation);
                        }

                        MainWindow.OpenFile(bank);
                        this.BookMarkService.AddHistory(bookMarkName);
                    }
                    else if (component is SoundSetItem == true)
                    {
                        CommonTabPage page = MainWindow.AddPage(soundSet);
                        SoundSetPanel soundSetPanel = page.Panel as SoundSetPanel;

                        soundSetPanel.Show(component as SoundSetItem);
                        soundSetPanel.ActivatePanel(true, true);
                        this.BookMarkService.AddHistory(bookMarkName);
                    }

                    this.UpdateBookMarkList();
                }
            }
        }

        private SoundSet GetSoundSet(string soundSetName)
        {
            foreach (SoundSet soundSet in ApplicationBase.Instance.ProjectService.SoundSets)
            {
                if (soundSet.Name == soundSetName)
                {
                    return soundSet;
                }
            }

            return null;
        }

        private Projects.Component GetComponent(SoundSet soundSet, string nodeName)
        {
            if (soundSet != null)
            {
                foreach (SoundSetItem pack in soundSet.Children)
                {
                    if (pack.Name == nodeName)
                    {
                        return pack;
                    }

                    if (pack is WaveSoundSetPack || pack is SequenceSoundSetPack)
                    {
                        foreach (SoundSetItem itemSet in pack.Children)
                        {
                            if (itemSet.Name == nodeName)
                            {
                                return itemSet;
                            }

                            foreach (SoundSetItem item in itemSet.Children)
                            {
                                if (item.Name == nodeName)
                                {
                                    return item;
                                }
                            }
                        }
                    }
                    else
                    {
                        foreach (SoundSetItem item in pack.Children)
                        {
                            if (item.Name == nodeName)
                            {
                                return item;
                            }
                        }
                    }
                }
            }

            return null;
        }

        private void OnSelectedIndexChanged_BookMark(object sender, EventArgs e)
        {
            this.BuildCommandUI();
        }

        private void OnColumnClick_BookMark(object sender, ColumnClickEventArgs e)
        {
            switch (this.listView_BookMark.Sorting)
            {
            case SortOrder.None:
                this.listView_BookMark.Sorting = SortOrder.Ascending;
                break;

            case SortOrder.Ascending:
                this.listView_BookMark.Sorting = SortOrder.Descending;
                break;

            case SortOrder.Descending:
                this.listView_BookMark.Sorting = SortOrder.None;
                this.UpdateBookMarkList();
                break;
            }
        }
    }
}
