﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Utilities;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public partial class MultipleTrackSettingsDialog : Form
    {
        private SoundSetDocument      _SoundSetDocument = null;
        private StreamSound           _StreamSound = null;
        private CommonListAdapter       _ListAdapter = null;

        private ListHeaderAdapterDictionary  _headerAdapters;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MultipleTrackSettingsDialog(
                                           SoundSetDocument soundSetDocument,
                                           StreamSound streamSound,

                                           ListHeaderAdapterDictionary headerAdapters
                                           )
        {
            if ( null == headerAdapters ) { throw new ArgumentNullException( "headerAdapters" ); }

            InitializeComponent();

            _SoundSetDocument = soundSetDocument;
            _StreamSound = streamSound;
            _headerAdapters = headerAdapters;
            UpdatePartDrawerOptions( _ListCtrl );

            ListPartFilePathDrawer drawer =
                _ListCtrl.PartDrawers[ "FilePath" ] as ListPartFilePathDrawer;
            drawer.ShowFullPath = false;
        }

        /// <summary>
        ///
        /// </summary>
        public StreamSound Target
        {
            get
            {
                return this._StreamSound;
            }
        }

        public CanAppendHandler CanAppend { get; set; }
        public GetFilterHandler GetFilter { get; set; }

        /// <summary>
        ///
        /// </summary>
        private SoundProjectConfiguration ProjectConfiguration
        {
            get { return FormsApplication.Instance.ProjectConfiguration; }
        }

        /// <summary>
        ///
        /// </summary>
        private XmlSoundSetDocumentView SoundSetConfiguration
        {
            get
            {
                if ( null == ProjectConfiguration ) { return null; }
                if ( !ProjectConfiguration.DocumentViews.
                      ContainsKey( _SoundSetDocument.Resource.Key ) ) { return null; }

                return ProjectConfiguration.DocumentViews[ _SoundSetDocument.Resource.Key ]
                            as XmlSoundSetDocumentView;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnLoad( EventArgs e)
        {
            base.OnLoad( e );

            _ListAdapter = new CommonListAdapter();
            _ListAdapter.OperationHistory = _SoundSetDocument.OperationHistory;

            _ListCtrl.HeaderHeight = 28;
            _ListCtrl.SuspendTransaction = true;
            _ListCtrl.ItemsSource  = _ListAdapter;
            _ListCtrl.HeaderSource = _headerAdapters[ ListTraits.ListName_MultipleTrack ];
            _ListCtrl.SelectChanged += OnSelectChanged;
            _ListCtrl.EditEnded += OnEditEnded;

            _ListCtrl.HeaderCtrl.SortEnabled = false;

            _ListAdapter.AddItems( _StreamSound.Children );

            UpdateButtons();

            _SoundSetDocument.OperationHistory.BeginTransaction();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateButtons()
        {
            ComponentListItem[] items = null;
            bool                canAppend = false;
            bool                canDelete = false;

            if( _ListCtrl.ItemCount < ComponentConfiguration.Instance.StreamSoundTrackNumberMaximum ) {
                canAppend = true;
            }

            if( CanAppend != null &&
                CanAppend( this._StreamSound) == false )
            {
                canAppend = false;
            }

            //
            items = _ListCtrl.GetSelectedItems();
            if( _ListCtrl.ItemCount > 1 ) {
                canDelete = items.Length > 0 ? true : false;
            }

            if( _ListCtrl.ItemCount <= items.Length ) {
                canDelete = false;
            }

            button_Append.Enabled = canAppend;
            button_Delete.Enabled = canDelete;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSelectChanged( object sender, EventArgs e)
        {
            UpdateButtons();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnEditEnded( object sender, EventArgs e)
        {
            UpdateButtons();
        }

        /// <summary>
        /// 追加
        /// </summary>
        private void OnAppend(object sender, EventArgs e)
        {
            StreamSoundTrack  soundTrack = null;
            Operation operation = null;
            string filePath = null;
            string filter = WaveFileQuester.Filter;

            if( GetFilter != null )
            {
                filter = GetFilter( this._StreamSound, true);
            }

            if(( filePath = WaveFileQuester.QueryLoad( this, filter)) == null ) {
                return;
            }

            soundTrack = ApplicationBase.Instance.Traits.ComponentTraits.SoundSetComponentFactory.Create(typeof(StreamSoundTrack)) as StreamSoundTrack;
            soundTrack.FilePath = filePath;
            soundTrack.Volume   = 127;
            soundTrack.Pan      = 64;

            operation = new InsertComponentOperation
                ( _StreamSound, null, new[] { soundTrack });
            operation.Execute();
            _SoundSetDocument.OperationHistory.AddOperation( operation);

            UpdateButtons();
        }

        ///--------------------------------
        /// <summary>
        /// 削除
        /// </summary>
        private void OnDelete(object sender, EventArgs e)
        {
            Operation           operation = null;
            Component[]       components = null;

            components = _ListCtrl.GetSelectedItems()
                .Select( i => i.Target) .ToArray();

            operation = new RemoveComponentOperation( components);
            operation.Execute();
            _SoundSetDocument.OperationHistory.AddOperation( operation);

            UpdateButtons();
        }

        private void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            if(DialogResult == DialogResult.Cancel)
            {
                _SoundSetDocument.OperationHistory.CancelTransaction();
            }

            if(DialogResult == DialogResult.OK)
            {
                _SoundSetDocument.OperationHistory.EndTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdatePartDrawerOptions( ListCtrl listCtrl)
        {
            ListPartCommonRowHeaderDrawer       rowHeaderDrawer = null;
            bool                        showRowNumber = true;

            XmlOptionsApplicationGeneral        general = null;
            general = ApplicationBase.Instance.AppConfiguration.Options.Application.General;

            showRowNumber = general.ShowRowNumber;

            rowHeaderDrawer = listCtrl.PartDrawers[ "RowHeader" ] as
                ListPartCommonRowHeaderDrawer;
            rowHeaderDrawer.ShowRowNumber = showRowNumber;
        }

        /// <summary>
        ///
        /// </summary>
        public delegate bool CanAppendHandler( StreamSound streamSound);
        public delegate string GetFilterHandler( StreamSound streamSound, bool append);
    }
}
