﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    ///
    /// </summary>
    public partial class MultipleBankSettingsDialog : Form
    {
        private SoundProjectService _ProjectService = null;
        private SoundSetDocument _SoundSetDocument = null;
        private SequenceSound _SequenceSound = null;
        private MultipleBankParameterListAdapter _MultipleBankParameterListAdapter = null;

        private ListHeaderAdapterDictionary  _headerAdapters;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MultipleBankSettingsDialog(
                                          SoundProjectService projectService,
                                          SoundSetDocument soundSetDocument,
                                          SequenceSound sequenceSound,
                                          ListHeaderAdapterDictionary headerAdapters
                                          )
        {
            if ( null == headerAdapters ) { throw new ArgumentNullException( "headerAdapters" ); }

            InitializeComponent();

            _ProjectService = projectService;
            _SoundSetDocument = soundSetDocument;
            _SequenceSound = sequenceSound;
            _headerAdapters = headerAdapters;
            UpdatePartDrawerOptions(_ListCtrl);
        }

        private SoundProjectConfiguration ProjectConfiguration
        {
            get { return FormsApplication.Instance.ProjectConfiguration; }
        }

        private XmlSoundSetDocumentView SoundSetConfiguration
        {
            get
            {
                if ( null == ProjectConfiguration ) { return null; }
                if ( !ProjectConfiguration.DocumentViews.
                     ContainsKey( _SoundSetDocument.Resource.Key ) ) { return null; }

                return ProjectConfiguration.DocumentViews[ _SoundSetDocument.Resource.Key ]
                    as XmlSoundSetDocumentView;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnLoad( EventArgs e)
        {
            base.OnLoad( e );

            _MultipleBankParameterListAdapter = new MultipleBankParameterListAdapter();
            _MultipleBankParameterListAdapter.OperationHistory =
                _SoundSetDocument.OperationHistory;

            _ListCtrl.HeaderHeight = 28;
            _ListCtrl.SuspendTransaction = true;
            _ListCtrl.ItemsSource  = _MultipleBankParameterListAdapter;
            _ListCtrl.HeaderSource = _headerAdapters[ ListTraits.ListName_MultipleBank ];
            _ListCtrl.SelectChanged += OnSelectChanged;

            _ListCtrl.HeaderCtrl.SortEnabled = false;

            _MultipleBankParameterListAdapter.AddItems(_SequenceSound.SoundSetBankReferences);

            UpdateButtons();

            _SoundSetDocument.OperationHistory.BeginTransaction();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateButtons()
        {
            IListItem[] items = null;
            bool                canAppend = false;
            bool                canDelete = false;

            if( _ListCtrl.ItemCount < 4 ) {
                canAppend = true;
            }

            items = _ListCtrl.SelectedItems;
            if( _ListCtrl.ItemCount > 1 ) {
                canDelete = items.Length > 0 ? true : false;
            }

            if( _ListCtrl.ItemCount <= items.Length ) {
                canDelete = false;
            }

            button_Append.Enabled = canAppend;
            button_Delete.Enabled = canDelete;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectChanged( object sender, EventArgs e)
        {
            UpdateButtons();
        }

        ///--------------------------------
        /// <summary>
        /// 追加
        /// </summary>
        private void OnAppend(object sender, EventArgs e)
        {
            ComponentReference reference = null;
            Operation operation = null;
            string bankName = string.Empty;

            if (0 < _ProjectService.SoundSetBanks.Count())
            {
                bankName = _ProjectService.SoundSetBanks[0].Name;
            }

            reference = new ComponentReference();
            reference.TargetName = bankName;
            operation = new InsertParameterOperation(
                                                     _SequenceSound.SoundSetBankReferences,
                                                     null,
                                                     reference
                                                     );
            operation.Execute();
            _SoundSetDocument.OperationHistory.AddOperation( operation);

            UpdateButtons();
        }

        ///--------------------------------
        /// <summary>
        /// 削除
        /// </summary>
        private void OnDelete(object sender, EventArgs e)
        {
            Operation operation = null;
            ComponentReference[] references = null;

            references =
                _ListCtrl.GetSelectedItems().Select(i => i.Target)
                .Cast<ComponentReference>().ToArray();
            operation = new RemoveParameterOperation(
                                                     _SequenceSound.SoundSetBankReferences,
                                                     references
                                                     );
            operation.Execute();
            _SoundSetDocument.OperationHistory.AddOperation(operation);

            UpdateButtons();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnOK(object sender, EventArgs e)
        {
            DialogResult = DialogResult.OK;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCancel(object sender, EventArgs e)
        {
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            if(DialogResult == DialogResult.Cancel)
            {
                _SoundSetDocument.OperationHistory.CancelTransaction();
            }

            if(DialogResult == DialogResult.OK)
            {
                _SoundSetDocument.OperationHistory.EndTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdatePartDrawerOptions( ListCtrl listCtrl)
        {
            ListPartCommonRowHeaderDrawer rowHeaderDrawer = null;
            bool showRowNumber = true;

            XmlOptionsApplicationGeneral general = null;
            general = ApplicationBase.Instance.AppConfiguration.Options.Application.General;

            showRowNumber = general.ShowRowNumber;

            rowHeaderDrawer = listCtrl.PartDrawers[ "RowHeader" ] as
                ListPartCommonRowHeaderDrawer;
            rowHeaderDrawer.ShowRowNumber = showRowNumber;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ParameterListCtrl : ListCtrl
    {
        ///
        private static Brush    _DisableTextBrush = SystemBrushes.GrayText;

        ///
        private IParameterValue[]       _EncodingValues = new[] {
            new WaveEncodingParameterValue( WaveEncoding.Adpcm),
            new WaveEncodingParameterValue( WaveEncoding.Pcm16),
            new WaveEncodingParameterValue( WaveEncoding.Pcm8),
        };

        ///
        private IParameterValue[]       _ActorPlayerValues = new[] {
            new IntParameterValue( 0),
            new IntParameterValue( 1),
            new IntParameterValue( 2),
            new IntParameterValue( 3),
        };

        ///
        private IParameterValue[]       _3DDecayCurveValues = new[] {
            new DecayCurve3DParameterValue( DecayCurve3D.Log),
            new DecayCurve3DParameterValue( DecayCurve3D.Linear),
        };

        ///
        private IParameterValue[]       _PanModeValues = new[] {
            new PanModeParameterValue( PanMode.Dual),
            new PanModeParameterValue( PanMode.Balance),
        };

        ///
        private IParameterValue[]       _PanCurveValues = new[] {
            new PanCurveParameterValue( PanCurve.Sqrt),
            new PanCurveParameterValue( PanCurve.Sqrt_0Db),
            new PanCurveParameterValue( PanCurve.Sqrt_0Db_Clamp),
            new PanCurveParameterValue( PanCurve.SinCos),
            new PanCurveParameterValue( PanCurve.SinCos_0Db),
            new PanCurveParameterValue( PanCurve.SinCos_0Db_Clamp),
            new PanCurveParameterValue( PanCurve.Linear),
            new PanCurveParameterValue( PanCurve.Linear_0Db),
            new PanCurveParameterValue( PanCurve.Linear_0Db_Clamp),
        };

        ///
        private IParameterValue[]       _FileTypeValues = new[] {
            new SequenceSoundFileTypeParameterValue( SequenceSoundFileType.Text),
            new SequenceSoundFileTypeParameterValue( SequenceSoundFileType.Smf),
        };

        ///
        private IParameterValue[]       _GroupOutputTypeValues = new[] {
            new GroupOutputTypeParameterValue( GroupOutputType.Embedding ),
            new GroupOutputTypeParameterValue( GroupOutputType.Link ),
            new GroupOutputTypeParameterValue( GroupOutputType.UserManagement ),
            new GroupOutputTypeParameterValue( GroupOutputType.None ),
        };

        ///--------------------------------
        ///
        /// データ名、サブアイテム名、型、サブアイテムに渡す引数
        ///
        private ListColumnStyle[]       _ColumnStyles = new[] {

            new ListColumnStyle(ListTraits.ColumnName_Name, "Text", typeof(string)),
            new ListColumnStyle(ListTraits.ColumnName_Comment, "Text", typeof(string)),
            new ListColumnStyle(
                                ListTraits.ColumnName_GroupItemTargetSoundSet,
                                "Text",
                                typeof(string)),
            new ListColumnStyle(ListTraits.ColumnName_ItemLocation, "Text", typeof(string)),

            //new ListColumnStyle( ListTraits.ColumnName_FilePath, "FilePath", typeof(string)),

            new ListColumnStyle( ListTraits.ColumnName_Volume,           "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_ChannelPriority,  "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PlayerPriority,   "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_Pan,              "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_SurroundPan,      "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_AuxASend,         "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_AuxBSend,         "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_MainSend,         "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_DopplerFactor3D,  "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PlayerSoundLimit, "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PlayerHeapSize,   "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_ProgramNo,        "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeAttack,   "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeHold,     "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeDecay,    "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeSustain,  "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_EnvelopeRelease,  "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PitchSemitones,   "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_PitchCents,       "Integer", typeof(int)),
            new ListColumnStyle( ListTraits.ColumnName_KeyGroup,         "Integer", typeof(int)),

            new ListColumnStyle( ListTraits.ColumnName_UserParameter0,   "UserData", typeof(ulong)),

            new ListColumnStyle( ListTraits.ColumnName_Pitch,        "Float", typeof(float),
                                 new ListSubItemArgument( new ListDecimalDrawerArgument( 5),
                                                          null)),
            new ListColumnStyle( ListTraits.ColumnName_DecayRatio3D, "Float", typeof(float),
                                 new ListSubItemArgument( new ListDecimalDrawerArgument( 2),
                                                          null)),

            new ListColumnStyle(ListTraits.ColumnName_Enable3DVolume, "CheckBox", typeof(bool)),
            new ListColumnStyle(ListTraits.ColumnName_Enable3DPan, "CheckBox", typeof(bool)),
            new ListColumnStyle(
                                ListTraits.ColumnName_Enable3DSurroundPan,
                                "CheckBox",
                                typeof(bool)
                                ),
            new ListColumnStyle(ListTraits.ColumnName_Enable3DPriority, "CheckBox", typeof(bool)),
            new ListColumnStyle(ListTraits.ColumnName_Enable3DFilter, "CheckBox", typeof(bool)),
            new ListColumnStyle(
                                ListTraits.ColumnName_ReleasePriorityFixed,
                                "CheckBox",
                                typeof(bool)
                                ),
            new ListColumnStyle(ListTraits.ColumnName_PercussionMode, "CheckBox", typeof(bool)),

            new ListColumnStyle( ListTraits.ColumnName_DataSize,        "Size", typeof(int)),
        };

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ParameterListCtrl()
        {
            DropDownInplaceEditor.Argument dropDownArg = null;
            DropDownListInplaceEditor.Argument arg = null;

            foreach( ListColumnStyle style in _ColumnStyles ) {
                ColumnStyles.Add( style);
            }

            //
            ColumnStyles.Add( new ListColumnStyle
                              ( "RowHeader", "RowHeader", typeof(string), null, false, false));

            //
            OpenFilePathDialogInplaceEditor.Argument fileDialogArg = null;

            fileDialogArg = new OpenFilePathDialogInplaceEditor.Argument();
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_FilePath, "FilePath", typeof(string),
                                new ListSubItemArgument( null, fileDialogArg)));

            //
            ListTextDrawerArgument drawerArg = null;

            //
            drawerArg = new ListTextDrawerArgument();
            drawerArg.LineAlignment = StringAlignment.Far;
            drawerArg.Alignment     = StringAlignment.Far;

            ColumnStyles.Add( new ListColumnStyle
                              ( "Key", "RowHeader", typeof(string),
                                new ListSubItemArgument( drawerArg, null), false, false));

            //
            arg = new DropDownListInplaceEditor.Argument( OriginalKeyValues);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_OriginalKey, "DropDownList", typeof(string),
                                new ListSubItemArgument( null, arg)));

            arg = new DropDownListInplaceEditor.Argument( _EncodingValues);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_WaveEncoding, "DropDownList", typeof(string),
                                new ListSubItemArgument( null, arg)));

            //
            dropDownArg = new DropDownInplaceEditor.Argument( null);
            ColumnStyles.Add(new ListColumnStyle(
                                                 ListTraits.ColumnName_PlayerReference,
                                                 "AssistDropDown",
                                                 typeof(string),
                                                 new ListSubItemArgument( null, dropDownArg)
                                                 ));

            dropDownArg = new DropDownInplaceEditor.Argument( null);
            ColumnStyles.Add(new ListColumnStyle(
                                                 ListTraits.ColumnName_SoundSetBankReferences,
                                                 "AssistDropDown",
                                                 typeof(string),
                                                 new ListSubItemArgument( null, dropDownArg)
                                                 ));

            //
            arg = new DropDownListInplaceEditor.Argument( _GroupOutputTypeValues);
            ColumnStyles.Add(new ListColumnStyle(
                                                 ListTraits.ColumnName_GroupOutputType,
                                                 "DropDownList",
                                                 typeof(string),
                                                 new ListSubItemArgument( null, arg)
                                                 ));

            //
            dropDownArg = new DropDownInplaceEditor.Argument( null);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_GroupItemTargetName,
                                "GroupItemDropDownList", typeof(string),
                                new ListSubItemArgument( null, dropDownArg)));

            //
            drawerArg = new ListTextDrawerArgument();
            drawerArg.LineAlignment = StringAlignment.Far;
            drawerArg.Alignment     = StringAlignment.Far;

            arg = new DropDownListInplaceEditor.Argument( _ActorPlayerValues);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_ActorPlayer, "DropDownList", typeof(string),
                                new ListSubItemArgument( drawerArg, arg)));

            //
            arg = new DropDownListInplaceEditor.Argument( _3DDecayCurveValues);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_DecayCurve3D, "DropDownList", typeof(string),
                                new ListSubItemArgument( null, arg)));

            arg = new DropDownListInplaceEditor.Argument( _PanModeValues);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_PanMode, "DropDownList", typeof(string),
                                new ListSubItemArgument( null, arg)));

            arg = new DropDownListInplaceEditor.Argument( _PanCurveValues);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_PanCurve, "DropDownList", typeof(string),
                                new ListSubItemArgument( null, arg)));

            arg = new DropDownListInplaceEditor.Argument( _FileTypeValues);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_FileType, "DropDownList", typeof(string),
                                new ListSubItemArgument( null, arg)));

            //
            dropDownArg = new DropDownInplaceEditor.Argument( null);
            ColumnStyles.Add( new ListColumnStyle
                              ( ListTraits.ColumnName_StartPosition, "DropDown", typeof(string),
                                new ListSubItemArgument( null, dropDownArg)));

            BackgroundDrawer = new CommonListBackgroundDrawer();
            //             ContentDrawer = new CommonListContentDrawer();

            //             //Drawerの登録
            PartDrawers.Add( new ListPartCommonRowHeaderDrawer());

            //             PartDrawers.Remove( "Text");
            //             PartDrawers.Add( new ListPartTextDrawer());

            //SubItemの登録
            SubItems.Remove( "RowHeader");
            SubItems.Add( new ListSubItemCommonRowHeader());
            SubItems.Add( new ListSubItemUserData());
            SubItems.Add( new ListSubItemDropDownList
                          ( "GroupItemDropDownList", "GroupItemDropDownList"));

            //Editorの登録
            InplaceEditors.Add( "UserDataEditDialog", new UserDataEditDialogInplaceEditor());
            InplaceEditors.Add( "GroupItemDropDownList", new GroupItemDropDownList());
        }

        private static SoundProjectService ProjectService {
            get { return FormsApplication.Instance.ProjectService; }
        }

        private TextParameterValue[] OriginalKeyValues {
            get {
                return KeyNoteConverter.Notes
                    .Select( s => new TextParameterValue( s)) .ToArray();
            }
        }


        //         private static Brush    _DisableTextBrush = SystemBrushes.GrayText;

        //         public override DataObject TrackDragDrop()
        //         {
        //             DataObject dataObject = null;
        //             return dataObject;
        //         }

        ///--------------------------------
        /// <summary>
        /// エディタの引数の取得
        /// </summary>
        public override IInplaceEditorArgument GetEditorArgument(
                                                                 ListColumnStyle style,
                                                                 IListItem item
                                                                 )
        {
            IInplaceEditorArgument      arg = null;

            if(( arg = ReplaceArgument( style, item as ParameterListItem)) != null ) {
                return arg;
            }

            return base.GetEditorArgument( style, item);
        }

        ///--------------------------------
        /// <summary>
        /// 選択されているアイテムの取得
        /// </summary>
        public ParameterListItem[] GetSelectedItems()
        {
            return SelectedItems.Cast<ParameterListItem>().ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IInplaceEditorArgument ReplaceArgument(
                                                       ListColumnStyle style,
                                                       ParameterListItem item
                                                       )
        {
            switch( style.Name ) {
            case ProjectParameterNames.SequenceSound.SoundSetBankReferences:
                return new DropDownInplaceEditor.Argument
                    (GetComponentsLabel(ProjectService.SoundSetBanks));

            }

            return null;
        }

        private IParameterValue[] GetComponentsLabel(Component[] components)
        {
            return components
                .Cast<SoundSetItem>()
                .Select( i => new TextParameterValue( i.Name)).ToArray();
        }
    }

    public class MultipleBankParameterListAdapter : ParameterListAdapter
    {
        /// <summary>
        /// 値の設定
        /// </summary>
        public override void SetValue(IParameterProvider provider, string name, object value)
        {
            //Operation: ここで Operationにする

            SetParameterOperation operation = null;

            if (name == ProjectParameterNames.SequenceSound.SoundSetBankReferences)
            {
                operation = new SetParameterOperation(
                                                      provider.Parameters,
                                                      ProjectParameterNames.TargetName,
                                                      value
                                                      );
                operation.Execute();
                OnOperationExecuted(operation);
            }
            else
            {
                base.SetValue(provider, name, value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override ParameterListItem CreateListItem(IParameterProvider provider)
        {
            return (ParameterListItem)(new MultipleBankParameterListItem(provider));
        }
    }

    public class MultipleBankParameterListItem : ParameterListItem
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public MultipleBankParameterListItem(IParameterProvider provider)
            : base(provider)
        {
            this.Length = 20;
        }

        /// <summary>
        ///
        /// </summary>
        public override IParameterValue GetValue( string name)
        {
            ComponentReference reference = Target as ComponentReference;

            if (reference == null)
            {
                return null;
            }

            if (name == ProjectParameterNames.SequenceSound.SoundSetBankReferences)
            {
                return CreateParameterEditor(name, new TextParameterValue(reference.TargetName));
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override IConstParameterValue GetConstValue( string name)
        {
            ComponentReference reference = Target as ComponentReference;

            if (name == ProjectParameterNames.SequenceSound.SoundSetBankReferences)
            {
                return new TextParameterValue(reference.TargetName);
            }

            if (name == "RowHeader")
            {
                return new TextParameterValue("*");
            }

            return null;
        }
    }
}
