﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.IO;
using System.Windows.Forms;


namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    public partial class ReferenceFileFindDialog : Form
    {
        public enum Actions
        {
            SpecifyFilePath,
            SpecifyDirectory,
            Ignore,
        }

        private static Actions _Action = Actions.SpecifyFilePath;
        private static string _DefaultFilePath = String.Empty;
        private static string _DefaultDirectory = String.Empty;

        private static bool _SpecifyBounds = false;
        private static Rectangle _Bounds = new Rectangle();

        private string _FilePath = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ReferenceFileFindDialog(string message, string filePath)
        {
            _FilePath = filePath;

            InitializeComponent();

            //
            if (_SpecifyBounds == false)
            {
                _SpecifyBounds = true;
            }
            else
            {
                StartPosition = FormStartPosition.Manual;
                Bounds = _Bounds;
            }

            //
            //label_FilePath.Text = filePath;
            label_FilePath.Text = message;
            textBox_FilePath.Text = _DefaultFilePath;
            textBox_Directory.Text = _DefaultDirectory;

            //
            switch (_Action)
            {
                case Actions.SpecifyFilePath:
                    radioButton_SpecifyFilePath.Checked = true;
                    break;

                case Actions.SpecifyDirectory:
                    radioButton_SpecifyDirectory.Checked = true;
                    break;

                case Actions.Ignore:
                    radioButton_Ignore.Checked = true;
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Actions Action
        {
            get
            {
                if (radioButton_SpecifyFilePath.Checked != false)
                {
                    return Actions.SpecifyFilePath;
                }
                if (radioButton_SpecifyDirectory.Checked != false)
                {
                    return Actions.SpecifyDirectory;
                }
                return Actions.Ignore;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string FilePath
        {
            get { return textBox_FilePath.Text; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Directory
        {
            get { return textBox_Directory.Text; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DoSpecifyFindDirectory
        {
            get { return checkBox_NextDirectory.Checked; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DoIgnoreAll
        {
            get { return checkBox_IgnoreAll.Checked; }
            set { checkBox_IgnoreAll.Checked = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 指定パスからディレクトリを取得
        /// 確実に存在するパスを返す
        /// </summary>
        public string GetValidateDirectory(string directory)
        {
            if (directory == null) { return null; }
            if (directory == String.Empty) { return null; }

            if (System.IO.Directory.Exists(directory) == false)
            {
                return null;
            }

            return directory;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCheckedChangedSpecifyFilePath(object sender, EventArgs e)
        {
            _Action = Actions.SpecifyFilePath;

            panel_SpecifyFilePath.Enabled = true;
            panel_SpecifyDirectory.Enabled = false;
            panel_Ignore.Enabled = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCheckedChangedSpecifyDirectory(object sender, EventArgs e)
        {
            _Action = Actions.SpecifyDirectory;

            panel_SpecifyFilePath.Enabled = false;
            panel_SpecifyDirectory.Enabled = true;
            panel_Ignore.Enabled = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCheckedChangedIgnore(object sender, EventArgs e)
        {
            _Action = Actions.Ignore;

            panel_SpecifyFilePath.Enabled = false;
            panel_SpecifyDirectory.Enabled = false;
            panel_Ignore.Enabled = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickReferenceFilePath(object sender, EventArgs e)
        {
            OpenFileDialog dialog = new OpenFileDialog();
            string filter = "All Files (*.*)|*.*";
            string directory = null;

            try
            {
                if ((directory = GetValidateDirectory(_FilePath)) != null)
                {
                    dialog.InitialDirectory = directory;
                }

                dialog.Filter = filter;
                dialog.RestoreDirectory = true;
                if (dialog.ShowDialog() != DialogResult.OK)
                {
                    return;
                }

                textBox_FilePath.Text = dialog.FileName;
            }

            catch { }
            finally { dialog.Dispose(); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickReferenceDirectory(object sender, EventArgs e)
        {
            FolderBrowserDialog dialog = new FolderBrowserDialog();

            if (_FilePath != string.Empty)
            {
                dialog.SelectedPath = Path.GetDirectoryName(_FilePath);
            }

            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            textBox_Directory.Text = dialog.SelectedPath;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnOK(object sender, EventArgs e)
        {
            string filePath = textBox_FilePath.Text;
            string directory = textBox_Directory.Text;

            if (filePath != String.Empty)
            {
                _DefaultFilePath = filePath;
            }

            if (directory != String.Empty)
            {
                _DefaultDirectory = directory;
            }

            //
            _Bounds = Bounds;

            DialogResult = DialogResult.OK;
        }
    }
}
