﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.IO;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Windows;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public partial class OutputWaveFileDialog : Form
    {
        public delegate void OutputWaveFileDialogInitialize(OutputWaveFileDialog dialog);
        public static OutputWaveFileDialogInitialize InitializeInternal { get; set; }

        private string _initDestFolderPath = string.Empty;
        private Sound[] _sounds = new Sound[0];
        private string[] _destinationFilePaths;

        public OutputWaveFileDialog(Sound[] sounds, string initDestinationFolderPath)
            : this()
        {
            if (null == sounds) { throw new ArgumentNullException("sourceFilePaths"); }

            _sounds = sounds;
            _initDestFolderPath = (null != initDestinationFolderPath) ?
                                        initDestinationFolderPath : string.Empty;

            _destinationLabel.Text = IsMultipleOutput ?
                Resources.MessageResource.Label_OutputWaveFileDialog_DestinationFolder :
                Resources.MessageResource.Label_OutputWaveFileDialog_DestinationFile;

            if (IsMultipleOutput)
            {
                _destination.Text = _initDestFolderPath;
            }
            else
            {
                _destination.Text = Path.Combine(_initDestFolderPath, sounds[0].Name + ".wav");
            }
        }

        private OutputWaveFileDialog()
        {
            InitializeComponent();
            Initialize();
        }

        public string[] DestinationFilePaths
        {
            get { return _destinationFilePaths; }
        }

        public string DestinationFolderPath
        {
            get
            {
                if (0 == _destination.Text.Length) { return string.Empty; }
                if (IsMultipleOutput) { return _destination.Text; }
                return Path.GetDirectoryName(_destination.Text);
            }
        }

        public int MaxDuration
        {
            get { return Convert.ToInt32(_maxDuration.Value); }
            set { _maxDuration.Value = value; }
        }

        public bool IsMonaural
        {
            get { return this.radioButton_Monaural.Checked; }
            set
            {
                this.radioButton_Monaural.Checked = value;
                this.radioButton_Stereo.Checked = !value;
            }
        }

        public bool Is48kHz
        {
            get { return this.radioButton_48kHz.Checked; }
            set
            {
                this.radioButton_48kHz.Checked = value;
                this.radioButton_32kHz.Checked = !value;
            }
        }

        public bool IsOpenDestinationFolder
        {
            get { return _isOpenDestinationFolder.Checked; }
            set { _isOpenDestinationFolder.Checked = value; }
        }

        private bool IsMultipleOutput
        {
            get { return (1 < _sounds.Length); }
        }

        private bool IsValid
        {
            get
            {
                if (0 < _errorProvider.GetError(_destination).Length) { return false; }
                return true;
            }
        }

        public void ShowSamplingRatePanel()
        {
            this.panel_OutputSamplingRate.Enabled = true;
            this.panel_OutputSamplingRate.Visible = true;
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            if (!ValidateSounds())
            {
                DialogResult = DialogResult.Cancel;
                return;
            }

            ValidateChildren();
        }

        private void Initialize()
        {
            if (InitializeInternal != null)
            {
                InitializeInternal(this);
            }
        }

        private void UpdateButton()
        {
            _okButton.Enabled = IsValid;
        }

        private void SetError(Control control, string message)
        {
            if (null == control) { throw new ArgumentNullException("control"); }
            if (null == message) { throw new ArgumentNullException("message"); }

            _errorProvider.SetError(control, message);
            UpdateButton();
        }

        private bool ValidateSounds()
        {
            HashSet<string> names = new HashSet<string>();

            foreach (Sound sound in _sounds)
            {
                if (names.Contains(sound.Name))
                {
                    FormsApplication.Instance.UIService.ShowMessageBox(
                        FrameworkResources.MessageResource.Message_FailedToProcessDueToDuplicateItemName);
                    return false;
                }

                names.Add(sound.Name);
            }

            return true;
        }

        private bool ValidateDestinationFolder()
        {
            if (Directory.Exists(DestinationFolderPath)) { return true; }

            if (FormsApplication.Instance.UIService.ShowMessageBox(
                    FrameworkResources.MessageResource.Message_ConfirmCreateFolder,
                    null, AppMessageBoxButton.OKCancel, AppMessageBoxImage.Question,
                    AppMessageBoxResult.OK) != AppMessageBoxResult.OK) { return false; }

            return true;
        }

        private bool ConfirtmDestinationFilesOverwrite(string[] filePaths)
        {
            if (null == filePaths) { throw new ArgumentNullException("filePaths"); }

            bool showMessage = false;
            string message = FrameworkResources.MessageResource.Message_ConfirmOverwriteFiles;

            foreach (string filePath in filePaths)
            {
                if (!File.Exists(filePath)) { continue; }

                message += "\n" + filePath;
                showMessage = true;
            }

            if (!showMessage) { return true; }

            if (FormsApplication.Instance.UIService.ShowMessageBox(
                               message, null, AppMessageBoxButton.OKCancel,
                               AppMessageBoxImage.Question, AppMessageBoxResult.OK)
                 != AppMessageBoxResult.OK)
            {
                return false;
            }

            return true;
        }

        private void ValidateDestination()
        {
            if (0 == _destination.Text.Length)
            {
                SetError(_destination, FrameworkResources.MessageResource.Message_InputPath);
                return;
            }

            ValidationResult result = FilePathValidator.ValidateFilePath(_destination.Text);

            if (!result.IsValid)
            {
                SetError(_destination, result.ToString());
                return;
            }

            SetError(_destination, string.Empty);
        }

        private void OnReferClick(object sender, EventArgs e)
        {
            string initPath = (0 < _destination.Text.Length) ?
                                     _destination.Text : _initDestFolderPath;

            if (IsMultipleOutput)
            {
                FolderBrowserDialog dialog = new FolderBrowserDialog()
                {
                    SelectedPath = initPath,
                };

                if (dialog.ShowDialog() != DialogResult.OK)
                {
                    return;
                }

                _destination.Text = dialog.SelectedPath;
            }
            else
            {
                FileDialog dialog = new SaveFileDialog()
                {
                    InitialDirectory = Path.GetDirectoryName(initPath),
                    FileName = Path.GetFileName(initPath),
                    Filter = "Wave Files (*.wav)|*.wav|All Files (*.*)|*.*",
                    OverwritePrompt = false,
                };

                if (dialog.ShowDialog() != DialogResult.OK)
                {
                    return;
                }

                _destination.Text = dialog.FileName;
            }

            ValidateChildren();
        }

        private void OnOKClick(object sender, EventArgs e)
        {
            if (!ValidateDestinationFolder())
            {
                return;
            }

            if (IsMultipleOutput)
            {
                List<string> paths = new List<string>();

                foreach (Sound sound in _sounds)
                {
                    paths.Add(Path.Combine(_destination.Text, sound.Name + ".wav"));
                }

                if (!ConfirtmDestinationFilesOverwrite(paths.ToArray()))
                {
                    return;
                }

                _destinationFilePaths = paths.ToArray();
            }
            else
            {
                if (!ConfirtmDestinationFilesOverwrite(new string[] { _destination.Text }))
                {
                    return;
                }

                _destinationFilePaths = new string[] { _destination.Text };
            }

            DialogResult = DialogResult.OK;
        }

        private void OnDestinationValidating(object sender, CancelEventArgs e)
        {
            ValidateDestination();
        }

        private void OnDestinationTextChanged(object sender, EventArgs e)
        {
            ValidateDestination();
        }
    }
}
