﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public partial class ImportDLSDialog : Form
    {
        private Process _Process = null;
        private Thread _Thread = null;

        private string _FilePath = null;
        private string _OutputDirectory = null;

        //private List<string>    logList = new List<string>();
        private delegate void SetMessageDelegate(string msg);

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ImportDLSDialog()
        {
            InitializeComponent();

            button_Cancel.Bounds = button_Close.Bounds;
            button_Close.Enabled = false;
            button_Close.Visible = false;

            AcceptButton = button_Cancel;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string FilePath
        {
            set { _FilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string OutputDirectory
        {
            set { _OutputDirectory = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            Assembly assembly = Assembly.GetExecutingAssembly();
            string exeFilePath = null;
            string command = null;

            exeFilePath = Path.Combine(Path.GetDirectoryName(assembly.Location),
                                        "dls2rbnk.exe");

            command = //exeFilePath +
                " \"" + _FilePath + "\"" +
                " -o " + "\"" + _OutputDirectory + "\"";

            //ProcessStartInfoの作成
            ProcessStartInfo startInfo = new ProcessStartInfo(exeFilePath);
            startInfo.Arguments = command; //strArg.ToString();
            startInfo.CreateNoWindow = true;
            startInfo.UseShellExecute = false;
            startInfo.RedirectStandardOutput = true;
            startInfo.RedirectStandardError = true;
            startInfo.WorkingDirectory = _OutputDirectory;

            //プロセスの起動
            //Invoke( new SetMessageDelegate( SetMessage ), new object[]{ "Converting ..." } );
            try
            {
                _Process = Process.Start(startInfo);
            }

            catch
            {
                Invoke(new SetMessageDelegate(AppendLine),
                        new object[] { "dls2rbnk.exe was not found" });
                return;
            }

            _Thread = new Thread(new ThreadStart(ThreadProc));
            _Thread.Start();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnClosed(EventArgs e)
        {
            if (_Thread != null)
            {
                _Thread.Join();
            }

            base.OnClosed(e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ThreadProc()
        {
            string message = null;
            int exitCode = 0;

            //
            while ((message = _Process.StandardOutput.ReadLine()) != null)
            {
                BeginInvoke(new SetMessageDelegate(AppendLine),
                             new object[] { message });
            }

            //
            while ((message = _Process.StandardError.ReadLine()) != null)
            {
                BeginInvoke(new SetMessageDelegate(AppendLine),
                             new object[] { message });
            }

            //
            exitCode = _Process.ExitCode;
            if (exitCode == 0)
            {
                BeginInvoke(new MethodInvoker(ConvertComplete));
            }
            else
            {
                BeginInvoke(new MethodInvoker(ConvertError));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AppendLine(string msg)
        {
            textBox_Text.AppendLine(msg);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ConvertComplete()
        {
            button_Cancel.Enabled = false;
            button_Cancel.Visible = false;
            button_Close.Enabled = true;
            button_Close.Visible = true;

            //DialogResult = DialogResult.OK;
            AcceptButton = button_Close;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ConvertError()
        {
            button_Cancel.Enabled = true;
            button_Cancel.Visible = true;
            button_Close.Enabled = false;
            button_Close.Visible = false;

            //button_Close.DialogResult = DialogResult.Cancel;
            AcceptButton = button_Cancel;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickCancel(object sender, EventArgs e)
        {
            DialogResult = DialogResult.Cancel;

            try
            {
                if (_Process != null)
                {
                    _Process.Kill();
                }
            }

            catch (InvalidOperationException)
            {
                //プロセスが既に終了しています。
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickClose(object sender, EventArgs e)
        {
            DialogResult = DialogResult.OK;
        }

        private void ToolStripMenuItem_Copy_Click(object sender, EventArgs e)
        {
            Clipboard.SetDataObject(textBox_Text.Text, true);
        }

    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// LogViewControl の概要の説明です。
    /// </summary>
    public class LogViewControl : UserControl
    {
        private class DoubleBufferPanel : Control
        {
            public DoubleBufferPanel()
            {
                this.SetStyle(
                              ControlStyles.AllPaintingInWmPaint
                              | ControlStyles.DoubleBuffer
                              | ControlStyles.UserPaint,
                              true
                              );
            }
        }

        private class LineInfo
        {
            public LineInfo(string text)
            {
                this.Text = text;
                IsSizeValid = false;
            }

            public string Text { get; set; }
            public SizeF Size { get; set; }
            public bool IsSizeValid { get; set; }
        }

        private VScrollBar scrollBar;
        private Control panel;
        private LinkedList<LineInfo> lines = new LinkedList<LineInfo>();
        private int maxLine = 5000;
        private float totalHeight = 0.0f;
        private bool invalidSizeFlag = false;
        private bool autoScrollFlag = false;

        public LogViewControl()
        {
            panel = new DoubleBufferPanel();
            panel.Parent = this;
            panel.Dock = DockStyle.Fill;
            //        panel.BorderStyle = BorderStyle.Fixed3D;
            panel.Paint += new PaintEventHandler(PanelOnPaint);
            panel.EnabledChanged += new EventHandler(PanelOnEnabledChanged);
            panel.SizeChanged += new EventHandler(PanelOnSizeChanged);
            panel.GotFocus += new EventHandler(PanelOnFocusChanged);
            panel.LostFocus += new EventHandler(PanelOnFocusChanged);
            panel.Enabled = true;

            scrollBar = new VScrollBar();
            scrollBar.Parent = this;
            scrollBar.Dock = DockStyle.Right;
            scrollBar.Scroll += new ScrollEventHandler(ScrollBarOnScroll);
        }

        public int MaxLine
        {
            get { return maxLine; }
            set
            {
                maxLine = value;
                while (lines.Count > maxLine)
                {
                    lines.RemoveFirst();
                }
                panel.Invalidate();
            }
        }

        public override string Text
        {
            get
            {
                StringBuilder sb = new StringBuilder();
                foreach (LineInfo line in lines)
                {
                    sb.AppendLine(line.Text);
                }
                return sb.ToString();
            }
            set
            {
                lines.Clear();

                StringReader strReader = new StringReader(value);
                while (true)
                {
                    string text = strReader.ReadLine();
                    if (text == null) break;

                    lines.AddLast(new LineInfo(text));
                }
                panel.Invalidate();
            }
        }

        public void AppendLine(string text)
        {
            while (lines.Count > maxLine)
            {
                lines.RemoveFirst();
            }

            lines.AddLast(new LineInfo(text));

            autoScrollFlag = true;
            panel.Invalidate();
        }

        private void PanelOnEnabledChanged(object sender, EventArgs e)
        {
            if (panel.Enabled)
            {
                panel.ForeColor = SystemColors.WindowText;
                panel.BackColor = SystemColors.Window;
            }
            else
            {
                panel.ForeColor = SystemColors.GrayText;
                panel.BackColor = SystemColors.Control;
            }
        }

        private void PanelOnSizeChanged(object sender, EventArgs e)
        {
            invalidSizeFlag = true;
            panel.Invalidate();
        }

        private void PanelOnFocusChanged(object sender, EventArgs e)
        {
            panel.Invalidate();
        }

        private void ScrollBarOnScroll(object sender, ScrollEventArgs e)
        {
            autoScrollFlag = false;
            panel.Invalidate();
        }

        private void UpdateScrollBar()
        {
            int clientHeight = panel.ClientSize.Height;

            scrollBar.Maximum = (int)totalHeight;
            scrollBar.LargeChange = panel.ClientSize.Height;
            scrollBar.SmallChange = Font.Height;
            scrollBar.Enabled = (scrollBar.Maximum > scrollBar.LargeChange);

            if (autoScrollFlag)
            {
                scrollBar.Value = scrollBar.Maximum + 1 - scrollBar.LargeChange;
            }
        }

        private void PanelOnPaint(object sender, PaintEventArgs ea)
        {
            Graphics gfx = ea.Graphics;

            // テキストサイズの更新
            {
                float y = 0.0f;
                foreach (LineInfo line in lines)
                {
                    if (invalidSizeFlag || !line.IsSizeValid)
                    {
                        line.Size = gfx.MeasureString(line.Text, Font, panel.ClientSize.Width);
                        line.IsSizeValid = true;
                    }
                    y += line.Size.Height;
                }

                invalidSizeFlag = false;
                totalHeight = y;
            }

            UpdateScrollBar();

            {
                Brush brush = new SolidBrush(ForeColor);

                int yOffset;
                if (scrollBar.Maximum > 0)
                {
                    yOffset = (int)(totalHeight * scrollBar.Value / scrollBar.Maximum);
                }
                else
                {
                    yOffset = 0;
                }

                float y = 0.0f;
                foreach (LineInfo line in lines)
                {
                    SizeF size = line.Size;

                    if (y + size.Height > yOffset && y <= yOffset + panel.ClientSize.Height)
                    {
                        RectangleF rect = new RectangleF(0, y - yOffset, panel.ClientSize.Width, panel.ClientSize.Height);
                        gfx.DrawString(line.Text, Font, brush, rect);
                    }

                    y += size.Height;
                }
            }

            if (Focused)
            {
                //            ControlPaint.DrawFocusRectangle( gfx, panel.ClientRectangle );
            }
            ControlPaint.DrawBorder3D(gfx, panel.ClientRectangle, Border3DStyle.Sunken);
        }
    }
}
