﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    /// <summary>
    /// ビュータイトルバークラス。
    /// </summary>
    [ToolboxItem(true)]
    public sealed class ViewTitleBar : Control
    {
        // テキスト配置方向
        private HorizontalAlignment _textAlign = HorizontalAlignment.Left;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ViewTitleBar()
        {
            base.SetStyle(ControlStyles.Opaque, true);
            base.SetStyle(ControlStyles.ResizeRedraw, true);
            base.SetStyle(ControlStyles.Selectable, false);
            base.DoubleBuffered = true;
            base.TabStop = false;
        }

        /// <summary>
        /// テキスト配置方向。
        /// </summary>
        [DefaultValue(HorizontalAlignment.Left)]
        [Description("テキストの配置方向を示します。")]
        public HorizontalAlignment TextAlign
        {
            get { return _textAlign; }
            set
            {
                _textAlign = value;
                Invalidate();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string Text
        {
            get { return base.Text; }
            set
            {
                base.Text = value;
                Invalidate();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(120, 18); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            Rectangle rcClient = this.ClientRectangle;

            // 背景
            using (LinearGradientBrush brush = new LinearGradientBrush(rcClient, ControlPaint.Light(BackColor), BackColor, LinearGradientMode.Vertical))
            {
                e.Graphics.FillRectangle(brush, rcClient);
            }

            // 枠
            e.Graphics.DrawRectangle(SystemPens.Control, rcClient.X, rcClient.Y, rcClient.Width - 1, rcClient.Height - 1);
            {
                Rectangle rect = rcClient;
                rect.Width -= 1;
                rect.Height -= 1;
                e.Graphics.DrawLine(SystemPens.ControlDark, rect.Left + 1, rect.Top, rect.Right - 1, rect.Top);
                e.Graphics.DrawLine(SystemPens.ControlDark, rect.Right, rect.Top + 1, rect.Right, rect.Bottom - 1);
                e.Graphics.DrawLine(SystemPens.ControlDark, rect.Left, rect.Top + 1, rect.Left, rect.Bottom - 1);
                e.Graphics.DrawLine(SystemPens.ControlDark, rect.Left + 1, rect.Bottom, rect.Right - 1, rect.Bottom);
            }

            // テキスト
            using (StringFormat sf = new StringFormat())
            {
                sf.LineAlignment = StringAlignment.Center;
                sf.Trimming = StringTrimming.EllipsisCharacter;
                sf.FormatFlags |= StringFormatFlags.NoWrap;

                switch (_textAlign)
                {
                    case HorizontalAlignment.Left:
                        sf.Alignment = StringAlignment.Near;
                        break;
                    case HorizontalAlignment.Center:
                        sf.Alignment = StringAlignment.Center;
                        break;
                    case HorizontalAlignment.Right:
                        sf.Alignment = StringAlignment.Far;
                        break;
                    default:
                        break;
                }

                Rectangle rcText = rcClient;
                rcText.Inflate(-1, -1);
                rcText.X += 2;
                rcText.Width -= 2;
                e.Graphics.DrawString(this.Text, this.Font, SystemBrushes.WindowText, rcText, sf);
            }
        }
    }
}
