﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Parameters;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundMaker.Framework.Resources;
using NintendoWare.SoundMaker.Framework.Utilities;
using RegularExpressions = System.Text.RegularExpressions;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public partial class ProjectCtrl : TreeCtrl
    {
        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ProjectCtrl()
        {
            InitializeComponent();

            MultiSelect = false;
            SelectChanged += OnSelectChanged;

            Images.Add("IconProject", ImageResource.BitmapIconSoundProject);
            Images.Add("IconSoundSet", ImageResource.BitmapIconSoundSet);
            Images.Add("IconStreamSoundPack", ImageResource.BitmapIconStreamSoundFolder);
            Images.Add("IconWaveSoundSetPack", ImageResource.BitmapIconWaveSoundSetFolder);
            Images.Add("IconWaveSoundSet", ImageResource.BitmapIconWaveSoundSet);
            Images.Add("IconSequenceSoundPack", ImageResource.BitmapIconSequenceSoundFolder);
            Images.Add("IconSequenceSoundSetPack", ImageResource.BitmapIconSequenceSoundSetFolder);
            Images.Add("IconSequenceSoundSet", ImageResource.BitmapIconSequenceSoundSet);
            Images.Add("IconBankPack", ImageResource.BitmapIconBankFolder);
            Images.Add("IconBank", ImageResource.BitmapIconBank);
            Images.Add("IconWaveArchivePack", ImageResource.BitmapIconWaveArchiveFolder);
            Images.Add("IconGroupPack", ImageResource.BitmapIconGroupFolder);
            Images.Add("IconPlayerPack", ImageResource.BitmapIconPlayerFolder);
            Images.Add("IconAddonSoundSet", ImageResource.BitmapIconAddonSoundSet);
            Images.Add("IconAddonGroupPack", ImageResource.BitmapIconAddonGroupFolder);
            Images.Add("IconAddonPlayerPack", ImageResource.BitmapIconAddonPlayerFolder);
            Images.Add("IconNoExportSoundSet", ImageResource.BitmapIconNoExportSoundSet);
            Images.Add("IconFolderOpen", ImageResource.BitmapIconFolderOpen);
            Images.Add("IconFolderClose", ImageResource.BitmapIconFolderClose);

            ProjectPartIconDrawer iconDrawer = new ProjectPartIconDrawer();
            PartDrawers.Remove(iconDrawer.Name);
            PartDrawers.Add(iconDrawer);

            ProjectPartCaptionDrawer captionDrawer = new ProjectPartCaptionDrawer();
            PartDrawers.Remove(captionDrawer.Name);
            PartDrawers.Add(captionDrawer);
        }

        private SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override TreeDoubleClickedAction OnDoubleClicked(ITreeItem origItem, string name)
        {
            SoundSet soundSet = null;
            CommonTabPage page = null;
            SoundSetPanel soundSetPanel = null;
            ComponentTreeItem item = origItem as ComponentTreeItem;
            SoundSetItem soundSetItem = item.Target as SoundSetItem;
            string filePath = null;

            if (item.Target is SoundSet)
            {
                return TreeDoubleClickedAction.ToggleExpand;
            }

            soundSet = GetSoundSet(item);

            //
            if (item.Target is SoundSetBankBase)
            {
                SoundSetBankBase bank = item.Target as SoundSetBankBase;

                filePath = new ReferenceFileFinder()
                {
                    Text = bank.FilePath,
                }.Validate(bank.FilePath);

                if (null == filePath) { return TreeDoubleClickedAction.None; }

                if (filePath != bank.FilePath)
                {
                    SoundSetDocument soundSetDocument = MainWindow.GetSoundSetDocument(soundSet);

                    Operation operation = new SetParameterOperation
                        (item.Target.Parameters,
                          ProjectParameterNames.FilePath, filePath);
                    operation.Execute();
                    soundSetDocument.OperationHistory.AddOperation(operation);
                }

                MainWindow.OpenFile(bank);
                return TreeDoubleClickedAction.None;
            }

            //
            if (soundSet != null)
            {
                page = MainWindow.AddPage(soundSet);
                soundSetPanel = page.Panel as SoundSetPanel;

                soundSetPanel.Show(soundSetItem);
                soundSetPanel.ActivatePanel(true, true);
            }

            return TreeDoubleClickedAction.None;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetCaret(Component component)
        {
            TreeItemSelectedDictionary selectedItems = GetItemSelecteds();
            ComponentTreeItem item = null;

            selectedItems.Clear();
            if ((item = FindItems(component)) != null)
            {
                SetCaret(item);

                while ((item = item.Parent) != null)
                {
                    item.Expanded = true;
                }

                UpdateInfos();
            }

            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        /// 選択されているアイテムが参照している NwComponentを取得
        /// </summary>
        public virtual Component[] GetComponentsBySelectedItem()
        {
            return SelectedItems.Cast<ComponentTreeItem>().Select(i => i.Target).ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanMoveSelectedItemToChild(ITreeItem targetItem, ITreeItem[] selectedItems)
        {
            var targetComponentItem = targetItem as ComponentTreeItem;
            if (IsContainerItem(targetComponentItem) == false)
            {
                return false;
            }

            var movable = selectedItems
                .OfType<ComponentTreeItem>()
                .All(i => IsMovableItem(i) == true);
            if (movable == false)
            {
                return false;
            }

            var sameNames = GetSameNames(targetComponentItem.Children, selectedItems);
            if (sameNames.Count() > 0)
            {
                ShowSameNameErrorMessage(sameNames);
                return false;
            }

            return true;
        }

        protected override bool CanMoveSelectedItemToPreviousSibling(ITreeItem targetItem, ITreeItem[] selectedItems)
        {
            if (base.CanMoveSelectedItemToPreviousSibling(targetItem, selectedItems) == false)
            {
                return false;
            }
            return CanMove(targetItem, selectedItems);
        }

        protected override bool CanMoveSelectedItemToNextSibling(ITreeItem targetItem, ITreeItem[] selectedItems)
        {
            if (base.CanMoveSelectedItemToNextSibling(targetItem, selectedItems) == false)
            {
                return false;
            }
            return CanMove(targetItem, selectedItems);
        }

        protected bool CanMove(ITreeItem targetItem, ITreeItem[] items)
        {
            var parentItem = targetItem.Parent as ComponentTreeItem;
            if (parentItem == null || IsContainerItem(parentItem) == false)
            {
                return false;
            }

            var movable = items
                .OfType<ComponentTreeItem>()
                .All(i => IsMovableItem(i) == true);
            if (movable == false)
            {
                return false;
            }

            var childItems = parentItem.Children
                .Where(i => items.Contains(i) == false);
            var sameNames = GetSameNames(childItems, items);
            if (sameNames.Count() > 0)
            {
                ShowSameNameErrorMessage(sameNames);
                return false;
            }

            return true;
        }

        private IEnumerable<string> GetSameNames(IEnumerable<ITreeItem> destItems, IEnumerable<ITreeItem> srcItems)
        {
            var destNames = new HashSet<string>(destItems
                .OfType<ComponentTreeItem>()
                .Select(i => i.Target.Name));
            return srcItems
                .OfType<ComponentTreeItem>()
                .Select(i => i.Target.Name)
                .Where(n => destNames.Contains(n));
        }

        private void ShowSameNameErrorMessage(IEnumerable<string> names)
        {
            var sb = new StringBuilder();
            foreach (var name in names)
            {
                sb.Append(name);
            }

            var messageBox = new TextDisplayMessageBox(MessageResource.Message_ErrorContainsSameNameFolder,
                sb.ToString(), TextDisplayMessageBoxStyle.OKButton);
            messageBox.ShowDialog();
        }

        private bool IsContainerItem(ComponentTreeItem item)
        {
            return item.Target is SoundProject || item.Target is FolderComponent ? true : false;
        }

        private bool IsMovableItem(ComponentTreeItem item)
        {
            return item.Target is SoundSet || item.Target is FolderComponent ? true : false;
        }

        protected override void MovedItemsInternal(ITreeItem[] items)
        {
            ProjectService.ProjectDocument.SetDirty();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanExpanded(ITreeItem targetItem)
        {
            ComponentTreeItem item = targetItem as ComponentTreeItem;

            if (item != null &&
                item.Target is SoundProject)
            {
                return false;
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragEnter(DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
            {
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragDrop(DragEventArgs e)
        {
            string[] filePaths = null;

            filePaths = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            MainWindow.DropFiles(filePaths);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectChanged(object sender, EventArgs e)
        {
            ComponentTreeItem[] items = SelectedItems;
            if (items.Length != 1) { return; }

            ActivateItem(items[0]);

            string filePath = GetFilePath(items[0]);
            MainWindow.UpdateStatusText((null == filePath) ? string.Empty : filePath);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        private void ActivateItem(ComponentTreeItem item)
        {
            Debug.Assert(null != item, "invalid argument");

            SoundSetBankBase soundSetBank = GetSoundSetBank(item, true);
            if (null != soundSetBank)
            {
                Bank bank = ApplicationBase.Instance.BankServices[soundSetBank.FilePath].Bank;

                FormsUIService service = FormsApplication.Instance.UIService;
                CommonTabPage page = service.MainWindow.FindPage(bank);

                if (page != null)
                {
                    (page.Panel as BankPanel).SoundSetBank = soundSetBank;
                    MainWindow.ActivatePage(page);
                }

                return;
            }


            SoundSet soundSet = GetSoundSet(item);

            if (null != soundSet)
            {
                FormsUIService service = FormsApplication.Instance.UIService;
                CommonTabPage page = service.MainWindow.FindPage(soundSet);

                if (page != null)
                {
                    SoundSetItem soundSetItem = item.Target as SoundSetItem;

                    SoundSetPanel soundSetPanel = page.Panel as SoundSetPanel;
                    if (soundSetPanel.Show(soundSetItem) != false)
                    {
                        MainWindow.ActivatePage(page);
                    }
                }

                return;
            }
        }

        private string GetFilePath(ComponentTreeItem item)
        {
            Debug.Assert(null != item, "invalid argument");

            if (item.Target is SoundProject)
            {
                if (null == ProjectService.ProjectDocument) { return null; }
                return ProjectService.ProjectDocument.Resource.Key;
            }

            SoundSetBankBase soundSetBank = GetSoundSetBank(item, false);
            if (null != soundSetBank)
            {
                return soundSetBank.FilePath;
            }

            SoundSet soundSet = GetSoundSet(item);
            if (null != soundSet)
            {
                return GetSoundSetFilePath(soundSet);
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string GetSoundSetFilePath(SoundSet soundSet)
        {
            Debug.Assert(null != soundSet, "invalid argument");

            foreach (SoundSetDocument soundSetDocument in ProjectService.SoundSetDocuments)
            {
                if (soundSetDocument.SoundSet != soundSet) { continue; }
                return soundSetDocument.Resource.Key;
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSet GetSoundSet(ComponentTreeItem item)
        {
            while (true)
            {
                if (item.Target is SoundSet)
                {
                    return item.Target as SoundSet;
                }

                if ((item = item.Parent) == null)
                {
                    return null;
                }

                if (item.Target is SoundSet)
                {
                    return item.Target as SoundSet;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSetBankBase GetSoundSetBank(ComponentTreeItem item, bool openedBankOnly)
        {
            while (true)
            {
                if (item.Target is SoundSetBankBase)
                {
                    SoundSetBankBase bank = item.Target as SoundSetBankBase;

                    if (openedBankOnly && !FormsApplication.Instance.BankServices.Contains(bank.FilePath))
                    {
                        return null;
                    }

                    return bank;
                }

                if ((item = item.Parent) == null)
                {
                    return null;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ProjectTreeItem FindItems(Component component)
        {
            ProjectTreeAdapter adapter = ItemsSource as ProjectTreeAdapter;
            ProjectTreeItem foundItem = null;

            foreach (ProjectTreeItem item in adapter.Items)
            {
                if ((foundItem = FindItem(item, component)) != null)
                {
                    return foundItem;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ProjectTreeItem FindItem(ProjectTreeItem item, Component component)
        {
            ProjectTreeItem foundItem = null;

            if (item.Target == component)
            {
                return item;
            }

            foreach (ProjectTreeItem childItem in item.Children)
            {
                if ((foundItem = FindItem(childItem, component)) != null)
                {
                    return foundItem;
                }
            }
            return null;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ProjectTreeItem : ComponentTreeItem
    {
        private static Dictionary<Type, string> _NameDictionary = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static ProjectTreeItem()
        {
            _NameDictionary = new Dictionary<Type, string>();

            _NameDictionary.Add(typeof(StreamSoundPack),
                                 MessageResource.ProjectCtrl_StreamSound);
            _NameDictionary.Add(typeof(WaveSoundSetPack),
                                 MessageResource.ProjectCtrl_WaveSoundSet);
            _NameDictionary.Add(typeof(SequenceSoundSetPack),
                                 MessageResource.ProjectCtrl_SequenceSoundSet);
            _NameDictionary.Add(typeof(SequenceSoundPack),
                                 MessageResource.ProjectCtrl_SequenceSound);
            _NameDictionary.Add(typeof(SoundSetBankPack),
                                 MessageResource.ProjectCtrl_Bank);
            _NameDictionary.Add(typeof(WaveArchivePack),
                                 MessageResource.ProjectCtrl_WaveArchive);
            _NameDictionary.Add(typeof(GroupPack),
                                 MessageResource.ProjectCtrl_Group);
            _NameDictionary.Add(typeof(PlayerPack),
                                 MessageResource.ProjectCtrl_Player);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ProjectTreeItem(Component component)
            : base(component)
        {
            Children.CollectionChanged += OnCollectionChanged;

            Container = false;
            Expander = false;
            Expanded = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component Target
        {
            get { return base.Target; }
            set
            {
                if (Target != null)
                {
                    Target.NameChanged -= OnNameChanged;
                }

                base.Target = value;

                if (Target != null)
                {
                    Target.NameChanged += OnNameChanged;
                }
            }
        }

        public string AddonSoundArchiveIconName
        {
            get;
            set;
        }

        public string NoExportIconName
        {
            get;
            set;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void GetConstValues(Dictionary<string, string> parameters)
        {
            base.GetConstValues(parameters);

            if (IsSoundSetItemPack(Target) != false)
            {
                parameters["Caption"] = _NameDictionary[Target.GetType()];
            }

            //
            if (Target is SoundProject)
            {
                SoundProjectDocument projectDocument = null;
                projectDocument = ProjectService.ProjectDocument;

                if (projectDocument.IsDirty != false)
                {
                    parameters["Caption"] = (parameters["Caption"] as string) + " *";
                }
            }

            //
            if (Target is SoundSet)
            {
                SoundSetDocument soundSetDocument = null;
                string caption = null;

                soundSetDocument = MainWindow.GetSoundSetDocument(Target as SoundSet);
                if (soundSetDocument.IsDirty != false)
                {
                    caption = parameters["Caption"] as string;
                    parameters["Caption"] = caption += " *";
                }
            }

            // 出力しない、追加サウンドアーカイブが設定されていたらアイコンを変更します。
            if (this.AddonSoundArchiveIconName != null)
            {
                switch (AddonSoundArchiveUtility.GetSoundArchiveOutputType(this.Target))
                {
                    case SoundArchiveOutputTypes.None:
                        parameters["ExpandIcon"] = this.NoExportIconName;
                        parameters["CollapseIcon"] = this.NoExportIconName;
                        break;

                    case SoundArchiveOutputTypes.SoundArchive:
                        parameters["ExpandIcon"] = this.ExpandIconName;
                        parameters["CollapseIcon"] = this.CollapseIconName;
                        break;

                    case SoundArchiveOutputTypes.AddonSoundArchive:
                        parameters["ExpandIcon"] = this.AddonSoundArchiveIconName;
                        parameters["CollapseIcon"] = this.AddonSoundArchiveIconName;
                        break;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void VisibleCheckSoundSetPack()
        {
            if (IsSoundSetItemPack(Target) == false)
            {
                return;
            }

            //
            SoundSet soundSet = GetSoundSet(Target.Parent);
            ProjectTreeAdapter adapter = Adapter as ProjectTreeAdapter;
            bool visible = true;

            if (soundSet.HideEmptyNode != false)
            {
                visible = Target.Children.Count <= 0 ? false : true;
            }

            if (visible != Visible)
            {
                Visible = visible;
                adapter.NotificationVisibleChange(this);
            }
            else
            {
                Visible = visible;
            }

            UpdateExpander();
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateExpander()
        {
            if (Target is FolderComponent == false)
            {
                var expandable = Children.Count > 0 ? true : false;
                Container = expandable;
                Expander = expandable;
                Expanded = expandable;
            }
        }

        protected override IParameterValue GetParameterValue(string name)
        {
            if (name == "Name")
            {
                var targetName = (string)Target.Parameters[name].Value;
                var siblingNames = Target.Parent.Children
                        .OfType<Component>()
                        .Where(c => c != Target)
                        .Select(c => c.Name);

                if (Target is SoundSet)
                {
                    return new SoundSetNameParameterValue(targetName, siblingNames);
                }
                else if (Target is FolderComponent)
                {
                    return new FolderNameParameterValue(targetName, siblingNames);
                }
            }

            return base.GetParameterValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnTargetCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            base.OnTargetCollectionChanged(e);

            VisibleCheckSoundSetPack();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, EventArgs e)
        {
            UpdateExpander();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnNameChanged(object sender, NameChangedEventArgs e)
        {
            if (sender is SoundSet ||
                sender is FolderComponent)
            {
                ProjectService.ProjectDocument.SetDirty();
            }

            Adapter?.Update();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSet GetSoundSet(Component component)
        {
            while (true)
            {
                if (component is SoundSet)
                {
                    return component as SoundSet;
                }

                component = component.Parent;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool IsSoundSetItemPack(Component component)
        {
            if (component is StreamSoundPack ||
                component is WaveSoundSetPack ||
                component is SequenceSoundPack ||
                component is SequenceSoundSetPack ||
                component is SoundSetBankPack ||
                component is WaveArchivePack ||
                component is GroupPack ||
                component is PlayerPack)
            {
                return true;
            }
            return false;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ProjectTreeAdapter : ComponentTreeAdapter
    {
        private SoundProject _Project = null;
        private ProjectTreeItem _ProjectItem = null;

        public string FilterText { get; set; } = string.Empty;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void AttachProject(SoundProject project)
        {
            DisposeItems();

            _ProjectItem = null;
            if ((_Project = project) != null)
            {
                _ProjectItem = Create(_Project) as ProjectTreeItem;
            }

            Items.Clear();
            if (_Project != null)
            {
                Items.Add(_ProjectItem);
            }

            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override string NameConvert(string name)
        {
            if (name == "Caption")
            {
                return "Name";
            }
            return name;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void NotificationVisibleChange(ProjectTreeItem item)
        {
            Update();
        }

        public void UpdateEmptyNodesExpander()
        {
            foreach (SoundSetDocument soundSetDocument in ProjectService.SoundSetDocuments)
            {
                ProjectTreeItem parentItem = FindTreeItem
                    (soundSetDocument.SoundSet) as ProjectTreeItem;
                Debug.Assert(parentItem != null, "parentItem is not ProjectTreeItem");

                foreach (ProjectTreeItem item in parentItem.Children)
                {
                    item.UpdateExpander();
                }
            }
        }

        public void HideEmptyNode(SoundSet soundSet, bool b)
        {
            soundSet.HideEmptyNode = b;
            UpdateNodeVisibility();
        }

        public void UpdateNodeVisibility()
        {
            var projectItem = FindTreeItem(ProjectService.Project);
            foreach (var childItem in projectItem.Children.OfType<ProjectTreeItem>())
            {
                FilterItems(FilterText, childItem);
            }

            Update();
        }

        private void FilterItems(string text, ProjectTreeItem item)
        {
            if (item.Target is SoundSetItemPack)
            {
                item.VisibleCheckSoundSetPack();
            }
            else if (item.Target is SoundSet ||
                     item.Target is WaveSoundSetBase ||
                     item.Target is SequenceSoundSetBase ||
                     item.Target is SoundSetBankBase ||
                     item.Target is FolderComponent)
            {
                if (item.Target.Name.IndexOf(text, StringComparison.OrdinalIgnoreCase) >= 0)
                {
                    item.Visible = true;

                    var parentItem = item.Parent;
                    while (parentItem.Target is SoundProject == false)
                    {
                        // フィルタ文字列が空白の時にはエキスパンダの状態は変更しません。
                        if (text != string.Empty)
                        {
                            parentItem.Expanded = true;
                        }

                        parentItem.Visible = true;
                        parentItem = parentItem.Parent;
                    }
                }
                else
                {
                    item.Visible = false;
                }
            }
            else
            {
                item.Visible = true;
            }

            foreach (var childItem in item.Children.OfType<ProjectTreeItem>())
            {
                FilterItems(text, childItem);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override ComponentTreeItem CreateTreeItem(Component component)
        {
            if (component is StreamSoundBase ||
                component is WaveSoundBase ||
                component is SequenceSoundBase ||
                component is WaveArchiveBase ||
                component is PlayerBase ||
                component is GroupBase)
            {
                return null;
            }

            var data = GetTreeItemData(component);
            if (data == null)
            {
                return null;
            }

            var item = new ProjectTreeItem(component);
            item.Container = data.Container;
            item.Expander = data.Expander;
            item.Expanded = data.Expanded;
            item.CanEdit = data.Editable;
            item.ExpandIconName = data.ExpandIconName;
            item.CollapseIconName = data.CollapseIconName;
            item.AddonSoundArchiveIconName = data.AddonSoundArchiveIconName;
            item.NoExportIconName = data.NoExportIconName;

            return item;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private class TreeItemData
        {
            public Type ComponentType { get; set; }
            public string ExpandIconName { get; set; }
            public string CollapseIconName { get; set; }
            public string AddonSoundArchiveIconName { get; private set; }
            public string NoExportIconName { get; private set; }
            public bool Container { get; set; }
            public bool Expander { get; set; }
            public bool Expanded { get; set; }
            public bool Editable { get; set; }

            public TreeItemData(Type componentType, string iconName, bool container, bool expander, bool expanded, bool editable) :
                this(componentType, iconName, iconName, null, null, container, expander, expanded, editable)
            { }

            public TreeItemData(Type componentType, string expandIconName, string collapseIconName, bool container, bool expander, bool expanded, bool editable) :
                this(componentType, expandIconName, collapseIconName, null, null, container, expander, expanded, editable)
            { }

            public TreeItemData(Type componentType, string iconName, string addonSoundArchiveIconName, string noExportIconName, bool container, bool expander, bool expanded, bool editable) :
                this(componentType, iconName, iconName, addonSoundArchiveIconName, noExportIconName, container, expander, expanded, editable)
            { }

            private TreeItemData(Type componentType, string expandIconName, string collapseIconName, string addonSoundArchiveIconName, string noExportIconName, bool container, bool expander, bool expanded, bool editable)
            {
                ComponentType = componentType;
                ExpandIconName = expandIconName;
                CollapseIconName = collapseIconName;
                AddonSoundArchiveIconName = addonSoundArchiveIconName;
                NoExportIconName = noExportIconName;
                Container = container;
                Expander = expander;
                Expanded = expanded;
                Editable = editable;
            }
        }

        ///
        private TreeItemData[] _TreeItemData = new TreeItemData[] {
            new TreeItemData( typeof(SoundProject),
                              "IconProject",
                              true, true, true, false),
            new TreeItemData( typeof(SoundSet),
                              "IconSoundSet",
                              "IconAddonSoundSet",
                              "IconNoExportSoundSet",
                              true, true, true, true),

            new TreeItemData( typeof(WaveSoundSetBase),
                              "IconWaveSoundSet",
                              false, false, false, false),
            new TreeItemData( typeof(SequenceSoundSetBase),
                              "IconSequenceSoundSet",
                              false, false, false, false),
            new TreeItemData( typeof(SoundSetBankBase),
                              "IconBank",
                              false, false, false, false),

            new TreeItemData( typeof(StreamSoundPack),
                              "IconStreamSoundPack",
                              false, false, false, false),
            new TreeItemData( typeof(WaveSoundSetPack),
                              "IconWaveSoundSetPack",
                              true, true, true, false),
            new TreeItemData( typeof(SequenceSoundSetPack),
                              "IconSequenceSoundSetPack",
                              true, true, true, false),
            new TreeItemData( typeof(SequenceSoundPack),
                              "IconSequenceSoundPack",
                              false, false, false, false),
            new TreeItemData( typeof(SoundSetBankPack),
                              "IconBankPack",
                              true, true, true, false),
            new TreeItemData( typeof(WaveArchivePack),
                              "IconWaveArchivePack",
                              false, false, false, false),
            new TreeItemData( typeof(GroupPack),
                              "IconGroupPack",
                              "IconAddonGroupPack",
                              "IconGroupPack",
                              false, false, false, false),
            new TreeItemData( typeof(PlayerPack),
                              "IconPlayerPack",
                              "IconAddonPlayerPack",
                              "IconPlayerPack",
                              false, false, false, false),

            new TreeItemData( typeof(FolderComponent),
                "IconFolderOpen",
                "IconFolderClose",
                true, true, false, true),
        };

        ///
        private TreeItemData GetTreeItemData(Component component)
        {
            foreach (TreeItemData data in _TreeItemData)
            {
                if (TypeComparer.HasAncestor(data.ComponentType, component.GetType()) == true)
                {
                    return data;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentTreeItem CreatePack(Component component)
        {
            ComponentTreeItem item = null;
            ComponentTreeItem childItem = null;

            if ((item = CreateTreeItem(component)) == null)
            {
                return null;
            }

            //
            if (component is WaveSoundSetPack ||
                component is SequenceSoundSetPack ||
                component is SoundSetBankPack)
            {

                foreach (Component childComponent in component.Children)
                {
                    childItem = CreateTreeItem(childComponent);
                    item.Children.Add(childItem);
                }
            }

            return item;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component FindComponentByType(ComponentList list, Type componentType)
        {
            foreach (Component component in list)
            {
                if (component.GetType() == componentType)
                {
                    return component;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentTreeItem CreateSoundSet(SoundSet soundSet)
        {
            Type[] componentTypes = new[] {
                typeof(StreamSoundPack),
                typeof(WaveSoundSetPack),
                typeof(SequenceSoundPack),
                typeof(SequenceSoundSetPack),
                typeof(SoundSetBankPack),
                typeof(WaveArchivePack),
                typeof(GroupPack),
                typeof(PlayerPack),
            };

            ComponentTreeItem soundSetItem = null;
            ComponentTreeItem item = null;
            Component component = null;

            soundSetItem = CreateTreeItem(soundSet);

            foreach (Type componentType in componentTypes)
            {
                component = FindComponentByType(soundSet.Children, componentType);
                if ((item = CreatePack(component)) != null)
                {
                    soundSetItem.Children.Add(item);
                }
            }
            return soundSetItem;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentTreeItem Create(SoundProject project)
        {
            ComponentTreeItem projectItem = null;

            Suspend = true;

            projectItem = CreateTreeItem(project);
            Create(projectItem, project);

            Suspend = false;
            return projectItem;
        }

        private void Create(ComponentTreeItem parentItem, Component parentComponent)
        {
            foreach (var component in parentComponent.Children)
            {
                if (component is FolderComponent)
                {
                    var item = CreateTreeItem(component);
                    parentItem.Children.Add(item);

                    Create(item, component);
                }
                else if (component is SoundSet)
                {
                    var soundSet = component as SoundSet;
                    var item = CreateSoundSet(soundSet);
                    parentItem.Children.Add(item);
                }
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class SoundSetNameParameterValue : TextParameterValue
    {
        private HashSet<string> _unacceptableNames;

        public SoundSetNameParameterValue(string name, IEnumerable<string> unacceptableNames)
            : base(name)
        {
            _unacceptableNames = new HashSet<string>(unacceptableNames);
        }

        private SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        protected override ValidationResult ValidateInternal(string value)
        {
            if (value == null) { return new ValidationResult(false); }

            RegularExpressions.Regex regex = new RegularExpressions.Regex(@"\r\n");

            if (value == String.Empty ||
                (new RegularExpressions.Regex(@"\r\n")).IsMatch(value) != false ||
                (new RegularExpressions.Regex("[:\\\\/*?\"<>|]")).IsMatch(value) != false)
            {
                return new ValidationResult
                    (false, MessageResource.Message_InvalidLabel);
            }

            foreach (SoundSetDocument soundSetDocument in ProjectService.SoundSetDocuments)
            {
                if (soundSetDocument.SoundSet.Name == value)
                {
                    return new ValidationResult
                        (false, MessageResource.Message_ItemNameAlreadyExisting);
                }
            }

            if (_unacceptableNames.Contains(value) == true)
            {
                return new ValidationResult(false, MessageResource.Message_ItemNameAlreadyExisting);
            }

            return ValidationResult.NoError;
        }
    }

    public class FolderNameParameterValue : TextParameterValue
    {
        private HashSet<string> _unacceptableNames;

        public FolderNameParameterValue(string name, IEnumerable<string> unacceptableNames)
            : base(name)
        {
            _unacceptableNames = new HashSet<string>(unacceptableNames);
        }

        protected override ValidationResult ValidateInternal(string value)
        {
            if (value == null)
            {
                return new ValidationResult(false);
            }

            if (value == String.Empty ||
                (new RegularExpressions.Regex(@"\r\n")).IsMatch(value) != false ||
                (new RegularExpressions.Regex("[:\\\\/*?\"<>|]")).IsMatch(value) != false)
            {
                return new ValidationResult(false, MessageResource.Message_InvalidLabel);
            }

            if (_unacceptableNames.Contains(value) == true)
            {
                return new ValidationResult(false, MessageResource.Message_ItemNameAlreadyExisting);
            }

            return ValidationResult.NoError;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ProjectPartDrawer
    {
        public static bool IsTarget(Component component)
        {
            if (component is SoundSetBankBase ||
                component is SequenceSoundSetBase ||
                component is WaveSoundSetBase)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ProjectPartIconDrawer : ITreePartDrawer
    {
        private TreePartIconDrawer drawer = new TreePartIconDrawer();

        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get
            {
                return this.drawer.Name;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void Draw(TreeItemDrawDescriptor desc)
        {
            this.drawer.Draw(desc);

            ProjectTreeItem treeItem = desc.TreeItem as ProjectTreeItem;
            Component component = treeItem.Target;
            if (ProjectPartDrawer.IsTarget(component) != false)
            {
                if (component.IsEnabled == false)
                {
                    Image image = ImageResource.PngOverlayDisabled;
                    ImageAttributes attr = new ImageAttributes();
                    attr.SetColorKey(Color.Magenta, Color.Magenta);
                    int width = desc.Bounds.Width;
                    int height = desc.Bounds.Height;
                    int x = desc.Bounds.X + (int)((double)width * 0.4);
                    int y = desc.Bounds.Y + (int)((double)height * 0.4);

                    desc.Gc.DrawImage(image, new Rectangle(x, y, width, height),
                                       0, 0, width, height,
                                       GraphicsUnit.Pixel, attr);
                }
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ProjectPartCaptionDrawer : TreePartCaptionDrawer
    {
        protected override Brush GetCaptionBrush(TreeItemDrawDescriptor desc)
        {
            ProjectTreeItem treeItem = desc.TreeItem as ProjectTreeItem;
            Component component = treeItem.Target;
            if (ProjectPartDrawer.IsTarget(component) != false)
            {
                if (component.IsEnabled == false)
                {
                    return SystemBrushes.GrayText;
                }
            }

            return base.GetCaptionBrush(desc);
        }
    }
}
