﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.ComponentModel;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;

    public partial class BitOccupancyControl : UserControl
    {
        private static readonly StringFormat bitLocationFormat = new StringFormat()
        {
            Alignment = StringAlignment.Far,
            LineAlignment = StringAlignment.Center,
            Trimming = StringTrimming.None,
            FormatFlags = StringFormatFlags.FitBlackBox | StringFormatFlags.NoWrap | StringFormatFlags.NoClip,
        };

        private readonly ObservableCollection<BitsInfo> usedBits = new ObservableCollection<BitsInfo>();
        private readonly DrawItemComponents drawItemComponents = new DrawItemComponents();
        private readonly HashSet<int> highlightedBits = new HashSet<int>();
        private readonly HashSet<int> errorBits = new HashSet<int>();
        private int totalSize = 32;

        //-----------------------------------------------------------------

        public class BitsInfo
        {
            public int Location { get; set; }
            public int Size { get; set; }

            public bool Contains(int bitLocation)
            {
                return
                    this.Location <= bitLocation &&
                    bitLocation < this.Location + this.Size;
            }

            public bool IsMsb(int bitLocation)
            {
                if (!this.Contains(bitLocation))
                {
                    return false;
                }

                return this.Location == bitLocation;
            }

            public bool IsLsb(int bitLocation)
            {
                if (!this.Contains(bitLocation))
                {
                    return false;
                }

                return bitLocation == this.Location + this.Size - 1;
            }
        }

        private struct DrawItemInfo
        {
            public int BitLocation { get; set; }
            public bool IsMsb { get; set; }
            public bool IsLsb { get; set; }
            public bool IsUsed { get; set; }
            public bool IsHighlighted { get; set; }
            public bool IsError { get; set; }

            public Rectangle Rectangle { get; set; }
            public Rectangle TextRectangle { get; set; }

            public DrawItemComponents Components { get; set; }
        }

        private class DrawItemComponents
        {
            public Font Font { get; set; }

            public Pen UsedBitPen { get; set; }
            public Pen UnusedBitPen { get; set; }

            public Brush UsedBitBrush { get; set; }
            public Brush UnusedBitBrush { get; set; }
            public Brush HighlightedUsedBitBrush { get; set; }
            public Brush HighlightedUnusedBitBrush { get; set; }

            public Brush UsedBitTextBrush { get; set; }
            public Brush UnusedBitTextBrush { get; set; }
            public Brush HighlightedUsedBitTextBrush { get; set; }
            public Brush HighlightedUnusedBitTextBrush { get; set; }
            public Brush ErrorUsedBitTextBrush { get; set; }
        }

        //-----------------------------------------------------------------

        public BitOccupancyControl()
        {
            InitializeComponent();

            this.usedBits.CollectionChanged += (sender, e) => this.Invalidate();
        }

        //-----------------------------------------------------------------

        public int TotalSize
        {
            get { return this.totalSize; }

            set
            {
                if (this.totalSize == value)
                {
                    return;
                }

                this.totalSize = value;
                this.Invalidate();
            }
        }

        public IList<BitsInfo> UsedBits
        {
            get { return this.usedBits; }
        }

        public HashSet<int> HighlightedBits
        {
            get { return this.highlightedBits; }
        }

        public HashSet<int> ErrorBits
        {
            get { return this.errorBits; }
        }

        /// <summary>
        /// ちらつきを軽減または回避するために、2 次バッファを使用してコントロールの表面を再描画するかどうかを示す値を取得または設定します。
        /// </summary>
        [DefaultValue(true)]
        protected override bool DoubleBuffered
        {
            get { return base.DoubleBuffered; }
            set { base.DoubleBuffered = value; }
        }

        //-----------------------------------------------------------------

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (this.totalSize <= 0)
            {
                return;
            }

            this.PreparePaint();

            // パディング分を差し引きます。
            // 下線が描画されるように高さを -1 します。
            Rectangle drawRect = this.ClientRectangle;
            drawRect.X += this.Padding.Left;
            drawRect.Y += this.Padding.Top;
            drawRect.Width -= this.Padding.Right;
            drawRect.Height -= this.Padding.Bottom + 1;

            int itemWidth = drawRect.Width / this.totalSize;
            int itemHeight = drawRect.Height;

            SizeF locationSize = e.Graphics.MeasureString(this.totalSize.ToString(), this.Font);
            itemWidth = Math.Max(itemWidth, Convert.ToInt32(Math.Ceiling(locationSize.Width)));

            // 使用ビットの境界線を上に描くために、使用ビットを後で描きます。
            List<DrawItemInfo> usedBitDrawItemInfos = new List<DrawItemInfo>();

            for (int bitLocation = this.totalSize - 1; bitLocation >= 0; --bitLocation)
            {
                int index = this.totalSize - bitLocation - 1;
                BitsInfo bitsInfo = this.GetUsedBitsInfo(bitLocation);

                DrawItemInfo drawItemInfo = new DrawItemInfo()
                {
                    BitLocation = bitLocation,
                    IsMsb = bitsInfo != null ? bitsInfo.IsMsb(bitLocation) : true,
                    IsLsb = bitsInfo != null ? bitsInfo.IsLsb(bitLocation) : true,
                    IsUsed = bitsInfo != null,
                    IsHighlighted = this.highlightedBits.Contains(bitLocation),
                    IsError = this.errorBits.Contains(bitLocation),
                    Rectangle = new Rectangle(index * itemWidth, 0, itemWidth, itemHeight),
                    TextRectangle = new Rectangle(index * itemWidth, 2, itemWidth, itemHeight - 2),
                    Components = this.drawItemComponents,
                };

                if (bitsInfo != null)
                {
                    usedBitDrawItemInfos.Add(drawItemInfo);
                }
                else
                {
                    this.DrawItem(e.Graphics, drawItemInfo);
                }
            }

            // 使用ビットを描画します。
            foreach (DrawItemInfo drawItemInfo in usedBitDrawItemInfos)
            {
                this.DrawItem(e.Graphics, drawItemInfo);
            }
        }

        protected override void OnSystemColorsChanged(EventArgs e)
        {
            base.OnSystemColorsChanged(e);
            this.drawItemComponents.UsedBitPen = null;
            this.drawItemComponents.UsedBitBrush = null;
            this.drawItemComponents.HighlightedUsedBitBrush = null;
            this.drawItemComponents.HighlightedUnusedBitBrush = null;
            this.drawItemComponents.HighlightedUsedBitTextBrush = null;
            this.drawItemComponents.HighlightedUnusedBitTextBrush = null;
        }

        protected override void OnForeColorChanged(EventArgs e)
        {
            base.OnForeColorChanged(e);
            this.drawItemComponents.UsedBitTextBrush = null;
            this.drawItemComponents.UnusedBitTextBrush = null;
        }

        protected override void OnBackColorChanged(EventArgs e)
        {
            base.OnBackColorChanged(e);
            this.drawItemComponents.UnusedBitBrush = null;
        }

        private void PreparePaint()
        {
            this.drawItemComponents.Font = this.Font;

            // ペン
            if (this.drawItemComponents.UsedBitPen == null)
            {
                this.drawItemComponents.UsedBitPen = new Pen(
                    Color.FromArgb(
                        Math.Max(SystemColors.Highlight.R - 0x80, 0),
                        Math.Max(SystemColors.Highlight.G - 0x80, 0),
                        Math.Max(SystemColors.Highlight.B - 0x80, 0))
                    );
            }

            this.drawItemComponents.UnusedBitPen = SystemPens.ControlDark;

            // ブラシ
            if (this.drawItemComponents.UsedBitBrush == null)
            {
                this.drawItemComponents.UsedBitBrush = new SolidBrush(
                    Color.FromArgb(72, SystemColors.Highlight));
            }

            if (this.drawItemComponents.UnusedBitBrush == null)
            {
                this.drawItemComponents.UnusedBitBrush = new SolidBrush(this.BackColor);
            }

            // ブラシ（ハイライト）
            if (this.drawItemComponents.HighlightedUsedBitBrush == null)
            {
                this.drawItemComponents.HighlightedUsedBitBrush = SystemBrushes.Highlight;
            }

            if (this.drawItemComponents.HighlightedUnusedBitBrush == null)
            {
                this.drawItemComponents.HighlightedUnusedBitBrush = SystemBrushes.Highlight;
            }

            // テキストブラシ
            if (this.drawItemComponents.UsedBitTextBrush == null)
            {
                this.drawItemComponents.UsedBitTextBrush = new SolidBrush(this.ForeColor);
            }

            this.drawItemComponents.UnusedBitTextBrush = SystemBrushes.GrayText;

            // テキストブラシ（ハイライト）
            if (this.drawItemComponents.HighlightedUsedBitTextBrush == null)
            {
                this.drawItemComponents.HighlightedUsedBitTextBrush = SystemBrushes.HighlightText;
            }

            if (this.drawItemComponents.HighlightedUnusedBitTextBrush == null)
            {
                this.drawItemComponents.HighlightedUnusedBitTextBrush = SystemBrushes.HighlightText;
            }

            // テキストブラシ（エラー）
            if (this.drawItemComponents.ErrorUsedBitTextBrush == null)
            {
                this.drawItemComponents.ErrorUsedBitTextBrush = Brushes.OrangeRed;
            }
        }

        private BitsInfo GetUsedBitsInfo(int bitLocation)
        {
            if (bitLocation < 0 || this.totalSize <= bitLocation)
            {
                return null;
            }

            return this.usedBits.
                FirstOrDefault(bitsInfo => bitsInfo.Contains(bitLocation));
        }

        private void DrawItem(Graphics graphics, DrawItemInfo drawItemInfo)
        {
            // 背景色
            Brush fillBrush = drawItemInfo.Components.UnusedBitBrush;

            if (drawItemInfo.IsUsed)
            {
                fillBrush = drawItemInfo.IsHighlighted ?
                    drawItemInfo.Components.HighlightedUsedBitBrush : drawItemInfo.Components.UsedBitBrush;
            }
            else
            {
                fillBrush = drawItemInfo.IsHighlighted ?
                    drawItemInfo.Components.HighlightedUnusedBitBrush : drawItemInfo.Components.UnusedBitBrush;
            }

            graphics.FillRectangle(fillBrush, drawItemInfo.Rectangle);

            // 境界線
            Pen borderPen = drawItemInfo.IsUsed ?
                drawItemInfo.Components.UsedBitPen : drawItemInfo.Components.UnusedBitPen;

            if (drawItemInfo.IsLsb && drawItemInfo.IsMsb)
            {
                graphics.DrawRectangle(borderPen, drawItemInfo.Rectangle);
            }
            else
            {
                graphics.DrawLine(
                    borderPen,
                    drawItemInfo.Rectangle.Left,
                    drawItemInfo.Rectangle.Top,
                    drawItemInfo.Rectangle.Right,
                    drawItemInfo.Rectangle.Top);

                graphics.DrawLine(
                    borderPen,
                    drawItemInfo.Rectangle.Left,
                    drawItemInfo.Rectangle.Bottom,
                    drawItemInfo.Rectangle.Right,
                    drawItemInfo.Rectangle.Bottom);

                if (drawItemInfo.IsMsb)
                {
                    graphics.DrawLine(
                        borderPen,
                        drawItemInfo.Rectangle.Right,
                        drawItemInfo.Rectangle.Top,
                        drawItemInfo.Rectangle.Right,
                        drawItemInfo.Rectangle.Bottom);
                }

                if (drawItemInfo.IsLsb)
                {
                    graphics.DrawLine(
                        borderPen,
                        drawItemInfo.Rectangle.Left,
                        drawItemInfo.Rectangle.Top,
                        drawItemInfo.Rectangle.Left,
                        drawItemInfo.Rectangle.Bottom);
                }
            }

            // テキスト
            Brush textBrush = drawItemInfo.Components.UnusedBitTextBrush;

            if (drawItemInfo.IsUsed)
            {
                textBrush = drawItemInfo.IsHighlighted ?
                    drawItemInfo.Components.HighlightedUsedBitTextBrush : drawItemInfo.Components.UsedBitTextBrush;

                if (drawItemInfo.IsError)
                {
                    textBrush = drawItemInfo.Components.ErrorUsedBitTextBrush;
                }
            }
            else
            {
                textBrush = drawItemInfo.IsHighlighted ?
                    drawItemInfo.Components.HighlightedUnusedBitTextBrush : drawItemInfo.Components.UnusedBitTextBrush;
            }

            graphics.DrawString(
                drawItemInfo.BitLocation.ToString(),
                drawItemInfo.Components.Font,
                textBrush,
                drawItemInfo.TextRectangle,
                BitOccupancyControl.bitLocationFormat);
        }
    }
}
