﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Runtime.InteropServices;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.SoundFoundation.Core.Win32;
    using NintendoWare.SoundFoundation.FileFormats.Audio;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using Win32 = NintendoWare.SoundFoundation.Core.Win32;

    /// <summary>
    /// WindowsForms ベースのアプリケーションクラスです。
    /// </summary>
    public abstract class FormsApplication : ApplicationBase
    {
        ///
        private static string reservedActivateProjectFilePath = null;
        private static string reservedActivateItemName = null;
        private static string reservedActivateCommand = null;
        private static string reservedActivateValue1 = null;
        private static string reservedActivateValue2 = null;
        private static string reservedActivateValue3 = null;
        private static string reservedActivateValue4 = null;
        private static string reservedActivateValue5 = null;

        private string targetProjectFilePath = string.Empty;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="traits">アプリケーション特性を指定します。</param>
        protected FormsApplication(IApplicationTraits traits)
            : base(traits)
        {
            this.InitializeQuesterSettings();

            Application.ThreadException += OnThreadException;
            System.Threading.Thread.GetDomain().UnhandledException += OnUnhandledException;
        }

        /// <summary>
        /// インスタンスを取得します。
        /// </summary>
        public static new FormsApplication Instance
        {
            get { return ApplicationBase.Instance as FormsApplication; }
        }

        /// <summary>
        /// UI サービスを取得します。
        /// </summary>
        public new FormsUIService UIService
        {
            get { return base.UIService as FormsUIService; }
        }

        /// <summary>
        /// パラメータパネルを取得します。
        /// </summary>
        public ParameterPanel ParameterPanel
        {
            get
            {
                return UIService.MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public InGamePanel InGamePanel
        {
            get
            {
                return UIService.MainWindow.ToolPages[InGamePanel.PageName] as InGamePanel;
            }
        }

        /// <summary>
        /// オーディオデバイスのエラー状態、true エラー状態 false 正常な状態
        /// </summary>
        public bool IsErrorAudioDevice
        {
            get; set;
        }

        /// <summary>
        /// オーディオデバイス初期化の失敗を警告メッセージで表示します。
        /// </summary>
        public void ShowWarningMessageCanNotInitializeAudioDevice()
        {
            UIService.ShowMessageBox(
                MessageResource.Message_CanNotInitializeAudioDevice,
                MessageResource.Message_Warning,
                AppMessageBoxButton.OK,
                AppMessageBoxImage.Warning);
        }

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数。</param>
        protected override void RunApplication(string[] args)
        {
            try
            {
                CommandLineParser commandLineParser = new CommandLineParser();
                commandLineParser.ParseArguments(args);

                if (commandLineParser.IsValueEnabled())
                {
                    this.targetProjectFilePath = commandLineParser.GetValue().GetFullPath();
                }
            }
            catch
            {
            }

            UIService.MainWindow.Activated += OnMainWindowActivated;
            UIService.MainWindow.Deactivate += OnMainWindowDeactivated;

            UIService.MainWindow.Load += OnMainWindowLoad;
            UIService.MainWindow.Shown += OnMainWindowShown;

            Application.Run(UIService.MainWindow);
        }

        /// <summary>
        /// アプリケーションを終了します。
        /// </summary>
        protected override bool ExitApplication()
        {
            Application.Exit();
            return true;
        }

        /// <summary>
        /// サービスのインスタンスを作成します。
        /// </summary>
        /// <param name="serviceType">作成するサービスの種類。</param>
        /// <returns>サービスのインスタンス。</returns>
        protected override object CreateService(Type serviceType)
        {
            if (serviceType == typeof(UIServiceBase))
            {
                return new FormsUIService();
            }

            return base.CreateService(serviceType);
        }

        /// <summary>
        /// コマンドバインディングを作成します。
        /// </summary>
        /// <param name="serviceType">関連付けるコマンド。</param>
        /// <returns>コマンドバインディングのインスタンス。</returns>
        protected override CommandBinding CreateCommandBinding(Command command)
        {
            // ファイル関連コマンドハンドラ
            if (command == FileCommands.NewProject)
            {
                return new CommandBinding(this, new QueryAndNewProjectHandler(this));
            }
            if (command == FileCommands.OpenProject)
            {
                return new CommandBinding(this, new QueryAndOpenProjectHandler(this));
            }
            if (command == FileCommands.ExitApplication)
            {
                return new CommandBinding(this, new ExitApplicationHandler());
            }
            if (command == FileCommands.ImportProject)
            {
                return new CommandBinding(this, new QueryAndImportProjectHandler(this));
            }
            if (command == FileCommands.ImportSoundSet)
            {
                return new CommandBinding(this, new QueryAndImportSoundSetHandler(this));
            }

            return base.CreateCommandBinding(command);
        }

        /// <summary>
        /// 指定コマンドを実行できる他のコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        protected override ICommandTarget FindOtherTarget(Command command)
        {
            // FormsApplication 以外のクラスでコマンド実行する場合は、ここにコードを追加。
            if (null == UIService) { return null; }
            if (null == UIService.MainWindow) { return null; }

            return (UIService.MainWindow as ICommandTarget).FindTarget(command);
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        protected override bool ExecuteCommandInternal(Command command)
        {
#if DEBUG
            // VisualStudio でデバッグ実行すると、
            // Form.ActiveForm に Microsoft.VisualStudio.HostingProcess.ParkingWindow が
            // 設定されることがあるので、Microsoft.VisualStudio から始まるクラスの場合を除外する
            if (Form.ActiveForm != null &&
                Form.ActiveForm.GetType().Namespace != null &&
                !Form.ActiveForm.GetType().Namespace.StartsWith("Microsoft.VisualStudio"))
#else
            if (null != Form.ActiveForm)
#endif
            {
                if (Form.ActiveForm.GetType().GetCustomAttributes(
                    typeof(CommandKeyProcessableAttribute), true).Length == 0)
                {
                    return false;
                }
            }

            return base.ExecuteCommandInternal(command);
        }

        /// <summary>
        /// アプリケーションが実行されると発生します。
        /// </summary>
        /// <param name="e"></param>
        protected virtual void OnRun(EventArgs e)
        {
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="e"></param>
        protected virtual void OnShown(EventArgs e)
        {
            UIService.TopMostMessageBox = true;
            this.StartupProject(this.targetProjectFilePath, UIService.MainWindow.OpenProjectFile);
            UIService.TopMostMessageBox = false;
        }

        /// <summary>
        ///
        /// </summary>
        [DllImport("user32.dll")]
        private static extern int SendMessage(IntPtr hWnd, uint Msg, IntPtr wParam, IntPtr lParam);

        /// <summary>
        /// 既に起動している SoundMakerにメッセージを送ります。
        /// メッセージは、"プロジェクトパス|コマンド|アイテム名|パラメータ名|パラメータ値"です。
        /// 戻り値、true = 処理が行われた、 false = 処理が行われていない
        /// </summary>
        public static bool SendToSoundMaker(string applicationProcessName, string projectFilePath, string command)
        {
            return SendToSoundMaker(applicationProcessName, projectFilePath, command, string.Empty);
        }

        public static bool SendToSoundMaker(string applicationProcessName, string projectFilePath, string command, string itemName)
        {
            return SendToSoundMaker(applicationProcessName, projectFilePath, command, itemName, string.Empty, string.Empty, string.Empty, string.Empty, string.Empty);
        }

        public static bool SendToSoundMaker(string applicationProcessName, string projectFilePath, string command, string itemName, string parameterName, string parameterValue)
        {
            return SendToSoundMaker(applicationProcessName, projectFilePath, command, itemName, parameterName, parameterValue, string.Empty, string.Empty, string.Empty);
        }

        public static bool SendToSoundMaker(string applicationProcessName, string projectFilePath, string command, string itemName, string value1, string value2, string value3, string value4, string value5)
        {
            if (String.IsNullOrEmpty(projectFilePath) != false ||
                command == null ||
                itemName == null)
            {
                return false;
            }

            // SoundMakerのプロセスを取得します。
            List<Process> list = new List<Process>();

            Process[] processes = Process.GetProcessesByName(applicationProcessName);
            if (processes.Length > 0)
            {
                list.AddRange(processes);
            }

#if DEBUG
            processes = Process.GetProcessesByName(applicationProcessName + ".vshost");
            if (processes.Length > 0)
            {
                list.AddRange(processes);
            }
#endif

            int currentProcessId = Process.GetCurrentProcess().Id;
            processes = list
                .Where(p => p.Id != currentProcessId)
                .ToArray();

            if (processes.Length <= 0)
            {
                // 1つ目のSoundMakerの起動なので、プロジェクトファイルパス、アイテム名を記録して
                // メインウインドウが表示されてからアクティブ処理を行います
                reservedActivateProjectFilePath = projectFilePath;
                reservedActivateItemName = itemName;
                reservedActivateCommand = command;
                reservedActivateValue1 = value1;
                reservedActivateValue2 = value2;
                reservedActivateValue3 = value3;
                reservedActivateValue4 = value4;
                reservedActivateValue5 = value5;

                return false;
            }

            //
            foreach (Process process in processes)
            {
                IntPtr hWnd = process.MainWindowHandle;
                if (SendItemActivateMessage(hWnd, projectFilePath, command, itemName, value1, value2, value3, value4, value5) == true)
                {
                    return true;
                }
            }

            reservedActivateProjectFilePath = projectFilePath;
            reservedActivateItemName = itemName;
            reservedActivateCommand = command;
            reservedActivateValue1 = value1;
            reservedActivateValue2 = value2;
            reservedActivateValue3 = value3;
            reservedActivateValue4 = value4;
            reservedActivateValue5 = value5;

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private static bool SendItemActivateMessage(IntPtr hWnd, string projectFilePath, string command, string itemName, string value1, string value2, string value3, string value4, string value5)
        {
            if (hWnd == IntPtr.Zero)
            {
                return false;
            }

            string message;
            if (command == string.Empty && itemName == string.Empty)
            {
                message = String.Format("{0}", projectFilePath);
            }
            else
            {
                message = String.Format("{0}|{1}|{2}|{3}|{4}|{5}|{6}|{7}", projectFilePath, command, itemName, value1, value2, value3, value4, value5);
            }

            COPYDATASTRUCT cds = new COPYDATASTRUCT();
            cds.dwData = IntPtr.Zero;
            cds.lpData = Marshal.StringToHGlobalUni(message);
            cds.cbData = (uint)(message.Length * 2);

            IntPtr data = Marshal.AllocHGlobal(Marshal.SizeOf(cds));
            Marshal.StructureToPtr(cds, data, false);
            int result = SendMessage(hWnd, Win32.WM.WM_COPYDATA, IntPtr.Zero, data);
            return result != 0 ? true : false;
        }

        /// <summary>
        ///
        /// </summary>
        private void InitializeQuesterSettings()
        {
            ProjectFileQuester.Extension =
                this.Traits.IntermediateOutputTraits.SoundProjectFileExtension;
            SoundSetFileQuester.Extension =
                this.Traits.IntermediateOutputTraits.SoundSetFileExtension;
            BankFileQuester.Extension = this.Traits.IntermediateOutputTraits.BankFileExtension;
            SeqFileQuester.ExtensionText =
                this.Traits.IntermediateOutputTraits.TextSequenceSoundFileExtension;
        }

#if true
        /// <summary>
        ///
        /// </summary>
        private void OnThreadException(object sender, System.Threading.ThreadExceptionEventArgs e)
        {
            ProcessUnhandledException(e.Exception.StackTrace);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnUnhandledException(object sender, System.UnhandledExceptionEventArgs e)
        {
            Exception exception = e.ExceptionObject as Exception;
            ProcessUnhandledException(exception.StackTrace);
        }

        /// <summary>
        ///
        /// </summary>
        private void ProcessUnhandledException(string message)
        {
            if (UnhandleExceptionHelper.ShowDialog != false)
            {
                UnhandledExceptionDialog dialog = new UnhandledExceptionDialog();
                dialog.Message = message;
                dialog.ShowDialog();

                Environment.Exit(-1);
            }
        }
#endif

        /// <summary>
        /// メインウィンドウがロードされると発生します。
        /// </summary>
        private void OnMainWindowLoad(object sender, EventArgs e)
        {
            this.OnRun(new EventArgs());
        }

        /// <summary>
        /// メインウィンドウが初めて表示される時に発生します。
        /// </summary>
        private void OnMainWindowShown(object sender, EventArgs e)
        {
            this.OnShown(new EventArgs());

            //
            if (reservedActivateProjectFilePath != null &&
                reservedActivateItemName != null &&
                reservedActivateCommand != null &&
                reservedActivateValue1 != null &&
                reservedActivateValue2 != null &&
                reservedActivateValue3 != null &&
                reservedActivateValue4 != null &&
                reservedActivateValue5 != null)
            {
                IntPtr hWnd = Process.GetCurrentProcess().MainWindowHandle;
                SendItemActivateMessage(hWnd, reservedActivateProjectFilePath,
                                         reservedActivateCommand,
                                         reservedActivateItemName,
                                         reservedActivateValue1,
                                         reservedActivateValue2,
                                         reservedActivateValue3,
                                         reservedActivateValue4,
                                         reservedActivateValue5);

                reservedActivateProjectFilePath = null;
                reservedActivateItemName = null;
                reservedActivateCommand = null;
                reservedActivateValue1 = null;
                reservedActivateValue2 = null;
                reservedActivateValue3 = null;
                reservedActivateValue4 = null;
                reservedActivateValue5 = null;
            }
        }

        /// <summary>
        /// メインウィンドウがアクティブになると発生します。
        /// </summary>
        /// <param name="sender">メインウィンドウ。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnMainWindowActivated(object sender, EventArgs e)
        {
            this.RaiseAppActivated();
        }

        /// <summary>
        /// メインウィンドウが非アクティブになると発生します。
        /// </summary>
        /// <param name="sender">メインウィンドウ。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnMainWindowDeactivated(object sender, EventArgs e)
        {
            this.RaiseAppDeactivated();
        }
    }
}
