﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;

    public class QueryAndOutputTemplateTextSequence : CommandHandler
    {
        private OutputTemplateTextSequence outputSeq;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryParameter">パラメータの問い合わせ先。</param>
        public QueryAndOutputTemplateTextSequence(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
            this.outputSeq = new OutputTemplateTextSequence();
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return ProjectCommands.OutputTemplateTextSequence.ID; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command,
                                                             IQueryCommandParameter parameters)
        {
            if (null == GetTargetProjectService(parameters))
            {
                return CommandStatus.SupportedAndVisible;
            }

            Sound[] sounds = GetTargetSounds(parameters);
            if (sounds.Length == 0) { return CommandStatus.SupportedAndVisible; }

            foreach (Sound sound in sounds)
            {
                if (sound is SequenceSoundBase)
                {
                    SequenceSoundBase seqSound = sound as SequenceSoundBase;
                    if (seqSound.FileType == SequenceSoundFileType.Smf)
                    {
                        return CommandStatus.SupportedAndVisible;
                    }
                }
                else
                {
                    return CommandStatus.SupportedAndVisible;
                }
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            Sound[] targets = GetTargetSounds(parameters);
            if (0 == targets.Length) { return false; }

            try
            {
                this.outputSeq.Run(targets);
            }
            catch (Exception e)
            {
                FormsApplication.Instance.UIService.ShowMessageBox(e.Message);
                return false;
            }

            return true;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        private SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                as SoundProjectService;
        }

        /// <summary>
        /// 対象サウンドの一覧を取得します。
        /// </summary>
        private Sound[] GetTargetSounds(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            return (from Component component in
                    (parameters.GetParameter(CommandParameterNames.TargetComponents)
                     as IEnumerable<Component>)
                    where component is Sound
                    select component as Sound).ToArray();
        }



        private class OutputTemplateTextSequence
        {
            /// <summary>
            /// テンプレートを出力します。
            /// </summary>
            public void Run(Sound[] sounds)
            {
                if (null == sounds) { throw new ArgumentNullException("sounds"); }

                FormsApplication.Instance.ExecuteCommand(PreviewCommands.StopAll);
                FormsApplication.Instance.ExecuteCommand(ToolCommands.StopAllSounds);

                List<SequenceSoundBase> seqs = new List<SequenceSoundBase>();
                foreach (SequenceSoundBase seq in sounds)
                {
                    if (this.IsContainsStartPosition(seq) == false)
                    {
                        seqs.Add(seq);
                    }
                }

                if (0 < seqs.Count)
                {
                    Dictionary<string, List<string>> dic = new Dictionary<string, List<string>>();
                    foreach (SequenceSoundBase seq in seqs)
                    {
                        if (dic.ContainsKey(seq.FilePath) == false)
                        {
                            dic[seq.FilePath] = new List<string>();
                        }
                        dic[seq.FilePath].Add(seq.StartPosition);
                    }
                    StringBuilder message = new StringBuilder();
                    foreach (string filePath in dic.Keys)
                    {
                        message.AppendLine(filePath);
                        foreach (string startPosition in dic[filePath])
                        {
                            message.AppendLine(" \"" + startPosition + "\"");
                        }
                        message.AppendLine(string.Empty);
                    }

                    TextDisplayMessageBox messageBox =
                        new TextDisplayMessageBox(MessageResource.Message_ConfirmAddLabels,
                                                  message.ToString(),
                                                  TextDisplayMessageBoxStyle.YesAndNoButton);

                    if (messageBox.ShowDialog() == DialogResult.Yes)
                    {
                        bool result =
                            SaveHelper.Save(() =>
                            {
                                foreach (SequenceSoundBase seq in seqs)
                                {
                                    this.OutputTemplateTextSequenceStartPosition(seq);
                                }
                            }, seqs.Select(s => s.FilePath).Distinct().ToArray());

                        if (result == true)
                        {
                            MessageBox.Show(MessageResource.Message_ConfirmAddedLabels);
                        }
                    }
                }
                else
                {
                    MessageBox.Show(MessageResource.Message_ConfirmNoLabels);
                }
            }

            private void OutputTemplateTextSequenceStartPosition(SequenceSoundBase sequenceSound)
            {
                using (StreamWriter w = new StreamWriter(sequenceSound.FilePath, true))
                {
                    w.WriteLine(string.Empty);
                    w.WriteLine(string.Empty);
                    w.WriteLine("; ");
                    w.WriteLine(sequenceSound.StartPosition + ":");
                    w.WriteLine("fin");
                }
            }

            private bool IsContainsStartPosition(SequenceSoundBase sequenceSound)
            {
                return GetStartPositionLabels(sequenceSound.FilePath).Contains(sequenceSound.StartPosition);
            }

            private string[] GetStartPositionLabels(string filePath)
            {
                SequenceSoundFileParser parser = null;

                try
                {
                    parser = ApplicationBase.Instance.FileParserStore.Parse<SequenceSoundTextParser>(filePath);
                }
                catch (Exception e)
                {
                    LexerException le = e as LexerException;
                    if (le != null)
                    {
                        string message =
                            MessageResource.Message_IllegalFileFormat + "\r\n\r\n" +
                            MessageResource.Message_FileName + " " +
                            Path.GetFileName(le.FilePath) + "\r\n" +
                            MessageResource.Message_LineNo + " " + le.LineNumber + "\r\n" +
                            MessageResource.Message_IllegalStrings + " " + le.LastLine;

                        MessageBox.Show(message);
                    }
                }

                return parser.Labels.Select(p => p.Name).ToArray();
            }
        }
    }
}
