﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// "削除"ハンドラ
    /// </summary>
    public class ProjectPanelDeleteHandler : DeleteHandler
    {
        ///--------------------------------
        /// <summary>
        /// </summary>
        public ProjectPanelDeleteHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            Component[] components = GetTargetComponents(parameters);

            var soundProjectDocument = GetTargetDocument(parameters);
            if (soundProjectDocument == null ||
                soundProjectDocument.IsReadOnly == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            if (null == components) { return CommandStatus.SupportedAndVisible; }

            if (components.Length <= 0)
            {
                return CommandStatus.SupportedAndVisible;
            }

            var result = components
                .All(c => c is SoundSet || c is FolderComponent);
            return result == true ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            var document = GetTargetDocument(parameters) as SoundProjectDocument;
            var components = GetTargetComponents(parameters);

            try
            {
                var targetSoundSets = new List<SoundSet>(components.OfType<SoundSet>());
                var targetFolders = components
                    .OfType<FolderComponent>();

                if (targetFolders.Count() > 0)
                {
                    foreach (var folder in targetFolders)
                    {
                        targetSoundSets.AddRange(GetSoundSets(folder));
                    }

                    if (targetSoundSets.Count > 0)
                    {
                        var sb = new StringBuilder();
                        targetSoundSets.ForEach(s => sb.AppendLine(s.Name));

                        var messageBox = new TextDisplayMessageBox(MessageResource.Message_ContainsSoundSet,
                            sb.ToString(), TextDisplayMessageBoxStyle.YesAndNoButton);
                        if (messageBox.ShowDialog() != DialogResult.Yes)
                        {
                            return false;
                        }
                    }
                }

                foreach (var soundSet in targetSoundSets)
                {
                    DeleteSoundSet(soundSet);
                }

                foreach (var folder in targetFolders)
                {
                    var operation = new RemoveComponentOperation(folder);
                    operation.Execute();
                }

                document.SetDirty();
                return true;
            }
            catch
            {
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private FormsApplication Application
        {
            get { return FormsApplication.Instance; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected MainWindow MainWindow
        {
            get { return Application.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSetDocument GetSoundSetDocument(SoundSet soundSet)
        {
            SoundProjectService projectService = null;

            projectService = Application.ProjectService;
            foreach (SoundSetDocument soundSetDocument in projectService.SoundSetDocuments)
            {
                if (soundSetDocument.SoundSet == soundSet)
                {
                    return soundSetDocument;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private DocumentReference GetSoundSetDocumentReference(SoundSet soundSet)
        {
            SoundProjectService projectService = null;

            projectService = Application.ProjectService;
            foreach (SoundSetDocument soundSetDocument in projectService.SoundSetDocuments)
            {
                if (soundSetDocument.SoundSet == soundSet)
                {
                    return Application.DocumentService.OpenDocument(soundSetDocument.Resource);
                }
            }
            return null;
        }

        private IEnumerable<SoundSet> GetSoundSets(Component component)
        {
            if (component is SoundSet)
            {
                yield return component as SoundSet;
            }

            foreach (var childComponent in component.Children)
            {
                var soundSets = GetSoundSets(childComponent);
                foreach (var soundSet in soundSets)
                {
                    yield return soundSet;
                }
            }
        }

        private Operation DeleteSoundSet(SoundSet soundSet)
        {
            using (var docRef = GetSoundSetDocumentReference(soundSet))
            {
                var soundSetDocument = docRef.Document as SoundSetDocument;
                var key = soundSetDocument.Resource.Key;
                var operation = new RemoveSoundSetOperation(Application.ProjectService, key);
                operation.Execute();
                return operation;
            }
        }
    }
}
