﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Profile
{
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileManagement;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sre;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NW4R.ProtocolSound;

    /// <summary>
    ///
    /// </summary>
    public class ParameterValueChangedProxyService : IDisposable
    {
        private SoundProjectService projectService = null;
        private Dictionary<IParameterDictionary, Component> targets = new Dictionary<IParameterDictionary, Component>();
        private Dictionary<object, Info> listeners = new Dictionary<object, Info>();

        [Flags]
        public enum Kinds : uint
        {
            Unknown = 0x00000000,
            StreamSound = 0x00000001,
            WaveSound = 0x00000002,
            SequenceSound = 0x00000004,
            SoundSetBank = 0x00000008,

            All = 0xffffffff,
        }

        ///
        public ParameterValueChangedProxyService(SoundProjectService projectService)
        {
            this.projectService = projectService;

            this.projectService.Opened += OnProjectOpened;
            this.projectService.Closing += OnProjectClosing;
            this.projectService.Closed += OnProjectClosed;

            this.projectService.ComponentsAdded += OnComponentsAdded;
            this.projectService.ComponentsRemoved += OnComponentsRemoved;
        }

        ///
        void IDisposable.Dispose()
        {
            this.projectService.Opened -= OnProjectOpened;
            this.projectService.Closing -= OnProjectClosing;
            this.projectService.Closed -= OnProjectClosed;

            this.projectService.ComponentsAdded -= OnComponentsAdded;
            this.projectService.ComponentsRemoved -= OnComponentsRemoved;
        }

        ///
        public void Add(object listener, ParameterEventHandler changedhandler, ComponentEventHandler addedHandler, ComponentEventHandler removedHandler, Kinds kind)
        {
            Info info = null;
            if (this.listeners.TryGetValue(listener, out info) == false)
            {
                info = new Info();
                this.listeners.Add(listener, info);
            }

            info.ChangedHandler = changedhandler;
            info.AddedHandler = addedHandler;
            info.RemovedHandler = removedHandler;
            info.Kind = kind;
        }

        ///
        public void Remove(object listener)
        {
            if (this.listeners.ContainsKey(listener) == true)
            {
                this.listeners.Remove(listener);
            }
        }

        ///
        private void OnProjectOpened(object sender, EventArgs e)
        {
        }

        ///
        private void OnProjectClosing(object sender, EventArgs e)
        {
        }

        ///
        private void OnProjectClosed(object sender, EventArgs e)
        {
            this.targets.Clear();
        }

        ///
        private void OnComponentsAdded(object sender, ComponentEventArgs e)
        {
            List<Component> list = new List<Component>();
            foreach (Component component in e.Components)
            {
                CollectComponent(list, component);
            }

            foreach (Component component in list)
            {
                IParameterDictionary parameters = component.Parameters;

                if (this.targets.ContainsKey(parameters) == false)
                {
                    parameters.ParameterValueChanged += OnParameterValueChanged;
                    this.targets.Add(parameters, component);

                    Execute(component,
                             delegate (Info info)
                                 {
                                     if (info.AddedHandler != null)
                                     {
                                         info.AddedHandler(component, e);
                                     }
                                 });
                }
            }
        }

        ///
        private void OnComponentsRemoved(object sender, ComponentEventArgs e)
        {
            List<Component> list = new List<Component>();
            foreach (Component component in e.Components)
            {
                CollectComponent(list, component);
            }

            foreach (Component component in list)
            {
                IParameterDictionary parameters = component.Parameters;

                if (this.targets.ContainsKey(parameters) != false)
                {
                    parameters.ParameterValueChanged -= OnParameterValueChanged;
                    this.targets.Remove(parameters);

                    Execute(component,
                             delegate (Info info)
                                 {
                                     if (info.RemovedHandler != null)
                                     {
                                         info.RemovedHandler(component, e);
                                     }
                                 });
                }
            }
        }

        ///
        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            IParameterDictionary parameters = sender as IParameterDictionary;
            Component component = this.targets[parameters];

            Execute(component,
                     delegate (Info info)
                         {
                             if (info.ChangedHandler != null)
                             {
                                 info.ChangedHandler(component, e);
                             }
                         });
        }

        ///
        private delegate void ExecuteHandler(Info info);

        private void Execute(Component component, ExecuteHandler executor)
        {
            Kinds kind = GetKind(component);
            if (kind == Kinds.Unknown)
            {
                return;
            }

            foreach (Info info in this.listeners.Values)
            {
                if (info.Kind.HasFlag(kind) == true)
                {
                    executor(info);
                }
            }
        }

        ///
        private bool IsExclusionComponent(Component component)
        {
            if (component is SoundSet ||
                component is StreamSoundPack ||
                component is WaveSoundSetPack ||
                component is SequenceSoundSetPack ||
                component is SequenceSoundPack ||
                component is SoundSetBankPack ||
                component is PlayerPack ||
                component is WaveArchivePack ||
                component is GroupPack)
            {
                return true;
            }
            return false;
        }

        private void CollectComponent(List<Component> list, Component component)
        {
            if (IsExclusionComponent(component) == false)
            {
                list.Add(component);
            }

            foreach (Component child in component.Children)
            {
                CollectComponent(list, child);
            }
        }

        ///
        private Kinds GetKind(Component component)
        {
            if (component is StreamSoundBase)
            {
                return Kinds.StreamSound;
            }
            if (component is WaveSoundBase)
            {
                return Kinds.WaveSound;
            }
            if (component is SequenceSoundBase)
            {
                return Kinds.SequenceSound;
            }
            if (component is SoundSetBankBase)
            {
                return Kinds.SoundSetBank;
            }
            return Kinds.Unknown;
        }

        /// <summary>
        ///
        /// </summary>
        private class Info
        {
            public ParameterEventHandler ChangedHandler { get; set; }
            public ComponentEventHandler AddedHandler { get; set; }
            public ComponentEventHandler RemovedHandler { get; set; }
            public Kinds Kind { get; set; }
        }
    }
}
