﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography;

namespace NintendoWare.SoundMaker.Preview.Service
{
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sre;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NW4R.ProtocolSound;


    /// <summary>
    ///
    /// </summary>
    public static class SoundEditInfoCreator
    {
        private static readonly HashAlgorithm hashAlgorithm = new SHA1CryptoServiceProvider();

        // HACK : CTR と Cafe でクラス名が競合するため、とりあえず同じ文字列を定義している
        private const string StreamSoundTrackChannelCountParameterName = "StreamSoundTrack_ChannelCount";
        ///
        public static SendsGetterHandler SendsGetter { get; set; }
        public delegate SendsParam SendsGetterHandler(Sends sends);

        /// <summary>
        /// スタティックコンストラクタ
        /// </summary>
        static SoundEditInfoCreator()
        {
            SendsGetter = delegate (Sends sends)
                {
                    SendsParam sendsParam = new SendsParam();

                    sendsParam.mainSend = (byte)sends.MainSend;
                    sendsParam.fxSends[0] = (byte)sends.AuxASend;
                    sendsParam.fxSends[1] = (byte)sends.AuxBSend;
                    return sendsParam;
                };
        }

        /// <summary>
        ///
        /// </summary>
        public static SoundEditInfoPack Create(Component component)
        {
            SoundEditInfo soundEditInfo = null;
            SndEditPacketHeader.ResultTypes resultType = SndEditPacketHeader.ResultTypes.TRUE;

            if (component is StreamSoundBase)
            {
                // ストリームサウンド
                soundEditInfo = GetInfo(component as StreamSoundBase, ref resultType);
            }
            else if (component is WaveSoundBase)
            {
                // ウェーブサウンド
                soundEditInfo = GetInfo(component as WaveSoundBase, ref resultType);
            }
            else if (component is SequenceSoundBase)
            {
                // シーケンスサウンド
                soundEditInfo = GetInfo(component as SequenceSoundBase, ref resultType);
            }
            else if (component is SoundSetBankBase)
            {
                // バンク
                soundEditInfo = GetInfo(component as SoundSetBankBase, ref resultType);
            }
            else
            {
                return null;
            }

            CreatedSoundEditInfo(soundEditInfo);

            SoundEditInfoPack pack = new SoundEditInfoPack(soundEditInfo, resultType);
            return pack;
        }

        /// <summary>
        ///
        /// </summary>
        private static Func<IParameterValue, bool> OptionalFilter
        {
            get
            {
                string preprocessedTag = ApplicationBase.Instance.ProjectService.GetPreprocessedTag();
                Func<IParameterValue, bool> optionalFilter = null;
                if (string.IsNullOrEmpty(preprocessedTag) == false)
                {
                    optionalFilter =
                        parameterValue => string.Equals(preprocessedTag, parameterValue.Value);
                }

                return optionalFilter;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private static byte[] GetComponentHashCode(Component component)
        {
            byte[] buff = new byte[32];
            component.GetComponentHashCode
                (hashAlgorithm, OptionalFilter).Value.CopyTo(buff, 0);
            return buff;
        }

        /// <summary>
        ///
        /// </summary>
        private static byte[] GetComponentSoundArchiveHashCode(Component component)
        {
            byte[] buff = new byte[32];
            component.GetComponentSoundArchiveHashCode
                (hashAlgorithm, OptionalFilter).Value.CopyTo(buff, 0);
            return buff;
        }

        /// <summary>
        /// ストリームサウンド
        /// </summary>
        private static StreamSoundEditInfo GetInfo(StreamSoundBase streamSound, ref SndEditPacketHeader.ResultTypes resultType)
        {
            GroupBase[] groups = ApplicationBase.Instance.ProjectService.Groups;
            bool isPrefetchFileAvailable = GroupUtility.ContainsGroup(groups, streamSound);

            StreamSoundEditInfo info = new StreamSoundEditInfo();

            if (SetSoundEditInfo(info, streamSound) == false)
            {
                resultType = SndEditPacketHeader.ResultTypes.ITEM_NOT_FOUND;
                return null;
            }

            info.soundBasicParam.panMode = (byte)streamSound.PanMode;
            info.soundBasicParam.panCurve = (byte)streamSound.PanCurve;

            int trackIndex = 0;
            int channelIndex = 0;

            foreach (StreamSoundTrackBase track in streamSound.Children.Cast<StreamSoundTrackBase>())
            {
                if (track.IsEnabled == false)
                {
                    continue;
                }

                var trackParam = info.streamSoundParam.trackParams[trackIndex];

                if (track.PreviewMute == true)
                {
                    trackParam.volume = 0;
                }
                else
                {
                    trackParam.volume = (byte)track.Volume;
                }
                trackParam.pan = (byte)track.Pan;
                trackParam.span = (byte)track.SurroundPan;
                trackParam.lpf = (byte)track.LPF;
                trackParam.biquadType = BiquadTypeConverter.ConvertToByte(track.BiquadType);
                trackParam.biquadValue = (byte)track.Biquad;
                trackParam.flags = (byte)(track.FrontBypass == true ? 1 : 0);
                info.streamSoundParam.trackParams[trackIndex].sends = SendsGetter(track.Sends);

                uint channelCount = (uint)track.Parameters[StreamSoundTrackChannelCountParameterName].Value;
                trackParam.channelCount = (byte)channelCount;

                for (int trackChannelIndex = 0; trackChannelIndex < trackParam.channelCount; ++trackChannelIndex)
                {
                    trackParam.channelIndexes[trackChannelIndex] = (byte)channelIndex;
                    ++channelIndex;
                }

                ++trackIndex;
            }

            info.streamSoundParam.trackAllocationFlags = streamSound.TrackAllocationFlags();
            info.streamSoundParam.totalChannelCount = streamSound.TotalChannelCount();
            info.streamSoundParam.filePath = streamSound.BfstmFilePath();
            info.streamSoundParam.pitch = streamSound.Pitch;
            info.streamSoundParam.sends = SendsGetter(streamSound.Sends);
            info.streamSoundParam.streamFileType = (Byte)streamSound.Parameters[ParameterNames.StreamSound.ContainerType].Value;

            if (info.streamSoundParam.filePath == String.Empty)
            {
                resultType = SndEditPacketHeader.ResultTypes.ITEM_PREPAREING;
                return null;
            }

            if (isPrefetchFileAvailable == true)
            {
                info.streamSoundParam.prefetchFilePath = streamSound.BfstpFilePath();
            }
            else
            {
                info.streamSoundParam.prefetchFilePath = string.Empty;
            }
            info.streamSoundParam.bfstmFileHashCode = GetComponentHashCode(streamSound);

            info.streamSoundParam2.isLoop = streamSound.GetHasLoop();
            info.streamSoundParam2.loopStartFrame = streamSound.GetLoopStartFrame();
            info.streamSoundParam2.loopEndFrame = streamSound.GetLoopEndFrame();

            resultType = SndEditPacketHeader.ResultTypes.TRUE;
            return info;
        }

        /// <summary>
        /// ウェーブサウンド
        /// </summary>
        private static WaveSoundEditInfo GetInfo(WaveSoundBase waveSound, ref SndEditPacketHeader.ResultTypes resultType)
        {
            WaveSoundEditInfo info = new WaveSoundEditInfo();

            if (SetSoundEditInfo(info, waveSound) == false)
            {
                resultType = SndEditPacketHeader.ResultTypes.ITEM_NOT_FOUND;
                return null;
            }

            WaveSoundSetBase waveSoundSet = waveSound.Parent as WaveSoundSetBase;

            info.soundBasicParam.panMode = (byte)waveSound.PanMode;
            info.soundBasicParam.panCurve = (byte)waveSound.PanCurve;
            info.soundBasicParam.flags = (byte)(waveSound.FrontBypass == true ? 1 : 0);

            info.waveSoundParam.channelPriority = (byte)waveSound.ChannelPriority;
            info.waveSoundParam.isReleasePriorityFix = waveSound.ReleasePriorityFixed;
            info.waveSoundParam.waveIndex = waveSound.Index();
            info.waveSoundParam.bfwsdFilePath = waveSound.BfwsdFilePath();
            info.waveSoundParam.bfwarFilePath = waveSound.BfwarFilePath();

            info.waveSoundParam.bfwsdFileHashCode = GetComponentHashCode(waveSoundSet);
            info.waveSoundParam.bfwarFileHashCode = GetComponentSoundArchiveHashCode(waveSoundSet);

            if (info.waveSoundParam.bfwsdFilePath == String.Empty ||
                info.waveSoundParam.bfwarFilePath == String.Empty)
            {
                resultType = SndEditPacketHeader.ResultTypes.ITEM_PREPAREING;
                return null;
            }

            resultType = SndEditPacketHeader.ResultTypes.TRUE;
            return info;
        }

        /// <summary>
        /// シーケンスサウンド
        /// </summary>
        private static SequenceSoundEditInfo GetInfo(SequenceSoundBase sequenceSound, ref SndEditPacketHeader.ResultTypes resultType)
        {
            SequenceSoundEditInfo info = new SequenceSoundEditInfo();

            if (SetSoundEditInfo(info, sequenceSound) == false)
            {
                resultType = SndEditPacketHeader.ResultTypes.ITEM_NOT_FOUND;
                return null;
            }

            //info.soundBasicParam.panMode  = (byte)sequenceSound.PanMode;
            //info.soundBasicParam.panCurve = (byte)sequenceSound.PanCurve;
            info.soundBasicParam.flags = (byte)(sequenceSound.FrontBypass == true ? 1 : 0);

            info.sequenceSoundParam.startOffset = sequenceSound.GetConversionStartOffset();
            info.sequenceSoundParam.channelPriority = (byte)sequenceSound.ChannelPriority;
            info.sequenceSoundParam.isReleasePriorityFix = sequenceSound.ReleasePriorityFixed;
            info.sequenceSoundParam.trackAllocationFlags = sequenceSound.AllocateTrackFlags();

            //
            info.sequenceSoundParam.bfseqFilePath = sequenceSound.BfseqFilePath();
            info.sequenceSoundParam.bankNames = sequenceSound.BankNames();
            info.sequenceSoundParam.bfseqFileHashCode = GetComponentHashCode(sequenceSound);

            if (info.sequenceSoundParam.bfseqFilePath == String.Empty)
            {
                resultType = SndEditPacketHeader.ResultTypes.ITEM_PREPAREING;
                return null;
            }

            resultType = SndEditPacketHeader.ResultTypes.TRUE;
            return info;
        }

        /// <summary>
        /// バンク
        /// </summary>
        private static BankEditInfo GetInfo(SoundSetBankBase bank, ref SndEditPacketHeader.ResultTypes resultType)
        {
            BankEditInfo info = new BankEditInfo();

            info.bankParam.bfbnkFilePath = bank.BfbnkFilePath();
            info.bankParam.bfwarFilePath = bank.BfwarFilePath();
            info.bankParam.bfbnkFileHashCode = GetComponentHashCode(bank);
            info.bankParam.bfwarFileHashCode = GetComponentSoundArchiveHashCode(bank);

            if (info.bankParam.bfbnkFilePath == String.Empty ||
                info.bankParam.bfwarFilePath == String.Empty)
            {
                resultType = SndEditPacketHeader.ResultTypes.ITEM_PREPAREING;
                return null;
            }

            resultType = SndEditPacketHeader.ResultTypes.TRUE;
            return info;
        }

        /// <summary>
        /// 基本情報の設定を行います。
        /// </summary>
        private static bool SetSoundEditInfo(SoundEditInfo info, Sound sound)
        {
            if (sound.TargetPlayer == null)
            {
                return false;
            }

            //
            info.soundBasicParam.volume = (byte)sound.Volume;
            info.soundBasicParam.playerPriority = (byte)sound.PlayerPriority;
            info.soundBasicParam.actorPlayerID = (byte)sound.ActorPlayer;
            info.soundBasicParam.singlePlayType = (byte)sound.SinglePlayType;
            info.soundBasicParam.singlePlayEffectiveDuration = (ushort)sound.SinglePlayEffectiveDuration;
            info.soundBasicParam.userParam[0] = (uint)sound.UserParameter;
            info.soundBasicParam.userParam[1] = (uint)sound.UserParameter1;
            info.soundBasicParam.userParam[2] = (uint)sound.UserParameter2;
            info.soundBasicParam.userParam[3] = (uint)sound.UserParameter3;

            Sound3D sound3D = sound.Sound3D;
            info.sound3DParam.flags = (uint)((sound3D.Enable3DVolume ? (1 << 0) : 0) |
                                             (sound3D.Enable3DPriority ? (1 << 1) : 0) |
                                             (sound3D.Enable3DPan ? (1 << 2) : 0) |
                                             (sound3D.Enable3DSurroundPan ? (1 << 3) : 0) |
                                             (sound3D.Enable3DFilter ? (1 << 4) : 0));
            info.sound3DParam.decayRatio = sound3D.DecayRatio3D;
            info.sound3DParam.decayCurve = ConvertDecayCurve3DToByte(sound3D.DecayCurve3D);
            info.sound3DParam.dopplerFactor = (byte)sound3D.DopplerFactor3D;

            info.playerName = sound.TargetPlayer.Name;

            return true;
        }

        private static byte ConvertDecayCurve3DToByte(DecayCurve3D value)
        {
            switch (value)
            {
                case DecayCurve3D.Log:
                    return 1;
                case DecayCurve3D.Linear:
                    return 2;
            }

            return 1;
        }

        /// <summary>
        ///
        /// </summary>
        private static void CreatedSoundEditInfo(SoundEditInfo info)
        {
            ApplicationBase.Instance.RealtimeEditService.CreatedSoundEditInfo(info);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public partial class RealtimeEditService
    {
        private Dictionary<string, SoundEditInfoPack> soundEditInfoPackDictionary = new Dictionary<string, SoundEditInfoPack>();

        private object soundEditInfoLockObject = new Object();

        /// <summary>
        ///
        /// </summary>
        public SoundEditInfoPack PopSoundEditInfoPack(string name)
        {
            lock (this.soundEditInfoLockObject)
            {
                if (this.soundEditInfoPackDictionary.ContainsKey(name) == false)
                {
                    return null;
                }

                SoundEditInfoPack pack = this.soundEditInfoPackDictionary[name];
                this.soundEditInfoPackDictionary.Remove(name);
                return pack;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsTransferring(Component component)
        {
            lock (this.soundEditInfoLockObject)
            {
                return this.soundEditInfoPackDictionary.ContainsKey(component.Name);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsTransferring(Component[] components)
        {
            lock (this.soundEditInfoLockObject)
            {
                foreach (Component component in components)
                {
                    if (this.soundEditInfoPackDictionary.ContainsKey(component.Name) == true)
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ClearSoundEditInfo()
        {
            lock (this.soundEditInfoLockObject)
            {
                this.soundEditInfoPackDictionary.Clear();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ClearSoundEditInfo(string name)
        {
            lock (this.soundEditInfoLockObject)
            {
                if (this.soundEditInfoPackDictionary.ContainsKey(name) == true)
                {
                    this.soundEditInfoPackDictionary.Remove(name);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void PushSoundEditInfo(Component component)
        {
            lock (this.soundEditInfoLockObject)
            {
                SoundEditInfoPack pack = SoundEditInfoCreator.Create(component);
                if (pack != null)
                {
                    if (this.soundEditInfoPackDictionary.ContainsKey(component.Name) != false)
                    {
                        this.soundEditInfoPackDictionary[component.Name] = pack;
                    }
                    else
                    {
                        this.soundEditInfoPackDictionary.Add(component.Name, pack);
                    }
                }
            }
        }
    }
}
