﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Sync
{
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sre;
    using NintendoWare.SoundMaker.Preview.Service;

    using NW4R.ProtocolSound;

    /// <summary>
    /// Syncパケット
    /// <para>
    /// ビューアからツールに送られるパケットのベースクラスです。
    /// </para>
    /// </summary>
    public abstract class SyncPacket : CommPacket
    {
        private static ISyncPacketFactoryCollection _factories = new ISyncPacketFactoryCollection();

        /// <summary>
        /// Syncパケットの作成
        /// </summary>
        public static SyncPacket Create(CommPacketHeader header)
        {
            // 分類分けを行う
            SyncPacket packet = null;

            foreach (ISyncPacketFactory factory in _factories)
            {
                if (factory.Category != (header.PacketType & Constants.SRE_CATEGORY_MASK))
                {
                    continue;
                }

                packet = factory.CreatePacket(header);
                if (null != packet)
                {
                    return packet;
                }
            }

            switch (header.PacketType & Constants.SRE_CATEGORY_MASK)
            {
                case Constants.CATEGORY_TEST:
                    //packet = SyncTestPacket.CreateTestPacket( header.PacketType);
                    break;

                case Constants.SRE_CATEGORY_SYSTEM:
                    switch (header.PacketType & Constants.SRE_MESSAGE_MASK)
                    {
                        case Constants.SRE_SYNC:
                            packet = new SreSyncPacket();
                            break;
                    }
                    break;
            }

            return packet;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected SyncPacket()
        {
            Debug.Assert(!Error, "Error is true");
        }

        #region ** プロパティ

        public static ISyncPacketFactoryCollection Factories
        {
            get { return _factories; }
        }

        /// <summary>
        /// 返信フラグ
        /// </summary>
        public override bool Reply
        {
            get
            {
                return ((PacketType & Constants.SRE_MESSAGE_REPLY_FLAG) != 0);
            }
        }

        /// <summary>
        /// エラーフラグ
        /// </summary>
        public override bool Error
        {
            get
            {
                return false;
            }
        }

        #endregion
    }

    /// <summary>
    ///
    /// </summary>
    public class SyncPacketHeader : SndEditPacketHeader
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SyncPacketHeader(ProtocolSoundReader reader) : base(reader)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SyncPacketHeader(uint packetType, uint size) : base(packetType, size)
        {
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SreSyncPacket : SyncPacket
    {
        private static int timeOutMillisecond = 5000;

        private static DateTime lastReceivedTime;

        /// <summary>
        ///
        /// </summary>
        public static void ResetTimeOut(int timeout = 5)
        {
            timeOutMillisecond = timeout * 1000; // second -> milli second
            UpdateTimeOut();
        }

        /// <summary>
        ///
        /// </summary>
        public static bool DoTimeOut
        {
            get
            {
                TimeSpan span = DateTime.Now.Subtract(lastReceivedTime);
                return span.TotalMilliseconds >= timeOutMillisecond ? true : false;
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SreSyncPacket()
        {
        }

        /// <summary>
        /// ヘッダの作成
        /// </summary>
        public override CommPacketHeader CreateHeader()
        {
            return new SyncPacketHeader(PacketType | Constants.SRE_CATEGORY_SYSTEM, Size);
        }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.SRE_SYNC_REPLY; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)SreSyncReply.StructSize;
            }
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
#if false
            SreSync body = new SreSync();
            body.Read( reader);
#endif
            UpdateTimeOut();
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            // パラメータを出力する
            SreSyncReply body = new SreSyncReply();
            body.isAccept = true;
            body.Write(writer);

            return header;
        }

        /// <summary>
        ///
        /// </summary>
        private static void UpdateTimeOut()
        {
            lastReceivedTime = DateTime.Now;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SoundPacketFactory : ISyncPacketFactory
    {
        uint ISyncPacketFactory.Category
        {
            get { return GetCategory(); }
        }

        SyncPacket ISyncPacketFactory.CreatePacket(CommPacketHeader header)
        {
            return InternalCreatePacket(header);
        }

        protected virtual uint GetCategory()
        {
            return Constants.SRE_CATEGORY_SYSTEM;
        }

        protected virtual SyncPacket InternalCreatePacket(CommPacketHeader header)
        {
            return null;
        }
    }


    #region ** パケットファクトリー

    /// <summary>
    ///
    /// </summary>
    public interface ISyncPacketFactory
    {
        #region ** プロパティ

        uint Category { get; }

        #endregion

        #region ** メソッド

        SyncPacket CreatePacket(CommPacketHeader header);

        #endregion
    }

    #region ** コレクション

    public class ISyncPacketFactoryCollection : Collection<ISyncPacketFactory> { }

    #endregion

    #endregion

#if true

    /// <summary>
    ///
    /// </summary>
    public class SyncPacketFactory : ISyncPacketFactory
    {
        uint ISyncPacketFactory.Category
        {
            get { return GetCategory(); }
        }

        SyncPacket ISyncPacketFactory.CreatePacket(CommPacketHeader header)
        {
            return InternalCreatePacket(header);
        }

        protected virtual uint GetCategory()
        {
            return Constants.SRE_CATEGORY_SYSTEM;
        }

        protected virtual SyncPacket InternalCreatePacket(CommPacketHeader header)
        {
            switch (header.PacketType & Constants.SRE_MESSAGE_MASK)
            {
                case Constants.SRE_SYNC:
                    return new SndEditSyncPacket();

                default:
                    break;
            }

            return null;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SndEditSyncPacket : SreSyncPacket
    {
        /// <summary>
        // SoundPlayerとの接続を要求するのか？
        // これを trueにすることにより、Tool、Viewer、Pingの接続を開始します。
        /// </summary>
        private static bool queryConnectToSoundPlayer = false;

        public static bool QueryConnectToSoundPlayer
        {
            get
            {
                return queryConnectToSoundPlayer;
            }
            set
            {
                queryConnectToSoundPlayer = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            base.Read(reader, header);

            SreSync body = new SreSync();
            body.Read(reader);

            CommManager manager = CommManager.Instance;
            if (manager.IsDisconnecting != false ||
                manager.IsDisconnectingSndEdit != false)
            {
                return;
            }

            if (manager.IsConnectedSndEdit == false)
            {
                manager.SetStateSndEdit(ConnectionState.Connected);
            }

            if (manager.IsConnected == false)
            {
                if (body.isSoundPlayer != false &&
                    QueryConnectToSoundPlayer == false)
                {
                    QueryConnectToSoundPlayer = true;
                    manager.ConnectToSoundPlayer();
                }
            }

            //
            {
                RealtimeEditService service = Application.RealtimeEditService;
                service.SetTargetMachineInformation(body.memoryMax, body.memoryUsage,
                                                     body.editableItemCountMax,
                                                     body.editableItemCount,
                                                     body.editableFileCountMax,
                                                     body.editableFileCount,
                                                     body.isOutOfMemory,
                                                     body.isItemOverflow,
                                                     body.isFileOverflow);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected ApplicationBase Application
        {
            get
            {
                return ApplicationBase.Instance;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class QueryInvalidItemPacketFactory : ISyncPacketFactory
    {
        uint ISyncPacketFactory.Category
        {
            get { return GetCategory(); }
        }

        SyncPacket ISyncPacketFactory.CreatePacket(CommPacketHeader header)
        {
            return InternalCreatePacket(header);
        }

        protected virtual uint GetCategory()
        {
            //return Constants.SRE_CATEGORY_REALTIMEEDIT;
            return Constants.SRE_CATEGORY_SYSTEM;
        }

        protected virtual SyncPacket InternalCreatePacket(CommPacketHeader header)
        {
            switch (header.PacketType & Constants.SRE_MESSAGE_MASK)
            {
                case Constants.SRE_QUERY_ITEMS:
                    return new SndEditQueryItemsPacket();

                default:
                    break;
            }

            return null;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SndEditQueryItemsPacket : SyncPacket
    {
        private static int maxPopCount = 255;

        public static int MaxPopCount
        {
            get
            {
                return maxPopCount;
            }
            set
            {
                maxPopCount = value;
            }
        }

        /// <summary>
        /// ヘッダの作成
        /// </summary>
        public override CommPacketHeader CreateHeader()
        {
            return new SyncPacketHeader(PacketType | Constants.SRE_CATEGORY_SYSTEM, Size);
        }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.SRE_QUERY_ITEMS_REPLY; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                // この時点では計算できないので0を返しておきます。
                return 0;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string[] ItemNames
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
        }

        /// <summary>
        ///
        /// </summary>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            RealtimeEditService service = ApplicationBase.Instance.RealtimeEditService;
            QueryItemsReply body = new QueryItemsReply();
            List<ItemInfo> list = new List<ItemInfo>();

            List<string> updateList = new List<string>();
            List<string> removeList = new List<string>();
            service.PopChangedMonitoringTargets(updateList, removeList, MaxPopCount);

            list.AddRange
                (updateList
                  .Select(t => new ItemInfo(ActionType.UPDATE_ITEM, t))
                  .ToArray());

            list.AddRange
                (removeList
                  .Select(t => new ItemInfo(ActionType.DELETE_ITEM, t))
                  .ToArray());

#if DEBUG
            if( updateList.Count > 0 )
            {
                Console.WriteLine( "-------- Update --------");
                foreach( string name in updateList )
                {
                    Console.WriteLine( "{0}", name);
                }
                Console.WriteLine( "------------------------");
            }

            if( removeList.Count > 0 )
            {
                Console.WriteLine( "-------- Remove --------");
                foreach( string name in removeList )
                {
                    Console.WriteLine( "{0}", name);
                }
                Console.WriteLine( "------------------------");
            }
#endif

            body.items = list.ToArray();

            //
            body.allItemsAction = service.AllItemsAction;
            service.ResetAllItemsAction();

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Size = (uint)body.StructSize;
            header.Write(writer);

            // パラメータを出力する
            body.Write(writer);

            return header;
        }
    }
#endif
}
