﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Preset
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    public class PresetListColumnsService
    {
        private Dictionary<string, Dictionary<string, XmlList>> presetXmlList =
            new Dictionary<string, Dictionary<string, XmlList>>();
        private PresetListColumnsConfiguration presetListColumnsConfiguration = null;

        private Dictionary<string, string> currentNames = new Dictionary<string, string>();

        public PresetListColumnsService(PresetListColumnsConfiguration config)
        {
            this.presetListColumnsConfiguration = config;
        }

        public string ConfigPath { get; set; }
        public string ConfigFilePath
        {
            get
            {
                return Path.Combine(ConfigPath, "user.columns");
            }
        }

        public bool Contains(string listName, string name)
        {
            return (this.presetXmlList.ContainsKey(listName) == true &&
                    this.presetXmlList[listName].ContainsKey(name) == true);
        }

        public void Clear()
        {
            this.presetXmlList.Clear();
        }

        public void Add(string listName, string name, XmlList xmlList)
        {
            Debug.Assert(string.IsNullOrEmpty(listName) == false, "listName is null.");
            Debug.Assert(string.IsNullOrEmpty(name) == false, "name is null.");
            Debug.Assert(xmlList != null, "xmlList is null.");

            if (this.presetXmlList.ContainsKey(listName) == false)
            {
                this.presetXmlList.Add(listName, new Dictionary<string, XmlList>());
            }

            if (this.presetXmlList[listName].ContainsKey(name) == true)
            {
                return;
            }

            this.presetXmlList[listName].Add(name, xmlList);
        }

        public void Remove(string listName, string name)
        {
            if (this.presetXmlList.ContainsKey(listName) == true &&
                this.presetXmlList[listName].ContainsKey(name) == true)
            {
                this.presetXmlList[listName].Remove(name);
            }
        }

        public string[] GetPresetNames(string listName)
        {
            if (this.presetXmlList.ContainsKey(listName) == true)
            {
                return this.presetXmlList[listName].Keys.ToArray();
            }
            else
            {
                return new string[0];
            }
        }

        public string GetCurrentPresetName(string listName)
        {
            string currentName = string.Empty;

            if (this.currentNames.ContainsKey(listName) == true)
            {
                string[] presetNames = this.GetPresetNames(listName);
                if (presetNames.Length <= 0)
                {
                    currentName = string.Empty;
                }
                else if (presetNames.Contains(this.currentNames[listName]) == true)
                {
                    currentName = this.currentNames[listName];
                }
                else
                {
                    currentName = presetNames[0];
                }
            }
            else
            {
                string[] presetNames = this.GetPresetNames(listName);
                if (presetNames.Length > 0)
                {
                    currentName = presetNames[0];
                }
                else
                {
                    currentName = string.Empty;
                }
            }

            this.currentNames[listName] = currentName;

            return currentName;
        }

        public void SetCurrentPresetName(string listName, string presetName)
        {
            this.currentNames[listName] = presetName;
        }

        public XmlList GetListColumn(string listName, string name)
        {
            if (this.presetXmlList.ContainsKey(listName) == true &&
                this.presetXmlList[listName].ContainsKey(name) == true)
            {
                return this.presetXmlList[listName][name].DeepClone();
            }

            return null;
        }

        public void Save()
        {
            this.presetListColumnsConfiguration.PresetList = GetPresetListColumnsConfiguration();
            this.presetListColumnsConfiguration.Save(this.ConfigFilePath);
        }

        public void Load()
        {
            this.presetListColumnsConfiguration.Load(this.ConfigFilePath);
            SetPresetListColumnsConfiguration(this.presetListColumnsConfiguration.PresetList);
        }

        private void SetPresetListColumnsConfiguration(XmlPresetListColumnsConfiguration config)
        {
            this.presetXmlList.Clear();

            if (config != null)
            {
                XmlPresetLists xmlPresetLists = config.PresetLists;
                if (xmlPresetLists.PresetList != null)
                {
                    foreach (XmlPresetList xmlPresetList in xmlPresetLists.PresetList)
                    {
                        XmlList xmlList = TranslateDataXmlPresetListToXmlList(xmlPresetList);

                        this.Add(xmlPresetList.Name, xmlPresetList.PresetName, xmlList);
                    }
                }
            }
        }

        private XmlList TranslateDataXmlPresetListToXmlList(XmlPresetList xmlPresetList)
        {
            XmlList xmlList = null;

            if (xmlPresetList != null)
            {
                XmlListColumns xmlListColumns = TranslateDataXmlPresetListColumnsToXmlListColumns(xmlPresetList.PresetListColumns);
                xmlList = new XmlList();
                xmlList.Name = xmlPresetList.Name;
                xmlList.SortKey = xmlPresetList.SortKey;
                xmlList.SortOrder = (XmlListSortOrder)xmlPresetList.SortOrder;
                xmlList.ListColumns = xmlListColumns;
            }

            return xmlList;
        }

        private XmlListColumns TranslateDataXmlPresetListColumnsToXmlListColumns(XmlPresetListColumns xmlPresetListColumns)
        {
            XmlListColumns xmlListColumns = null;

            if (xmlPresetListColumns != null)
            {
                xmlListColumns = new XmlListColumns();
                List<XmlListColumn> columns = new List<XmlListColumn>();
                foreach (XmlPresetListColumn column in xmlPresetListColumns.PresetListColumn)
                {
                    XmlListColumn xmlListColumn = new XmlListColumn();
                    xmlListColumn.Name = column.Name;
                    xmlListColumn.Length = column.Length;
                    xmlListColumn.LengthSpecified = column.LengthSpecified;
                    xmlListColumn.Visible = column.Visible;
                    columns.Add(xmlListColumn);
                }

                xmlListColumns.ListColumn = columns.ToArray();
            }

            return xmlListColumns;
        }

        private XmlPresetListColumnsConfiguration GetPresetListColumnsConfiguration()
        {
            XmlPresetListColumnsConfiguration config = new XmlPresetListColumnsConfiguration();
            config.PresetLists = TranslateDataXmlListsToXmlPresetLists();

            return config;
        }

        private XmlPresetLists TranslateDataXmlListsToXmlPresetLists()
        {
            List<XmlPresetList> xmlPresetLists = new List<XmlPresetList>();

            foreach (string listName in this.presetXmlList.Keys)
            {
                Dictionary<string, XmlList> dic = this.presetXmlList[listName];
                foreach (string presetName in dic.Keys)
                {
                    XmlPresetList list = TranslateDataXmlListToXmlPresetList(dic[presetName]);
                    list.PresetName = presetName;
                    xmlPresetLists.Add(list);
                }
            }

            return new XmlPresetLists() { PresetList = xmlPresetLists.ToArray(), };
        }

        private XmlPresetList TranslateDataXmlListToXmlPresetList(XmlList xmlList)
        {
            XmlPresetListColumns xmlColumns = null;

            if (null != xmlList.ListColumns)
            {
                xmlColumns = new XmlPresetListColumns();
                if (null != xmlList.ListColumns.ListColumn)
                {
                    List<XmlPresetListColumn> workColumns = new List<XmlPresetListColumn>();

                    foreach (XmlListColumn srcColumn in xmlList.ListColumns.ListColumn)
                    {
                        XmlPresetListColumn column =
                            new XmlPresetListColumn()
                            {
                                Name = srcColumn.Name,
                                Length = srcColumn.Length,
                                LengthSpecified = srcColumn.LengthSpecified,
                                Visible = srcColumn.Visible,
                            };
                        workColumns.Add(column);
                    }

                    xmlColumns.PresetListColumn = workColumns.ToArray();
                }
            }

            XmlPresetList xmlPresetList =
                new XmlPresetList()
                {
                    Name = xmlList.Name,
                    SortKey = xmlList.SortKey,
                    SortOrder = (XmlPresetListSortOrder)(xmlList.SortOrder),
                    PresetListColumns = xmlColumns,
                };

            return xmlPresetList;
        }
    }
}
