﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.FileManagement
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Core.Resources;

    internal class FileSystemWatcherManager
    {
        private static readonly TimeSpan Interval = TimeSpan.FromMilliseconds(250); // 連続イベントを吸収する時間、0.25秒間

        private Dictionary<string, DateTime> updateFileDictionary;
        private Dictionary<string, FileSystemWatcher> fileSystemWatcherDictionary;
        private Object thisLock = new Object();

        /// <summary>
        /// ファイルの内容の変化のみを監視するかの設定を取得します。
        /// </summary>
        public bool WatchChangeOnly { get; }

        /// <summary>
        /// コンストラクタです。
        /// <para>
        /// <paramref name="watchChangeOnly"/> が true の場合、ファイルの内容の変化のみ監視します。指定のファイルが存在しない場合は監視を開始しません。
        /// </para>
        /// <para>
        /// <paramref name="watchChangeOnly"/> が false の場合は、ファイルの内容の変化に加えて名前の変更と削除も監視します。また指定のファイルが存在しなくても監視を開始します。
        /// </para>
        /// </summary>
        /// <param name="watchChangeOnly">名前の変更、削除も監視する場合は true を指定します。</param>
        public FileSystemWatcherManager(bool watchChangeOnly = true)
        {
            this.WatchChangeOnly = watchChangeOnly;
            updateFileDictionary = new Dictionary<string, DateTime>();
            fileSystemWatcherDictionary = new Dictionary<string, FileSystemWatcher>();
        }

        public bool Contains(string filePath)
        {
            return fileSystemWatcherDictionary.ContainsKey(filePath);
        }

        /// <summary>
        /// ファイルを監視する。
        /// </summary>
        /// <param name="filePath">監視対象のファイルパス。</param>
        public void Add(string filePath)
        {
            lock (thisLock)
            {
                if (this.WatchChangeOnly)
                {
                    if (File.Exists(filePath) == true &&
                        fileSystemWatcherDictionary.ContainsKey(filePath) == false)
                    {
                        FileSystemWatcher watcher = new FileSystemWatcher();
                        watcher.Path = Path.GetDirectoryName(filePath);
                        watcher.Filter = Path.GetFileName(filePath);
                        watcher.NotifyFilter = (NotifyFilters.LastWrite |
                                                NotifyFilters.FileName |
                                                NotifyFilters.DirectoryName |
                                                NotifyFilters.CreationTime);
                        watcher.Changed += OnFileChanged;
                        watcher.Created += OnFileChanged;
                        fileSystemWatcherDictionary.Add(filePath, watcher);
                        watcher.EnableRaisingEvents = true;
                    }
                }
                else
                {
                    if (fileSystemWatcherDictionary.ContainsKey(filePath) == false)
                    {
                        FileSystemWatcher watcher = new FileSystemWatcher();
                        watcher.Path = Path.GetDirectoryName(filePath);
                        watcher.Filter = Path.GetFileName(filePath);
                        watcher.NotifyFilter = (NotifyFilters.LastWrite |
                                                NotifyFilters.FileName |
                                                NotifyFilters.DirectoryName |
                                                NotifyFilters.CreationTime);
                        watcher.Changed += OnFileChanged;
                        watcher.Created += OnFileChanged;
                        watcher.Deleted += OnFileDeleted;
                        watcher.Renamed += OnFileRenamed;
                        fileSystemWatcherDictionary.Add(filePath, watcher);
                        watcher.EnableRaisingEvents = true;
                    }
                }
            }
        }

        /// <summary>
        /// ファイルの監視をやめる。
        /// </summary>
        public void Remove(string filePath)
        {
            lock (thisLock)
            {
                if (fileSystemWatcherDictionary.ContainsKey(filePath) == true)
                {
                    FileSystemWatcher watcher = fileSystemWatcherDictionary[filePath];
                    watcher.EnableRaisingEvents = false;
                    fileSystemWatcherDictionary.Remove(filePath);
                    watcher.Dispose();
                    this.RemoveUpdateFile(filePath);
                }
            }
        }

        /// <summary>
        /// 更新があったファイルパス群を得る。
        /// <para>
        /// getc のようにその時の状態を返す。
        /// 連続で２度読んでも同じ値にならない事に注意。
        /// </para>
        /// </summary>
        public string[] GetFilePaths()
        {
            List<string> filePaths = new List<string>();

            // イベントをまとめるためと、変更されたファイルに対する処理を他のプロセスと競合しづらくするため、
            // 最後のイベント受信が現在時刻よりも少し前だったファイルのみ、変更を通知します。
            DateTime latestEventTimeToNotice = DateTime.Now - Interval;

            lock (thisLock)
            {
                foreach (KeyValuePair<string, DateTime> pair in updateFileDictionary)
                {
                    DateTime eventTime = pair.Value;
                    if (eventTime < latestEventTimeToNotice)
                    {
                        filePaths.Add(pair.Key);
                    }
                }
                foreach (string filePath in filePaths)
                {
                    updateFileDictionary.Remove(filePath);
                }
            }

            return filePaths.ToArray();
        }

        /// <summary>
        /// 更新したファイルを登録する。
        /// </summary>
        private void AddUpdateFile(string filePath)
        {
            lock (thisLock)
            {
                // 監視をやめた後にイベントを受信したときは無視します。
                if (fileSystemWatcherDictionary.ContainsKey(filePath) == false)
                {
                    return;
                }

                updateFileDictionary[filePath] = DateTime.Now;
            }
        }

        /// <summary>
        /// ファイルが監視下から外れれば、不要なので更新したファイルにも登録しない。
        /// </summary>
        private void RemoveUpdateFile(string filePath)
        {
            lock (thisLock)
            {
                updateFileDictionary.Remove(filePath);
            }
        }

        private void OnFileChanged(object sender, FileSystemEventArgs e)
        {
            switch (e.ChangeType)
            {
                case WatcherChangeTypes.Changed:
                case WatcherChangeTypes.Created:
                    AddUpdateFile(e.FullPath);
                    break;
            }
        }

        private void OnFileDeleted(object sender, FileSystemEventArgs e)
        {
            switch (e.ChangeType)
            {
                case WatcherChangeTypes.Deleted:
                    AddUpdateFile(e.FullPath);
                    break;
            }
        }

        private void OnFileRenamed(object sender, RenamedEventArgs e)
        {
            lock (thisLock)
            {
                if (this.fileSystemWatcherDictionary.ContainsKey(e.FullPath))
                {
                    AddUpdateFile(e.FullPath);
                }

                if (this.fileSystemWatcherDictionary.ContainsKey(e.OldFullPath))
                {
                    AddUpdateFile(e.OldFullPath);
                }
            }
        }
    }
}
