﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.FileFormats
{
    using System.IO;
    using SoundFoundation.Conversion;
    using SoundFoundation.Conversion.NintendoWareBinary;
    using SoundFoundation.FileFormats.NintendoWareIntermediate;
    using ToolDevelopmentKit;

    /// <summary>
    /// シーケンスサウンドテキストを解析します。
    /// </summary>
    public class SequenceSoundTextParser : SequenceSoundFileParser
    {
        /// <summary>
        /// ファイルを解析します。
        /// </summary>
        /// <param name="filePath">ファイルパスを指定します。</param>
        protected override void ParseInternal(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            // シーケンスサウンドバイナリファイルにコンバートして解析します。
            ISequenceSoundTextConverter converter = ApplicationBase.Instance.Traits.ConversionTraits.SequenceSoundTextConverter;
            Assertion.Operation.ObjectNotNull(converter);

            converter.InputPath = filePath;
            converter.OutputPath = Path.GetTempFileName();

            try
            {
                converter.Run(
                    new ConversionContext(ApplicationBase.Instance.Traits.ConversionTraits,
                                          ApplicationBase.Instance.ProjectService.ProjectFilePath))
                    ;

                SequenceSoundBinaryParser binaryParser = new SequenceSoundBinaryParser()
                {
                    IsLittleEndian = this.IsLittleEndian,
                };
                binaryParser.Parse(converter.OutputPath);

                this.Labels = binaryParser.Labels;

                this.AddDependentFileTimes(filePath);
            }
            finally
            {
                File.Delete(converter.OutputPath);
            }
        }

        private void AddDependentFileTimes(string filePath)
        {
            Assertion.Argument.NotNull(filePath);
            Assertion.Argument.StringNotEmpty(filePath);

            this.DependentFileTimes[filePath] = File.GetLastWriteTime(filePath);

            SequenceSoundTextLexer lexer = new SequenceSoundTextLexer(filePath);

            while (lexer.ReadToken() != SequenceSoundTextLexer.Token.End)
            {
                // 依存ファイル収集のため空回し
            }

            foreach (string dependentFilePath in lexer.DependFiles)
            {
                this.DependentFileTimes[dependentFilePath] = File.GetLastWriteTime(dependentFilePath);
            }

            lexer.Dispose();
            lexer = null;
        }
    }
}
