﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.FileFormats
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;

    /// <summary>
    /// シーケンスサウンドファイルを解析します。
    /// </summary>
    public abstract class SequenceSoundFileParser : IFileParser
    {
        private IKeyedList<string, Label> labelInfos = new LabelInfoList();
        private IDictionary<string, DateTime> dependentFileTimes = new Dictionary<string, DateTime>();
        private bool isParsed = false;

        /// <summary>
        /// エンディアンを取得または設定します。
        /// </summary>
        public bool IsLittleEndian { get; set; }

        /// <summary>
        /// 解析したファイルの情報を保持しているかどうかを調べます。
        /// </summary>
        public bool IsParsed
        {
            get { return this.isParsed; }
        }

        /// <summary>
        /// 解析したファイルが更新されているかどうかを調べます。
        /// </summary>
        public bool IsUpdated
        {
            get
            {
                if (!this.isParsed)
                {
                    return true;
                }

                foreach (KeyValuePair<string, DateTime> value in this.dependentFileTimes)
                {
                    if (!File.Exists(value.Key))
                    {
                        continue;
                    }

                    DateTime time = File.GetLastWriteTime(value.Key);
                    if (time != value.Value)
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        /// <summary>
        /// ラベル辞書を取得します。
        /// </summary>
        public IKeyedList<string, Label> Labels
        {
            get
            {
                return this.labelInfos;
            }
            protected set
            {
                Ensure.Argument.NotNull(value);
                this.labelInfos = value;
            }
        }

        protected IDictionary<string, DateTime> DependentFileTimes
        {
            get { return this.dependentFileTimes; }
        }

        /// <summary>
        /// ファイルを解析します。
        /// </summary>
        /// <param name="filePath">ファイルパスを指定します。</param>
        public void Parse(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            this.isParsed = false;
            this.DependentFileTimes.Clear();

            this.ParseInternal(filePath);

            this.isParsed = true;
        }

        /// <summary>
        /// ファイルを解析します。
        /// </summary>
        /// <param name="filePath">ファイルパスを指定します。</param>
        protected abstract void ParseInternal(string filePath);

        /// <summary>
        /// ラベル
        /// </summary>
        public class Label
        {
            public string Name { get; set; }
            public long Address { get; set; }
            public long OffsetFromDataBlockBody { get; set; }
        }

        private class LabelInfoList : KeyedCollection<string, Label>, IKeyedList<string, Label>
        {
            /// <summary>
            /// 指定されたキーがコレクション内に存在するかどうか調べます。
            /// </summary>
            /// <param name="key">アイテムのキーを指定します。</param>
            /// <returns>
            /// 指定されたキーがコレクション内に存在する場合は true、
            /// 存在しない場合は false を返します。
            /// </returns>
            public bool ContainsKey(string key)
            {
                return base.Contains(key);
            }

            /// <summary>
            /// 指定されたキーに関連付けられている値を取得します。
            /// </summary>
            /// <param name="key">アイテムのキーを指定します。</param>
            /// <returns>指定されたキーに関連付けられているアイテムを返します。</returns>
            public Label GetValue(string key)
            {
                Ensure.Argument.NotNull(key);
                return this[key];
            }

            /// <summary>
            /// 指定されたキーに関連付けられている値を取得します。
            /// </summary>
            /// <param name="key">アイテムのキーを指定します。</param>
            /// <param name="value">
            /// 指定されたキーが存在する場合、そのキーに関連付けられている値を返します。
            /// それ以外の場合は value パラメータの型に対する既定の値を返します。
            /// </param>
            /// <returns>指定されたキーに関連付けられているアイテムを返します。</returns>
            public bool TryGetValue(string key, out Label value)
            {
                Ensure.Argument.NotNull(key);

                if (base.Contains(key))
                {
                    value = this[key];
                    return true;
                }
                else
                {
                    value = null;
                    return false;
                }
            }

            protected override string GetKeyForItem(Label item)
            {
                return item.Name;
            }
        }
    }
}
