﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace NintendoWare.SoundMaker.Framework.Configurations.Schemas
{
    /// <summary>
    /// 最近使ったファイルを格納します。
    /// </summary>
    public partial class XmlMruFiles
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public XmlMruFiles() { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="filePaths">ファイルパスの列挙子。</param>
        public XmlMruFiles(IEnumerable<string> filePaths)
        {
            FilePath = filePaths.ToArray();
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="filePaths">ファイルパスの列挙子。</param>
        /// <param name="count">適用するファイルパスの数。</param>
        public XmlMruFiles(IEnumerable<string> filePaths, int count)
        {
            FilePath = filePaths.Take(count).ToArray();
        }

        /// <summary>
        /// XmlMruFiles に含まれる全ファイルパスの一覧を取得します。
        /// </summary>
        /// <param name="limit">アイテム数の上限。</param>
        /// <returns>全ファイルパスの一覧。</returns>
        public XmlMruFileList CreateList(int limit)
        {
            return new XmlMruFileList(limit, FilePath);
        }
    }

    /// <summary>
    /// 最近使ったファイルのリストです。
    /// </summary>
    public class XmlMruFileList : IEnumerable<string>, IEnumerable
    {
        private InnerCollection _innerCollection = new InnerCollection();
        private int _limit = int.MaxValue;

        public XmlMruFileList() { }

        public XmlMruFileList(int limit) : this(limit, null) { }

        public XmlMruFileList(IEnumerable<string> items) : this(0, null) { }

        public XmlMruFileList(int limit, IEnumerable<string> items)
        {
            _limit = (0 > limit) ? int.MaxValue : limit;

            if (null == items) { return; }

            foreach (string item in items.Take(_limit))
            {
                _innerCollection.Add(item);
            }
        }

        /// <summary>
        /// アイテムが変更されると発生します。
        /// </summary>
        public event EventHandler ItemChanged;

        /// <summary>
        /// リストの先頭にアイテムを追加します。
        /// </summary>
        /// <param name="item">追加するアイテム。</param>
        public void PushFront(string item)
        {
            PushFrontInternal(item);
            OnItemChanged(EventArgs.Empty);
        }

        /// <summary>
        /// リストの先頭にアイテムを追加します。
        /// </summary>
        /// <param name="items">追加するアイテムの列挙子。</param>
        public void PushFront(IEnumerable<string> items)
        {
            if (null == items) { throw new ArgumentNullException("items"); }

            foreach (string item in items)
            {
                PushFrontInternal(item);
            }

            OnItemChanged(EventArgs.Empty);
        }

        /// <summary>
        /// 余分なアイテムを削除します。
        /// </summary>
        /// <param name="limit">アイテム数の上限。</param>
        public void Trim(int limit)
        {
            if (0 > limit) { throw new ArgumentOutOfRangeException("limit"); }

            while (_innerCollection.Count > limit)
            {
                _innerCollection.RemoveAt(_innerCollection.Count - 1);
            }
        }

        /// <summary>
        /// 指定アイテムが含まれているかどうかを調べます。
        /// </summary>
        /// <param name="item">削除するアイテム。</param>
        /// <returns>含まれている場合は true、含まれていない場合は false。</returns>
        public bool Contains(string item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }
            return _innerCollection.Contains(item);
        }

        /// <summary>
        /// 指定アイテムを削除します。
        /// </summary>
        /// <param name="item">削除するアイテム。</param>
        public void Delete(string item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            _innerCollection.Remove(item);
            OnItemChanged(EventArgs.Empty);
        }

        /// <summary>
        /// アイテムの列挙子を取得します。
        /// </summary>
        /// <returns>アイテムの列挙子</returns>
        IEnumerator<string> IEnumerable<string>.GetEnumerator()
        {
            return _innerCollection.GetEnumerator();
        }

        /// <summary>
        /// アイテムの列挙子を取得します。
        /// </summary>
        /// <returns>アイテムの列挙子</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return _innerCollection.GetEnumerator();
        }

        /// <summary>
        /// アイテムが変更されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnItemChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ItemChanged)
            {
                ItemChanged(this, e);
            }
        }

        /// <summary>
        /// リストの先頭にアイテムを追加します。
        /// </summary>
        /// <param name="item">追加するアイテム。</param>
        private void PushFrontInternal(string item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            if (_innerCollection.Contains(item))
            {
                _innerCollection.Remove(item);
            }

            _innerCollection.Insert(0, item);

            if (0 < _limit)
            {
                Trim(_limit);
            }
        }

        private class InnerCollection : KeyedCollection<string, string>
        {
            protected override string GetKeyForItem(string item)
            {
                return item;
            }
        }
    }
}
