﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// プロジェクトにサウンドセットを追加します。
    /// </summary>
    public abstract class AddSoundSetHandler : AddComponentHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">対象コマンドID。</param>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        protected AddSoundSetHandler(string commandID, bool insert, IQueryCommandParameter queryParameter)
            : base(commandID, insert, queryParameter) { }

        /// <summary>
        /// プロジェクトにサウンドセットを追加します。
        /// </summary>
        /// <param name="projectService">プロジェクトサービス。</param>
        /// <param name="documentRef">追加先のサウンドセットドキュメントへの参照。</param>
        /// <param name="targetComponent">親Componentの指定です。nullを指定するとSoundProjectが親になります。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(SoundProjectService projectService, DocumentReference documentRef, Component targetComponent = null)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == projectService.ProjectDocument)
            {
                throw new ArgumentException("projectService.ProjectDocument must not be null.");
            }
            if (null == projectService.Project)
            {
                throw new ArgumentException("projectService.Project must not be null.");
            }

            if (null == documentRef) { throw new ArgumentNullException("documentRef"); }
            if (null == documentRef.Document)
            {
                throw new ArgumentException("documentRef.Document must not be null.");
            }
            if (!(documentRef.Document is SoundSetDocument))
            {
                throw new ArgumentException("documentRef.Document must not be SoundSetDocument.");
            }

            if (ContainsSoundSet(projectService, documentRef.Document.Resource.Key))
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(MessageResource.Message_SoundSetAlreadyExisting);
                return false;
            }

            //
            string filePath = documentRef.Document.Resource.Key;
            string name = null;

            name = CreateUniqueName(Path.GetFileNameWithoutExtension(filePath));
            ((SoundSetDocument)documentRef.Document).SoundSet.Name = name;

            var addSoundSetOperation = new AddSoundSetOperation(projectService, filePath, targetComponent);
            if (!addSoundSetOperation.Execute()) { return false; }

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            SoundProjectService service = GetTargetComponentService(parameters) as SoundProjectService;
            if (service == null ||
                service.Project == null ||
                service.ProjectDocument.IsReadOnly != false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            var parentComponent = GetInsertParentComponent(parameters);
            if (parentComponent != null &&
                IsContainerComponent(parentComponent) == false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// プロジェクトにサウンドセットドキュメントを追加します。
        /// </summary>
        /// <param name="documentRef">追加するサウンドセットドキュメントへの参照。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected bool Execute(IQueryCommandParameter parameters, DocumentReference documentRef)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            if (null == documentRef) { throw new ArgumentNullException("documentRef"); }
            if (null == documentRef.Document) { throw new ArgumentException("documentRef.Document must not be null."); }
            if (!(documentRef.Document is SoundSetDocument))
            {
                throw new ArgumentException("documentRef.Document must not be SoundSetDocument.");
            }

            SoundProjectService targetProjectService = GetTargetComponentService(parameters) as SoundProjectService;
            if (null == targetProjectService) { return false; }

            var targetComponent = GetInsertParentComponent(parameters);

            return Execute(targetProjectService, documentRef, targetComponent);
        }

        /// <summary>
        /// 指定サウンドセットファイルパスが既にプロジェクトに含まれているかどうか調べます。
        /// </summary>
        /// <param name="projectService">プロジェクトサービス。</param>
        /// <param name="soundSetFilePath">サウンドセットのファイルパス。</param>
        /// <returns>含まれている場合は true、含まれていない場合は false。</returns>
        private static bool ContainsSoundSet(SoundProjectService projectService, string soundSetFilePath)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == soundSetFilePath) { throw new ArgumentNullException("soundSetFilePath"); }

            return projectService.ContainsSoundSetDocument(soundSetFilePath);
        }

        /// <summary>
        /// サウンドセット名の作成
        /// </summary>
        private static string CreateUniqueName(string baseName)
        {
            SoundProjectService projectService = null;
            string name = baseName;
            int number = 0;

            projectService = ApplicationBase.Instance.ProjectService;

            while (ContainsSoundSetName(name) != false)
            {
                name = baseName + string.Format("_{0}", number);
                number++;
            }
            return name;
        }

        ///
        private static bool ContainsSoundSetName(string name)
        {
            SoundSetDocument[] soundSetDocuments = null;

            soundSetDocuments = ApplicationBase.Instance.ProjectService.SoundSetDocuments
                .Where(d => d is SoundSetDocument).Cast<SoundSetDocument>().ToArray();

            foreach (SoundSetDocument soundSetDocument in soundSetDocuments)
            {
                if (soundSetDocument.SoundSet.Name == name)
                {
                    return true;
                }
            }
            return false;
        }

        private bool IsContainerComponent(Component component)
        {
            return component is SoundProject || component is FolderComponent ? true : false;
        }
    }
}
