﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    public delegate bool ClosedDelegate();

    /// <summary>
    /// プロジェクトを開きます。
    /// </summary>
    public class CloseProjectHandler : CommandHandler
    {
        public static ClosedDelegate ClosedDelegate;

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.CloseProject.ID; }
        }

        /// <summary>
        /// アプリケーションを取得します。
        /// </summary>
        private ApplicationBase Application
        {
            get { return ApplicationBase.Instance; }
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        public static bool Execute()
        {
            return new CloseProjectHandler().Execute(FileCommands.CloseProject);
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            return (null != ApplicationBase.Instance.ProjectService.ProjectDocument) ?
                    CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            AppMessageBoxResult result = ConfirmCloseProject();

            switch (result)
            {
                case AppMessageBoxResult.Cancel:
                    return false;

                case AppMessageBoxResult.Yes:
                    if (SaveHelper.SaveAll() == false)
                    {
                        return false;
                    }
                    CollectSoundSetGarbagesInProjectConfiguration();
                    CollectBankGarbagesInProjectConfiguration();
                    break;

                case AppMessageBoxResult.OK:
                    CollectSoundSetGarbagesInProjectConfiguration();
                    CollectBankGarbagesInProjectConfiguration();
                    break;

                case AppMessageBoxResult.No:
                    break;

                default:
                    throw new Exception("invalid messege box result.");
            }

            var defaultparamFilePath = Application.GetDefaultParametersFilePath();
            Application.CreateComponentService.OutputDefaultParametersFile(defaultparamFilePath);
            Application.ProjectConfiguration.Save();
            Application.PresetListColumnsService.Save();
            Application.BookmarkService.Save();

            // 保存確認が表示されないように未保存のバンクを閉じておく
            CloseAllBankService();

            return CloseProjectDocument(Application.ProjectService);
        }

        /// <summary>
        /// 対象ドキュメントの一覧を取得します。
        /// </summary>
        private IEnumerable<SoundDocument> GetTaretDocuments()
        {
            foreach (SoundDocument document in from SoundDocument document in Application.ProjectService.Documents.Values
                                               where document.IsDirty
                                               where (null != document.Resource)
                                               select document)
            {
                yield return document;
            }

            foreach (SoundDocument document in from BankService bankService in Application.BankServices.Values
                                               where (null != bankService.BankDocument)
                                               where (null != bankService.BankDocument.Resource)
                                               where bankService.BankDocument.IsDirty
                                               select bankService.BankDocument)
            {
                yield return document;
            }
        }

        /// <summary>
        /// プロジェクトを閉じてもよいか確認します。
        /// </summary>
        /// <returns>メッセージボックスの戻り値。</returns>
        private AppMessageBoxResult ConfirmCloseProject()
        {
            if (null == Application.ProjectService)
            {
                throw new ArgumentException("Application.ProjectService must not be null.");
            }
            if (null == Application.ProjectService.ProjectDocument) { return AppMessageBoxResult.Cancel; }

            SoundDocument[] targetDocuments = GetTaretDocuments().ToArray();
            if (0 == targetDocuments.Length) { return AppMessageBoxResult.OK; }

            string messageText = MessageResource.Message_ConfirmCloseDocument;

            foreach (Document document in targetDocuments)
            {
                messageText += "\n " + Path.GetFileName(document.Resource.Key);
            }

            return Application.UIService.ShowMessageBox(messageText, AppMessageBoxButton.YesNoCancel,
                                                         AppMessageBoxImage.Question, AppMessageBoxResult.Yes);
        }

        /// <summary>
        /// 開かれているサウンドプロジェクトを閉じます。
        /// </summary>
        private bool CloseProjectDocument(SoundProjectService projectService)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }

            projectService.Close();

            if (CloseProjectHandler.ClosedDelegate != null)
            {
                return CloseProjectHandler.ClosedDelegate();
            }

            return true;
        }

        private void CloseAllBankService()
        {
            Application.BankServices.Clear();
        }

        private void CollectSoundSetGarbagesInProjectConfiguration()
        {
            if (null == Application.ProjectService.ProjectDocument) { return; }
            if (null == Application.ProjectConfiguration) { return; }

            List<string> garbages = new List<string>();

            foreach (XmlDocumentView view in Application.ProjectConfiguration.DocumentViews.Values)
            {

                string targetFilePath = view.FilePath.GetFullPath();

                if (view is XmlSoundProjectDocumentView)
                {
                    if (Application.ProjectService.ProjectDocument.Resource.Key == targetFilePath) { continue; }
                }
                else if (view is XmlSoundSetDocumentView)
                {
                    if (Application.ProjectService.Documents.ContainsKey(targetFilePath)) { continue; }
                }
                else
                {
                    continue;
                }

                garbages.Add(targetFilePath);

            }

            foreach (string garbage in garbages)
            {
                Application.ProjectConfiguration.DocumentViews.Remove(garbage);
            }
        }

        private void CollectBankGarbagesInProjectConfiguration()
        {
            if (null == Application.ProjectConfiguration) { return; }

            List<XmlDocumentView> garbages = new List<XmlDocumentView>();

            foreach (XmlDocumentView view in Application.ProjectConfiguration.DocumentViews.Values)
            {

                if (!(view is XmlBankDocumentView)) { continue; }

                if ((from SoundSetBankBase bank in Application.ProjectService.SoundSetBanks
                     where bank.FilePath == view.FilePath
                     select bank).FirstOrDefault() != null) { continue; }

                garbages.Add(view);

            }

            foreach (XmlDocumentView garbage in garbages)
            {
                Application.ProjectConfiguration.DocumentViews.Remove(garbage.FilePath);
            }
        }
    }
}
