﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;

using NintendoWare.SoundFoundation.CommandHandlers;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Parameters;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Commands;
using NintendoWare.SoundMaker.Framework.Utilities;
using NintendoWare.SoundMaker.Framework.Windows;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    ///--------------------------------------------------------------------------
    /// <summary>
    /// "削除"、"切り取り"用の基本ハンドラ
    /// </summary>
    public class DeleteHandlerBase : EditCommandHandler
    {
        ///--------------------------------
        /// <summary>
        /// </summary>
        protected DeleteHandlerBase(string commandID, IQueryCommandParameter queryParameter)
            : base(commandID, queryParameter)
        {
        }

        ///--------------------------------
        /// <summary>
        /// </summary>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            Component[] components = GetTargetComponents(parameters);
            if (components == null) { return false; }

            SoundProjectService projectService = GetTargetComponentService(parameters) as SoundProjectService;
            SoundDocument document = GetTargetDocument(parameters);
            if (null == document) { return false; }

            Dictionary<SoundDocument, List<Component>> dictionary = new Dictionary<SoundDocument, List<Component>>();
            dictionary[document] = new List<Component>(components);

            bool referenced = false; // グループアイテムの参照の有無。

            if (projectService != null)
            {
                var groupItems = projectService.Groups
                    .SelectMany(g => g.Children.Cast<GroupItemBase>());

                // 参照しているグループアイテムを調べます。
                foreach (var component in components)
                {
                    if (component is SoundSetItem == false)
                    {
                        continue;
                    }

                    var referencedItems = groupItems
                        .Where(i => GroupUtility.DoReferred(i, component as SoundSetItem));
                    if (referencedItems.Any() == true)
                    {
                        referenced = true;

                        foreach (var referencedItem in referencedItems)
                        {
                            var refSoundSet = this.GetSoundSet(referencedItem);
                            var refDocument = this.GetSoundSetDocument(projectService, refSoundSet);

                            if (dictionary.ContainsKey(refDocument) == false)
                            {
                                dictionary[refDocument] = new List<Component>();
                            }

                            dictionary[refDocument].Add(referencedItem);
                        }
                    }
                }

                if (referenced)
                {
                    if (ApplicationBase.Instance.UIService.ShowMessageBox(
                            Resources.MessageResource.Message_ConfirmDeleteReferencedItem,
                            null,
                            AppMessageBoxButton.YesNo,
                            AppMessageBoxImage.Question,
                            AppMessageBoxResult.Yes) != AppMessageBoxResult.Yes)
                    {
                        return false; // ここは全体のキャンセルなので false を返す。
                    }
                }
            }

            if (!OnDeleting(parameters, components)) { return false; }

            try
            {
                dictionary.Keys.ToList().ForEach(d => d.OperationHistory.BeginTransaction());

                dictionary.Keys.ToList().ForEach(d => this.DeleteComponents(d, dictionary[d]));

                dictionary.Keys.ToList().ForEach(d => d.OperationHistory.EndTransaction());
            }
            catch
            {
                dictionary.Keys.ToList().ForEach(d => d.OperationHistory.CancelTransaction());
            }

            return true;
        }

        protected virtual bool OnDeleting(IQueryCommandParameter parameters, Component[] components)
        {
            return true;
        }

        private void DeleteComponents(SoundDocument document, IEnumerable<Component> components)
        {
            Debug.Assert(null != document, "invalid argument");
            Debug.Assert(null != components, "invalid argument");

            // 一度のオペレーションにするため同じ親のコンポーネントをまとめる。
            Dictionary<Component, List<Component>> dictionary = new Dictionary<Component, List<Component>>();
            foreach (Component component in components)
            {
                if (dictionary.ContainsKey(component.Parent) == false)
                {
                    dictionary[component.Parent] = new List<Component>();
                }

                dictionary[component.Parent].Add(component);
            }

            // 同じ親をもつコンポーネント達は一度のオペレーションで行うようにする。
            foreach (Component parent in dictionary.Keys)
            {
                Operation operation = new RemoveComponentOperation(dictionary[parent]);
                operation.Execute();
                document.OperationHistory.AddOperation(operation);
            }
        }

        /// <summary>
        /// Component から SoundSet を取得します。
        /// </summary>
        private SoundSet GetSoundSet(Component component)
        {
            if (component == null)
            {
                return null;
            }

            if (component is SoundSet)
            {
                return component as SoundSet;
            }

            return this.GetSoundSet(component.Parent);
        }

        /// <summary>
        /// SoundSet から SoundSetDocument を取得します。
        /// </summary>
        private SoundSetDocument GetSoundSetDocument(SoundProjectService projectService, SoundSet soundSet)
        {
            return (projectService.SoundSetDocuments
                    .Where(d => d.SoundSet == soundSet)
                    .First());
        }
    }
}
