﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Windows.Forms;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    public partial class SeqVariableEdit : NUserControl
    {
        // パラメータ
        private ISeqVariableEdit _editor = null;

        // 状態
        private bool _lockedEditVariables = false;	// 書き込みロック
        private bool _editVariablesDirty = false;	// 書き込み Dirtyフラグ

        public SeqVariableEdit()
        {
            InitializeComponent();
        }

        #region ** プロパティ

        [ReadOnly(true)]
        [Browsable(false)]
        public ISeqVariableEdit Editor
        {
            get { return _editor; }
            set
            {
                if (value == _editor) { return; }

                _editor = value;

                OnEditorChanged(new EventArgs());
            }
        }

        [ReadOnly(true)]
        [Browsable(false)]
        public string TypeName
        {
            get
            {
                if (null == _editor) { return string.Empty; }
                return _editor.TypeName;
            }
        }

        [ReadOnly(true)]
        [Browsable(false)]
        public string TargetName
        {
            get
            {
                if (null == _editor) { return string.Empty; }
                return _editor.TargetName;
            }
        }

        [ReadOnly(true)]
        [Browsable(false)]
        protected Preview.PreviewPlayerManager PreviewPlayerManager
        {
            get { return Preview.PreviewManager.Instance.PreviewPlayerManager; }
        }

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnEditorChanged(EventArgs e)
        {
            UpdateEditValue();
            UpdateCurrentValue();
        }

        /// <summary>
        /// チェック状態が変更されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnEditCheckedChanged(object sender, SeqVariableEventArgs e)
        {
            Debug.Assert(null != _editor, "unexpected error");

            // 編集値をモデルに反映する
            Preview.SeqVariable target = _editor.VariablesSet.Edit.Variables[(int)e.Control.Index];
            target.Checked = e.Control.Checked;

            if (!PreviewPlayerManager.Settings.AutoWriteVariables) { return; }


            // ロック中は処理しない
            if (_lockedEditVariables) { return; }

            ApplyVariables();
        }

        /// <summary>
        /// 変数の値が編集されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnEditValueChanged(object sender, SeqVariableValueChangedEventArgs e)
        {
            // ロック中は処理しない
            if (_lockedEditVariables) { return; }


            Debug.Assert(null != _editor, "unexpected error");

            // 編集値をモデルに反映する
            Preview.SeqVariable target = _editor.VariablesSet.Edit.Variables[(int)e.Control.Index];
            target.Value = e.Control.EditValue;


            LockEditVariables();

            // 編集したアイテムにチェックをつけて、値を書き込む
            e.Control.Checked = true;

            if (PreviewPlayerManager.Settings.AutoWriteVariables)
            {
                ApplyVariables();
            }

            UnlockEditVariables(true);
        }

        /// <summary>
        /// チェック反転ボタンが押下されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnReverseCheckButtonClicked(object sender, EventArgs e)
        {
            LockEditVariables();

            foreach (SeqVariableControl control in _varSet.VariableControls)
            {
                control.Checked = !control.Checked;
            }

            UnlockEditVariables(true);
        }

        /// <summary>
        /// チェッククリアボタンが押下されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClearCheckButtonClicked(object sender, EventArgs e)
        {
            LockEditVariables();

            foreach (SeqVariableControl control in _varSet.VariableControls)
            {
                control.Checked = false;
            }

            UnlockEditVariables(true);
        }

        /// <summary>
        /// 書き込みボタンが押下されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnApplyButtonClicked(object sender, EventArgs e)
        {
            ApplyVariables();
        }

        #endregion

        #region ** メソッド

        public void UpdateEditValue()
        {
            Debug.Assert(null != _editor, "unexpected error");

            if (null == _editor.VariablesSet)
            {
                InvalidateEditValue();
                return;
            }

            LockEditVariables();

            foreach (Preview.SeqVariable item in _editor.VariablesSet.Edit.Variables)
            {

                SeqVariableControl targetControl = _varSet.VariableControls[(int)item.No];

                targetControl.Checked = item.Checked;

                if (Preview.SeqVariable.InvalidValue == item.Value)
                {
                    targetControl.EditValue = Preview.SeqVariable.DefaultValue;
                }
                else
                {
                    targetControl.EditValue = item.Value;
                }

            }

            ResetEditDirty();
            UnlockEditVariables(false);
        }

        public void UpdateCurrentValue()
        {
            if (null == _editor || null == _editor.VariablesSet)
            {
                InvalidateCurrentValue();
                return;
            }

            foreach (Preview.SeqVariable item in _editor.VariablesSet.Current.Variables)
            {
                _varSet.VariableControls[(int)item.No].CurrentValue = item.Value;
            }
        }

        public void UpdateControls()
        {
            // 自動書き込みON の場合は、書き込みボタンを無効化する
            _applyButton.Enabled = !PreviewPlayerManager.Settings.AutoWriteVariables;
        }

        private void LockEditVariables()
        {
            if (_lockedEditVariables) { return; }

            _lockedEditVariables = true;
        }

        private void UnlockEditVariables(bool applyVariables)
        {
            if (!_lockedEditVariables) { return; }

            _lockedEditVariables = false;

            if (applyVariables && _editVariablesDirty)
            {
                ApplyVariables();
            }
        }

        private void ResetEditDirty()
        {
            _editVariablesDirty = false;
        }

        private void InvalidateEditValue()
        {
            LockEditVariables();

            foreach (SeqVariableControl control in _varSet.VariableControls)
            {
                control.EditValue = Preview.SeqVariable.DefaultValue;
            }

            ResetEditDirty();
            UnlockEditVariables(false);
        }

        private void InvalidateCurrentValue()
        {
            foreach (SeqVariableControl control in _varSet.VariableControls)
            {
                control.CurrentValue = Preview.SeqVariable.InvalidValue;
            }
        }

        private void ApplyVariables()
        {
            if (null == _editor || null == _editor.VariablesSet) { return; }

            if (_lockedEditVariables)
            {
                _editVariablesDirty = true;
                return;
            }

            // 編集値を送信する
            _editor.ApplyVariables();

            // トラックバーを動かす等して、Set->Get->Set->Get の繰り返しになると、
            // パケットを圧縮できず、パフォーマンスが落ちるため、コメントアウト
            //MCSSoundPlayer.GetSeqVariables( Variables );

            _editVariablesDirty = false;
        }

        #endregion
    }

    public interface ISeqVariableEdit
    {
        #region ** プロパティ

        string TypeName { get; }
        string TargetName { get; }

        Preview.SeqVariableContainerSet VariablesSet { get; }

        #endregion

        #region ** メソッド

        void ApplyVariables();

        #endregion
    }
}
