﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Data;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Core.Drawing;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundMaker.Framework.Windows;
using NW4R.ProtocolSound;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    public partial class EffectAuxBusControl : NUserControl
    {
        private const int ArrowWidth = 10;
        private const int ArrowHeight = 9;
        private const int LineHeight = 3;
        private readonly Brush ArrowBrush = Brushes.DimGray;

        #region ** 内部クラス

        private class EffectItem
        {
            private uint _index = uint.MaxValue;
            private Preview.EffectParameterSet _parameters = null;
            private RadioButton _button = null;

            public EffectItem(uint index, Preview.EffectParameterSet parameters, RadioButton button)
            {
                _index = index;
                _parameters = parameters;
                _button = button;
            }

            #region ** プロパティ

            public uint Index
            {
                get { return _index; }
            }

            public Preview.EffectParameterSet Parameters
            {
                get { return _parameters; }
            }

            public RadioButton Button
            {
                get { return _button; }
            }

            #endregion
        }

        #endregion

        #region ** フィールド

        private ToolSoundAuxBusType _busType = ToolSoundAuxBusType.AuxBusType_AuxA;

        private SortedList<uint, EffectItem> _effectItems = new SortedList<uint, EffectItem>();
        private EffectItem _selectedItem = null;

        #endregion

        public EffectAuxBusControl()
        {
            InitializeComponent();
        }

        #region ** プロパティ

        [DefaultValue("Label")]
        public string Label
        {
            get { return _label.Text; }
            set { _label.Text = value; }
        }

        [DefaultValue(ToolSoundAuxBusType.AuxBusType_AuxA)]
        public ToolSoundAuxBusType BusType
        {
            get { return _busType; }
            set { _busType = value; }
        }

        [Browsable(false)]
        public uint SelectedEffectIndex
        {
            get
            {
                if (null == _selectedItem) { return uint.MaxValue; }
                return _selectedItem.Index;
            }
            set
            {
                if (null != _selectedItem && value == _selectedItem.Index) { return; }

                // ラジオボタンの選択状態を更新する
                if (null != _selectedItem && uint.MaxValue != _selectedItem.Index)
                {

                    RadioButton target = _effectItems[_selectedItem.Index].Button;

                    target.Checked = false;
                    target.Font = new Font(Font, FontStyle.Regular);

                }

                if (uint.MaxValue != value)
                {

                    RadioButton target = _effectItems[value].Button;

                    target.Checked = true;
                    target.Font = new Font(Font, FontStyle.Bold);

                }

                // 選択アイテムを更新する
                _selectedItem = (uint.MaxValue == value) ? null : _effectItems[value];

                // イベント通知
                if (null != EffectSelectChanged)
                {
                    EffectSelectChanged(this, new EventArgs());
                }
            }
        }

        [Browsable(false)]
        public Preview.EffectParameterSet SelectedParameters
        {
            get
            {
                if (null == _selectedItem) { return null; }
                return _selectedItem.Parameters;
            }
        }

        #endregion

        #region ** イベント

        public event EventHandler EffectSelectChanged;

        #endregion

        #region ** イベントハンドラ

        private void OnPaint(object sender, PaintEventArgs e)
        {
            Rectangle lineRect = new Rectangle();
            lineRect.X = _label.Right;
            lineRect.Y = _radio0.Top + (_radio0.Height - LineHeight) / 2;
            lineRect.Width = _radio2.Right + 14 - lineRect.Left;
            lineRect.Height = LineHeight;

            Rectangle arrowRect = new Rectangle();
            arrowRect.X = lineRect.Right;
            arrowRect.Y = _radio0.Top + (_radio0.Height - ArrowHeight) / 2;
            arrowRect.Width = ArrowWidth;
            arrowRect.Height = ArrowHeight;


            e.Graphics.FillRectangle(ArrowBrush, lineRect);
            e.Graphics.FillPolygon(ArrowBrush,
                                    new Point[] {
                                        new Point( arrowRect.Left,  arrowRect.Top ),
                                        new Point( arrowRect.Left,  arrowRect.Bottom ),
                                        new Point( arrowRect.Right, arrowRect.Top + ( arrowRect.Height / 2 ) ),
                                    });
        }

        private void OnRadioButton0Clicked(object sender, EventArgs e)
        {
            SelectedEffectIndex = 0;
        }

        private void OnRadioButton1Clicked(object sender, EventArgs e)
        {
            SelectedEffectIndex = 1;
        }

        private void OnRadioButton2Clicked(object sender, EventArgs e)
        {
            SelectedEffectIndex = 2;
        }

        private void OnEffectParameterChanged(object sender, EventArgs e)
        {
            Preview.EffectParameter parameter = sender as Preview.EffectParameter;
            Debug.Assert(null != parameter);

            // 他の AuxBus の場合は処理しない
            if (_busType != parameter.EffectID.BusType) { return; }


            // Button.Text をエフェクトの種類に合わせる
            EffectItem target = _effectItems[parameter.EffectID.EffectIndex];
            Debug.Assert(null != target);

            target.Button.Text = EffectMessage.FromEffectType(parameter.EffectType);

            if (ToolSoundEffectType.EffectType_NoEffect == parameter.EffectType)
            {
                target.Button.ForeColor = SystemColors.GrayText;
            }
            else
            {
                target.Button.ForeColor = SystemColors.ControlText;
            }
        }

        #endregion

        #region ** メソッド

        public void Initialize(Preview.EffectParameterCollection parameterSets)
        {
            Debug.Assert(null != parameterSets);

            // アイテムリストを初期化する
            Preview.EffectParameterSet parameterSets0 =
                parameterSets[new Preview.EffectID(_busType, 0)];
            Preview.EffectParameterSet parameterSets1 =
                parameterSets[new Preview.EffectID(_busType, 1)];
            Preview.EffectParameterSet parameterSets2 =
                parameterSets[new Preview.EffectID(_busType, 2)];

            parameterSets0.EffectParameterChanged += OnEffectParameterChanged;
            parameterSets1.EffectParameterChanged += OnEffectParameterChanged;
            parameterSets2.EffectParameterChanged += OnEffectParameterChanged;

            AddEffectItem(new EffectItem(0, parameterSets0, _radio0));
            AddEffectItem(new EffectItem(1, parameterSets1, _radio1));
            AddEffectItem(new EffectItem(2, parameterSets2, _radio2));
        }

        public void Reset()
        {
            SelectedEffectIndex = uint.MaxValue;
        }

        private void AddEffectItem(EffectItem newItem)
        {
            Debug.Assert(null != newItem);
            Debug.Assert(uint.MaxValue != newItem.Index);
            Debug.Assert(null != newItem.Parameters);
            Debug.Assert(null != newItem.Button);

            _effectItems.Add(newItem.Index, newItem);
        }

        #endregion
    }
}
